/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.context;

/**
 * Abstract context factory base class for thread-local-based context factory
 * interface implementations being independent from other context factory
 * implementations. All writing optional operations can be supported in
 * independent context factories.
 *
 * @created 20. Januar 2003
 * @see IContextFactory
 */
public abstract class AbstractIndependentThreadContextFactory
   extends AbstractContextFactory {
  protected InheritableThreadLocal inheritableThreadLocal =
    new InheritableThreadLocal();

  /**
   * Construct object of class AbstractIndependentThreadContextFactory.
   *
   * @exception ContextFactoryRegistrationError when the context factory is
   *      already registered with the context factory registry
   */
  protected AbstractIndependentThreadContextFactory()
    throws ContextFactoryRegistrationError {
    super();
  }

  /**
   * Internally get the stack of contexts associated with the actual thread.
   *
   * @return stack of contexts
   */
  protected IContextStack getContextStack() {
    IContextStack contextStack =
      (IContextStack)inheritableThreadLocal.get();
    if (contextStack == null) {
      inheritableThreadLocal.set(contextStack = new ContextStack());
    }
    return contextStack;
  }

  /**
   * Internally get the stack of contexts associated with the actual thread.
   *
   * @param contextStack The new contextStack value
   */
  protected void setContextStack(IContextStack contextStack) {
    inheritableThreadLocal.set(contextStack);
  }

  /**
   * Checks if there is at least one actual context stacked.
   *
   * @return flag indicating whether there is at least one actual context
   *      stacked
   */
  public boolean isAvailable() {
    return getContextStack().isAvailable();
  }

  /**
   * Get the number of context entries in the stack of contexts.
   *
   * @return number of context entries in the stack of contexts
   */
  public int size() {
    return getContextStack().size();
  }

  /**
   * Get actual context from stack of contexts.
   *
   * @return actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   */
  public Object getTypeUnsafeContext()
    throws ContextUnavailableException {
    return getContextStack().getTypeUnsafeContext();
  }

  /**
   * Get actual context from stack of contexts. Return null if none is availble
   * and don't throw a ContextUnavailableException.
   *
   * @return actual context on the context stack
   */
  public Object getTypeUnsafeContextOrNullIfUnavailable() {
    return getContextStack().getTypeUnsafeContextOrNullIfUnavailable();
  }

  /**
   * Replace actual context with new one in stack of contexts. If no context is
   * stored, this one will be added.
   *
   * @param context new actual context replacing the old one on the stack
   */
  public void setTypeUnsafeContext(Object context) {
    getContextStack().setTypeUnsafeContext(context);
  }

  /**
   * Remove actual context including all other stacked contexts completely
   * (optional operation).
   *
   * @exception UnsupportedOperationException when the context factory isn't
   *      supporting this writing optional operation
   */
  public void removeTypeUnsafeContext()
    throws UnsupportedOperationException {
    setContextStack(null);
  }

  /**
   * Get and remove actual context from the stack of contexts.
   *
   * @return last actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   */
  public Object popTypeUnsafeContext()
    throws ContextUnavailableException {
    return getContextStack().popTypeUnsafeContext();
  }

  /**
   * Add new context to the top of the stack of contexts.
   *
   * @param context new actual context
   */
  public void pushTypeUnsafeContext(Object context) {
    getContextStack().pushTypeUnsafeContext(context);
  }
}
