package org.eclipse.draw2d.geometry;
/*
 * Licensed Material - Property of IBM
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved.
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 */

/**
 * Provides support for the physical entity of dimension.
 */
public class Dimension
	implements Cloneable, java.io.Serializable
{

public int width, height;

static final long serialVersionUID = 1;

/**
 * Constructs a Dimension of zero width and height.
 * 
 * @since 2.0
 */
public Dimension(){}

/**
 * Constructs a Dimension with the width and height of the 
 * passed Dimension. 
 *
 * @param d  Dimension supplying the initial values.
 * @since 2.0
 */
public Dimension(Dimension d){width = d.width; height = d.height;}

/**
 * Constructs a Dimension where the width and height are the 
 * x and y distances of the input point from the origin.
 *
 * @param pt  Point supplying the initial values.
 * @since 2.0
 */
public Dimension(org.eclipse.swt.graphics.Point pt){width = pt.x; height = pt.y;}

/**
 * Constructs a Dimension with the supplied width and 
 * height values.
 *
 * @param w  Width of the Dimension.
 * @param h  Height of the Dimension.
 * @since 2.0
 */
public Dimension(int w, int h){width = w; height = h;}

/**
 * Constructs a Dimension with the width and height of the Image
 * supplied as input.
 *
 * @param image  {@link org.eclipse.swt.graphics.Image SWT Image} 
 *                supplying the dimensions.
 * @see  org.eclipse.swt.graphics.Image
 * @since 2.0
 */
public Dimension(org.eclipse.swt.graphics.Image image){
	org.eclipse.swt.graphics.Rectangle r = image.getBounds();
	width = r.width;
	height = r.height;
}

/**
 * Returns whether the input Dimension fits into this Dimension.
 * A Dimension of the same size is considered to "fit". 
 *
 * @param d  Dimension being tested.
 * @param  <code>boolean</code> specifying the result of the fit test.
 * @since 2.0
 */
public boolean contains(Dimension d){
	return width >= d.width && height >= d.height;
}

/**
 * Returns whether the input Dimension fits into this Dimension.
 * A Dimension of the same size is considered to not "fit".
 *
 * @param d  Dimension being tested.
 * @param  <code>boolean</code> specifying the result of the fit test.
 * @since 2.0
 */
public boolean containsProper(Dimension d){
	return width > d.width && height > d.height;
}

/**
 * Copies the width and height values of the input Dimension to this Dimension.
 * 
 * @param d  Dimension supplying the values.
 * @since 2.0
 */
public void copyFrom(Dimension d){
	width = d.width;
	height= d.height;
}

/**
 * Returns the area covered by this Dimension.
 *
 * @return  Returns the area covered by this Dimension.
 * @since 2.0
 */
public int getArea(){
	return width * height;
}

/**
 * Returns a copy of this Dimension
 *
 * @return A copy of this Dimension
 * @since 2.0
 */
public Dimension getCopy(){
	return new Dimension( this );
}

/**
 * Returns a Dimension comprised of the differences in width and 
 * height of this Dimension compared to the passsed Dimension.   
 *
 * @param d  Dimension being compared.
 * @return  A new Dimension signifying the difference of width and height.
 * @see #getExpanded(Dimension)
 * @see #getExpanded(int,int)
 * @since 2.0
 */
public Dimension getDifference(Dimension d){
	return new Dimension(width-d.width, height-d.height);
}

/**
 * Returns a new Dimension which is comprised of
 * this Dimension's width expanded by <i>d</d>.width
 * and this Dimension's height expanded by <i>d</d>.height.
 *
 * @param d  Dimension providing the expansion width and height.
 * @return  A new Dimension containing the expanded width and height.
 * @see #getDifference(Dimension)
 * @since 2.0
 */
public Dimension getExpanded(Dimension d){
	return new Dimension(width+d.width, height+d.height);
}

/**
 * Returns a new Dimension which is comprised of
 * this Dimension's width expanded by <i>w</i>.
 * and this Dimension's height expanded by <i>h</i>.
 *
 * @param w  Value by which the width of this is to be expanded.
 * @param h  Value by which the height of this is to be expanded.
 * @return  A new Dimension containing the expanded width and height.
 * @see  #getDifference(Dimension)
 * @since 2.0
 */
public Dimension getExpanded(int w, int h){
	return new Dimension(width+w, height+h);
}

/**
 * Creates and returns a new Dimension representing the intersecting
 * region of this Dimension and the given Dimension.
 * 
 * @see #intersect(Dimension)
 * @since 2.0
 */
public Dimension getIntersected(Dimension d){
	return new Dimension(this).intersect(d);
}

/**
 * Returns a new Dimension with a negated copy of this Dimension. 
 *
 * @return	A negated copy of this Dimension
 * @since 2.0
 */
public Dimension getNegated(){
	return new Dimension( 0 - width, 0 - height );
}

/**
 * Returns whether the input Dimension is same
 * as this Dimension.
 * True if the height and width of this Dimension 
 * are equivalent to the input Dimension, false
 * otherwise.
 *
 * @param o  Object being tested for equality.
 * @return  Result of the size test.
 * @since 2.0
 */
public boolean equals(Object o){
	if (o instanceof Dimension){
		Dimension d = (Dimension)o;
		return (d.width == width && d.height == height);
	}
	return false;
}

/**
 * Returns whether this Dimension's width and height are
 * equal to the given width and height.
 * 
 * @since 2.0
 */
public boolean equals(int w, int h){
	return width == w && height == h;
}

/**
 * Increases the size of this Dimension by the Dimension
 * supplied as input, and returns this for convenience.
 *
 * @param d  Dimension providing the expansion width and height.
 * @return  Returns this Dimension with the expanded widht and height.
 * @see #expand(int,int)
 * @see #shrink(int,int)
 * @since 2.0
 */
public Dimension expand(Dimension d){
	width += d.width;
	height+= d.height;
	return this;
}

/**
 * Increases the size of this Dimension with the distances
 * of the input point from the origin along each of the axes,
 * and returns this for convenience.
 *
 * @param pt  Point supplying the dimensional values.
 * @return  Returns this Dimension with the incremented sizes.
 * @see #shrink(int,int)
 * @since 2.0
 */
public Dimension expand(Point pt){
	width += pt.x;
	height+= pt.y;
	return this;
}

/**
 * Increases the size of this Dimension by the width and height values 
 * supplied as input, and returns this for convenience.
 *
 * @param w  Value by which the width should be increased.
 * @param h  Value by which the height should be increased.
 * @return  Returns this Dimension with the incremented sizes.
 * @since 2.0
 */
public Dimension expand(int w, int h){
	width += w;
	height+= h;
	return this;
}

/**
 * Returns a new Dimension with its width and height scaled by the
 * input value supplied.
 *
 * @param amount  Value by which the width and height are scaled.
 * @return  A new Dimension containing the scaled width and height.
 * @since 2.0
 */
public Dimension getScaled(float amount){return new Dimension(this).scale(amount);}

/**
 * Returns a new Dimension with its height and width interchanged.
 * Useful in orientation change calculations.
 *
 * @return  Returns a new Dimension containing the switched width and height.
 * @since 2.0
 */
public Dimension getTransposed(){return new Dimension(this).transpose();}

/**
 * Returns a new Dimension containing the union of this
 * and the supplied Dimension. The union operation
 * takes the maximum of the width and height.
 *
 * @param d  Dimension to be unioned.
 * @return  Returns a new Dimension with the unioned sizes.'
 * @see  #getDifference(Dimension)
 * @since 2.0
 */
public Dimension getUnioned(Dimension d){return new Dimension(this).union(d);}

/**
 * Returns true if this <code>Dimension</code> is greater than  
 * the given <code>Dimension</code> in at least one dimension
 * and greater than or equal to in the other.
 *
 * @param	d 	Dimension against which this Dimension is compared.
 * @return	A boolean indicating whether or not this Dimension is 
 *			greater than <i>d</i>.
 * @since 2.0
 */
public boolean greaterThan( Dimension d ){
	if( width > d.width )
		return height >= d.height;
	if( height > d.height )
		return width >= d.width;
	return false;
}

/**
 * Modifies <code>this</code> Dimension in such a way that its 
 * width becomes the greater of <code>this</code> Dimension's height 
 * and <i>d</i>'s height, and its height becomes the greater of 
 * <code>this</code> Dimension's width and <i>d</i>'s width. <code>this</code> 
 * modified Dimension is returned for convenience.
 * 
 * @since 2.0
 */
public Dimension intersect(Dimension d){
	width = Math.min(d.width, width);
	height = Math.min(d.height, height);
	return this;
}

/**
 * Returns whether the Dimension has width or height greater than 0. 
 *
 * @return  <code>boolean<code> containing the emptiness test.
 * @since 2.0
 */
public boolean isEmpty(){
	return (width <= 0) || (height <= 0);
}

/**
 * Negates this Dimension
 *
 * @return	Returns this Dimension for convenience.
 * @since 2.0
 */
public Dimension negate(){
	width = 0 - width;
	height = 0 - height;
	return this;
}

/**
 * Scales the width and height of this Dimension by the amount supplied, and
 * returns this for convenience. 
 *
 * @param amount  Value by which this Dimension's width and 
 *                 height are to be scaled.
 * @return  Returns this Dimension with the scaled values.
 * @since 2.0
 */
public Dimension scale(float amount){return scale(amount, amount);}

/**
 * Scales the width of this Dimension by <i>w</i> and
 * scales the height of this Dimension by <i>h</i>.
 * Returns this for convenience. 
 *
 * @param w  Value by which the width is to be scaled.
 * @param h  Value by which the height is to be scaled.
 * @return  Returns this Dimension with the scaled values.
 * @since 2.0
 */
public Dimension scale(float w, float h){
	width  = (int)(width * w + 0.5);
	height = (int)(height* h + 0.5);
	return this;
}

/**
 * Reduces the width of this Dimension by <i>w</i>, and 
 * reduces the height of this Dimension by <i>h</i>.
 * Returns this for convenience.
 *
 * @param w  Value by which the width is to be reduced.
 * @param h  Value by which the height is to be reduced.
 * @return  Returns this Dimension with the reduced width and height values.
 * @since 2.0
 */
public Dimension shrink(int w, int h){
	return expand(-w,-h);
}

/**
 * Returns a String description of this Dimension. 
 *
 * @return  Description as a String.
 * @since 2.0
 */
public String toString(){
	return "Dimension("+ //$NON-NLS-1$
		width+", "+ //$NON-NLS-1$
		height+")"; //$NON-NLS-1$
}

/**
 * Switches the width and height of this Dimension, and
 * returns this for convenience. Can be useful in 
 * orientation changes.
 *
 * @return  This Dimension with the switched values.
 * @since 2.0
 */
public Dimension transpose(){
	int temp = width;
	width = height;
	height = temp;
	return this;
}

/**
 * Sets the width of this Dimension to the greater of
 * this Dimension's width and <i>d</i>.width.
 * Likewise for this Dimension's height.
 * 
 * @param d Dimension to union with this Dimension.
 * @return  Returns this Dimension with the unioned width and height.
 * @since 2.0
 */
public Dimension union (Dimension d){
	width = Math.max(width,d.width);
	height= Math.max(height,d.height);
	return this;
}

}