package org.eclipse.draw2d;
/*
 * Licensed Material - Property of IBM
 * (C) Copyright IBM Corp. 2001, 2002 - All Rights Reserved.
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 */

import org.eclipse.swt.graphics.Font;
import org.eclipse.draw2d.geometry.*;

/**
 * LabeledContainer - Container with a title bar
 * describing the contents of the container. The
 * frame is generated by a {@link LabeledBorder}.
 */
public class LabeledContainer
	extends Figure
{

/**
 * Constructs a default container with a {@link GroupBoxBorder} 
 * border.
 * 
 * @since 2.0
 */
public LabeledContainer(){
	this(new GroupBoxBorder());
}

/**
 * Constructs a labeled container with the Border
 * given as input.
 *
 * @param border  Labeled Border for the container.
 * @since 2.0
 */
public LabeledContainer(Border border) {
	setBorder(border);
	setOpaque(true);
}

private static LabeledBorder findLabeledBorder(Border border){
	if (border instanceof LabeledBorder)
		return (LabeledBorder)border;
	if (border instanceof CompoundBorder){
		CompoundBorder cb = (CompoundBorder)border;
		LabeledBorder labeled = findLabeledBorder(cb.getInnerBorder());
		if (labeled == null)
			labeled = findLabeledBorder(cb.getOuterBorder());
		return labeled;
	}
	return null;
}

/**
 * Returns the text of the LabeledContainer's label.
 *
 * @see  #setLabel(String)
 * @see  #setLabelFont(Font)
 * @since 2.0
 */
public String getLabel(){
	return getLabeledBorder().getLabel();
}

/**
 * Returns the LabeledBorder of this container.
 *
 * @return  Border of this figure as a LabeledContainer.
 * @since 2.0
 */
protected LabeledBorder getLabeledBorder(){
	return findLabeledBorder(getBorder());
}

/**
 * Sets the title of the container.
 *
 * @param s  Title of the container.
 * @see  #getLabel()
 * @see  #setLabelFont(Font)
 * @since 2.0
 */
public void setLabel(String s){
	getLabeledBorder().setLabel(s);
	revalidate();
	repaint();
}

/**
 * Sets the font to be used for the container title.
 *
 * @param f  Font to be used by the title.
 * @see #getLabel()
 * @see #setLabel(String)
 * @since 2.0
 */
public void setLabelFont(Font f){
	getLabeledBorder().setFont(f);
	revalidate();
	repaint();
}

}