package com.inqmy.ats.system.communication.impl0;

import java.io.*;

import com.inqmy.ats.system.communication.Connection;
import com.inqmy.ats.system.communication.RemoteInvokeException;

import com.inqmy.ats.system.communication.protocol.Protocol;

import com.inqmy.ats.system.communication.client.Proxy;
import com.inqmy.ats.system.communication.client.CommunicationContextImpl;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 *
 *
 * @author Tzvetan Georgiev (tsvetan.georgiev@sap.com)
 * @version 1.0
 */
public class ProtocolImpl implements Protocol {
  
  public static final byte INITIAL_MESSAGE = 1;
  public static final byte CALL_REQUEST_MESSAGE = 2;
  public static final byte ANSWER_MESSAGE = 3;
  public static final byte REGISTER_REQUEST_MESSAGE = 4;
  public static final byte ASK_FOR_PROXY_REQUEST_MESSAGE = 5;
  public static final byte UNREGISTER_REQUEST_MESSAGE = 6;
  public static final byte REGISTER_LISTENER_MESSAGE = 7;
  public static final byte UNREGISTER_LISTENER_MESSAGE = 8;


  public static int PROTOCOL_HEADER_LENGTH = 0;
  public static final byte[] PROTOCOL_NAME = "#ceco_#".getBytes();

  private CommunicationContextImpl context;
  private Message[] messages;
  private int[] freeIds;
  private int freeCount;

  private ThreadPool pool;

  private static Location location = Location.getLocation(ProtocolImpl.class);


  public ProtocolImpl(CommunicationContextImpl context) {
    //PROTOCOL_NAME.length + "byte->message type" 1 + "int->body length" 4 + "int->callId" 4;
    PROTOCOL_HEADER_LENGTH = PROTOCOL_NAME.length + 1 + 4 + 4;
    this.context = context;
    messages = new Message[1];
    freeIds = new int[1];
    freeCount = 1;
    for(int i = 0 ; i < 1 ; i++) {
      messages[i] = new Message((byte)-1 , i , -1);
      freeIds[i] = i;
    }
    this.pool = context.getThreadPool();
  }
  //return protocol header length
  public int getHeaderLength() {
     return PROTOCOL_HEADER_LENGTH;
  }
  public CommunicationContextImpl getContext(){
    return context;
  }
  //process information that is coded in header array
  //invoked when message is received
  //this method have to read body of message and then release connection for reading other messages
  public void processHeader(byte[] b , Connection con) {
    Message message = Parser.getMessage(b);
    byte[] bodyBytes = new byte[message.getBodyLength()];
    try{
      con.readBytes(bodyBytes);//reads body of message
    } catch(Exception e) {
      location.debugT("Cannot read body of the message : " + message.toString() + ". Error is : " + e.getMessage());
      location.traceThrowableT(Severity.DEBUG , "" , e);
      con.close();
      return;
    }
    if(message.getType() == ANSWER_MESSAGE) {

      Message m = this.getMessageWithId(message.getCallId());
      m.setBodyBytes(bodyBytes);
      synchronized(m) {
        m.notify();
      }
    }
    else {
      //now start thread for processing received message
      ProtocolProcessor processor = new ProtocolProcessor(this , message , bodyBytes, con);
      //take thread from pool
      if(pool != null) {
        try {
          pool.start(processor);
        } catch(Exception exc){
          location.debugT("Internal error in start thread from the pool");
          location.traceThrowableT(Severity.DEBUG , "" , exc);
          new Thread(processor).start();
        }
      } else {
        new Thread(processor).start();
      }
    }
    //now message and his body are readed and process thread is started ,method returns for receiving new messages
    //and to processes them
  }

  //client method
  //method for making remote call to Object , sends request and waits for answer message!
  //used in proxy class to call object with specefied register name
  public Object call(String registeredObjectName , int methodIndex , Object[] params , Connection con) throws Exception {
    Object resultObject = null;
    ObjectSerializator ser = ObjectSerializator.getSerializator();
    ser.addObject(registeredObjectName);
    ser.addInt(methodIndex);
    ser.addObject(params);
    byte[] bodyArray = ser.getCurrentBytes();
    Message m = this.getFreeMessage();
    m.setMessageInfo(CALL_REQUEST_MESSAGE,bodyArray.length);
    ObjectInputStream ois = null;
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problem occures
      m.wait();

    }
    if(!con.isAlive()) {
      this.releaseMessageWithId(m.getCallId());
      throw new Exception("Connection closed!");
    }
    ois = ser.getObjectInputStream(m.getBytes());
    this.releaseMessageWithId(m.getCallId());
    resultObject = ois.readObject();
    if(resultObject instanceof com.inqmy.ats.system.communication.client.Proxy) {
      ((Proxy)resultObject).setConnectionId(con.getId());
    }
    ois.close();
    return resultObject;
  }

  public void registerRemoteObjectToServer(String proxyClassName , String name , Connection con) throws Exception {
    ObjectSerializator ser = ObjectSerializator.getSerializator();
    ser.addObject(name);
    ser.addObject(proxyClassName);
    byte[] body = ser.getCurrentBytes();
    Message m = this.getFreeMessage();
    m.setMessageInfo(REGISTER_REQUEST_MESSAGE,body.length);
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , body);
      //can be notify if connection problem occures
      m.wait();

    }
    if(!con.isAlive()) {
      this.releaseMessageWithId(m.getCallId());
      throw new Exception("Connection closed!");
    }
    ObjectInputStream ois = null;
    ois = ser.getObjectInputStream(m.getBytes());
    this.releaseMessageWithId(m.getCallId());
    Object oo = ois.readObject();
    if(oo instanceof RemoteInvokeException) {
      throw (RemoteInvokeException)oo;
    }
    ois.close();
  }

  //this method is called from client.It asks server if there is registered object under specefied name!
  //first ask message is send with body containing registered name of wanted object!
  public String[] askForProxy(String name , Connection con) throws Exception {
    String[] resultObject = new String[2];
    ObjectSerializator ser = ObjectSerializator.getSerializator();
    ser.addObject(name);
    byte[] bodyArray = ser.getCurrentBytes();
    Message m = this.getFreeMessage();
    m.setMessageInfo(ASK_FOR_PROXY_REQUEST_MESSAGE,bodyArray.length);
    
    ObjectInputStream ois = null;
    //writing message and it's body to stream must be in synchronized block with monitor "connection" ,
    //in order stop other threads to write to stream !Every message must be followed by it's own body
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problem occures
      m.wait();

    }
    if(!con.isAlive()) {
      this.releaseMessageWithId(m.getCallId());
      throw new Exception("Connection closed!");
    }
    ois = ser.getObjectInputStream(m.getBytes());
    this.releaseMessageWithId(m.getCallId());
    Object oo = ois.readObject();
    if(oo instanceof RemoteInvokeException) {
      throw (RemoteInvokeException)oo;
    } else {
      resultObject[0] = (String)oo;
      resultObject[1] = (String)ois.readObject();
    }
    ois.close();
    return resultObject;
  }

  //sends message to server that listener is registered on this client
  public void registerListenerToServer(Connection con) throws Exception {
    byte[] bodyArray = new byte[0];
    Message m = this.getFreeMessage();
    m.setMessageInfo(REGISTER_LISTENER_MESSAGE,bodyArray.length);

    //writing message and it's body to stream must be in synchronized block with monitor "connection" ,
    //in order stop other threads to write to stream !Every message must be followed by it's own body
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problem occures
      m.wait();
    }

    if(!con.isAlive())  {
      this.releaseMessageWithId(m.getCallId());
      throw new Exception("Connection closed!");
    }
    this.releaseMessageWithId(m.getCallId());
  }

//sends message to server that listener is unregistered on this client
  public void unregisterListenerInServer(Connection con) throws Exception {

    byte[] bodyArray = new byte[0];
    Message m = this.getFreeMessage();
    m.setMessageInfo(UNREGISTER_LISTENER_MESSAGE,bodyArray.length);

    //writing message and it's body to stream must be in synchronized block with monitor "connection" ,
    //in order stop other threads to write to stream !Every message must be followed by it's own body
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problem occures
      m.wait();
      this.releaseMessageWithId(m.getCallId());
    }

    if(!con.isAlive()) throw new Exception("Connection closed!");

  }



  public void sendRegisterMessageToClient(Connection con , String name , String proxyClassName,byte type) throws Exception {
    ObjectSerializator ser = ObjectSerializator.getSerializator();
    ser.addObject(name);
    ser.addObject(proxyClassName);
    byte[] bodyArray = ser.getCurrentBytes();
//System.out.println(bodyArray.length + "public void sendRegisterMessageToClient() connection id = " + con.getId() + " name = "+name + " type = " + type );
    Message m = this.getFreeMessage();
    m.setMessageInfo(type , bodyArray.length);

    //writing message and it's body to stream must be in synchronized block with monitor "connection" ,
    //in order stop other threads to write to stream !Every message must be followed by it's own body
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problem occures
      m.wait();
      this.releaseMessageWithId(m.getCallId());
    }

    if(!con.isAlive()) throw new Exception("Connection closed!");


  }

  public void unregisterObject(String name , Connection con) throws Exception {
    ObjectSerializator ser = ObjectSerializator.getSerializator();
    ser.addObject(name);
    byte[] bodyArray = ser.getCurrentBytes();
    Message m = this.getFreeMessage();
    m.setMessageInfo(UNREGISTER_REQUEST_MESSAGE,bodyArray.length);
    synchronized(m) {
      con.writeMessageAndBody(Parser.getBytes(m) , bodyArray);
      //can be notify if connection problems occures!!!!!!
      //then body of message will be null method connectionClosed(Connection con) is called
      //in this case connection will not be alive!!!
      m.wait();

    }
    ObjectInputStream ois = null;
    byte[] responceMessageBytes =  m.getBytes();
    //returns message to pool
    this.releaseMessageWithId(m.getCallId());
    //if responceMessageBytes are null then connection error !!!!!!!
    //in this case connection is unregistred and
    if(!con.isAlive()) {
      //connection error
      //object is succesfully removed
       return;
    }
    ois = ser.getObjectInputStream(responceMessageBytes);

    Object oo = ois.readObject();
    ois.close();
    if(oo instanceof RemoteInvokeException) {
      throw (RemoteInvokeException)oo;
    }
  }
  
  public void connectionClosed(Connection con) {
    Message m = null;
    for(int i = 0 ; i < messages.length ; i++) {
      m = messages[i];
      synchronized(m) {
        m.notify();
      }
    }
    context.notifyConnectionClosed(con);
  }

  //returns free message if there is no such then creates new one and adds it to Message array 
  private synchronized Message getFreeMessage() {
    if(freeCount == 0) {
      //now have to make new free message and resize array
      int length = messages.length;
      Message m = new Message((byte)-1 , length , -1);
      Message[] mess = new Message[2*length];//new array
      System.arraycopy(messages,0,mess,0,length);
      mess[length] = m;
      messages = mess;
      freeCount = length -1;
      freeIds = new int[2*length];
      for(int i = 1 ; i < length ; i++) {
        messages[length+i] =  new Message((byte)-1 , length+i , -1);
        freeIds[i-1] = length+i;
      }
      return m;
    } else {
      //there is free message 
      --freeCount;
      int messageId = freeIds[freeCount];
      return messages[messageId];
    }
  }
  //returns message at specefied id and fix it as free 
  private synchronized void releaseMessageWithId(int id) {
    freeIds[freeCount] = id;
    messages[id].setBodyBytes(null);
    ++freeCount;
  }
  private synchronized Message getMessageWithId(int id){
    return messages[id];
  }
}