package com.inqmy.ats.system.communication.client;

import java.util.*;
import java.lang.reflect.*;


import com.inqmy.ats.system.communication.ObjectStore;
import com.inqmy.ats.system.communication.Connection;
import com.inqmy.ats.system.communication.ConnectionManager;
import com.inqmy.ats.system.communication.impl0.ProtocolImpl;
import com.inqmy.ats.system.communication.impl0.ObjectSerializator;
import com.inqmy.ats.system.communication.impl0.ThreadPool;
import com.inqmy.ats.system.communication.impl0.ReferencedLoader;
import com.inqmy.ats.system.tools.PropertiesChangeListener;
import com.inqmy.ats.system.tools.PropertiesWrapper;
import com.inqmy.ats.system.communication.RemoteInvokeException;
import com.inqmy.ats.system.communication.protocol.Protocol;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * Implementation of communication protocol
 *
 * @author Tzvetan Georgiev (tsvetan.georgiev@sap.com)
 * @version 1.0
 */

public class CommunicationContextImpl implements CommunicationContext , PropertiesChangeListener {

  //private Properties p;

  private boolean isServer;
  private ConnectionManager manager;
  private ThreadPool pool;

  private Vector listeners;
  private Vector remoteListeners;

  private com.inqmy.ats.system.communication.impl0.ReferencedLoader loader;
  private int port = 0;

  private static Location location = Location.getLocation(CommunicationContextImpl.class);

  private static final int ERROR_CONNECTION_ID = -100;

  public CommunicationContextImpl(Properties p) throws Exception {
    //this.p = p;
    this.loader = new ReferencedLoader();
    ObjectSerializator.setReferencedLoader(loader);
    isServer = Boolean.valueOf(p.getProperty("communication.server" , "true")).booleanValue();
    try{
      port =Integer.parseInt(p.getProperty("communication.server.port" , "9090").trim());
    } catch(Exception ee){
      port = 9090;
    }
    this.manager = new ConnectionManager(isServer ,port , p.getProperty("communication.server.host" , "localhost").trim(),this);
    this.port = manager.getPort();
    try {
      pool = new ThreadPool();
    } catch(Exception exc) {
      location.debugT("Cannot create thread pool. Error is : " + exc.getMessage());
      location.traceThrowableT(Severity.DEBUG , "",exc);
    }
  }

  public ThreadPool getThreadPool(){
    return pool;
  }

  public void setClassLoader(ClassLoader loader) {
    this.loader.addClassLoader(loader);
//    this.loader = loader;
  }

  public int getPort() {
    return port;
  }
  public String getHost() {
    return manager.getHost();
  }

  private Class load(String name) throws ClassNotFoundException , NoClassDefFoundError {
    Class clas = null;
    if (loader != null) {
      try {
        clas = loader.loadClass(name);
        return clas;
      } catch(ClassNotFoundException cnfexc) {
      } catch(NoClassDefFoundError ncdfe) {
      }
    }
    ClassLoader cl = Thread.currentThread().getContextClassLoader();
    if(cl != null){
      try {
        clas = cl.loadClass(name);
        return clas;
      } catch(ClassNotFoundException cnfexc) {
      } catch(NoClassDefFoundError ncdfe) {
      }
    }
    try {
      clas = Class.forName(name);
      return clas;
    } catch(ClassNotFoundException cnfexc) {
      throw cnfexc;
    } catch(NoClassDefFoundError ncdfe) {
      throw ncdfe;
    }
  }

  public Proxy toProxyInstance(Object obj) throws ClassNotFoundException, InstantiationException, IllegalAccessException {
    Class c = load(obj.getClass().getName() + "_Proxy");
    return (com.inqmy.ats.system.communication.client.Proxy)c.newInstance();
  }

  //this method return proxy of object who is registered
  //if there is no such registered object then server is asked
  public synchronized Proxy getProxy(String name) throws RemoteInvokeException {
    Object o = ObjectStore.getObject(name);
    if(o == null ) {
      if(isServer){
        throw new RemoteInvokeException("No such proxyRegistered!");
      } else {
        //ask server for such registered proxy
        Connection con = manager.getConnection(0);
        ProtocolImpl protocol = (ProtocolImpl)con.getProtocolHandler();
        String[] proxyInfo = null;
        try {
          proxyInfo = protocol.askForProxy(name, con);
        } catch(Exception exc) {
          if(exc instanceof RemoteInvokeException){
            throw (RemoteInvokeException)exc;
          }
          throw new RemoteInvokeException("No such proxyRegistered!Error getting proxy" , exc);
        }
        if(proxyInfo[0] == null) {
          throw new RemoteInvokeException("No such proxyRegistered!");
        } else {
          Class proxyClass = null;
          Object proxyObject = null;
          try{
            proxyClass = load(proxyInfo[0]);
          } catch(Throwable th) {
            throw new RemoteInvokeException(("Can not load proxy class : "+proxyInfo[0]) , th);
          }
          try{
            proxyObject = proxyClass.newInstance();
          } catch(InstantiationException  thr) {
            throw new RemoteInvokeException(("Can not load proxy class : "+proxyInfo[0]) , thr);
          } catch(IllegalAccessException  iexc) {
            throw new RemoteInvokeException(("Can not load proxy class : "+proxyInfo[0]) , iexc);
          }
          ((Proxy)proxyObject).setObjectRegisteredName(proxyInfo[1]);
          o = proxyObject;
          ObjectStore.storeObject(name , proxyObject);
        }
      }
    }
    return (Proxy)o;
  }

  //this method register object for remote calls
  //method from CommunicationContext interface
  public synchronized void registerObject(String name , Object object) throws RemoteInvokeException {
    String clName = object.getClass().getName();
    Class proxyClass = null;
    Object proxyObject = null;
    if(ObjectStore.getObject(name) != null) {
        throw new RemoteInvokeException("Allready registered object with this name : '"+ name +"' !");
      }
    try {
      proxyClass = load(clName + "_Proxy");
    } catch(Throwable thr) {
      throw new RemoteInvokeException(("Can not load proxy class : " + clName + "_Proxy") , thr);
    }
    try {
      proxyObject = proxyClass.newInstance();
    } catch(InstantiationException  thr) {
      throw new RemoteInvokeException(("Can not instantiate proxy class : " + clName + "_Proxy") , thr);
    } catch(IllegalAccessException  iexc) {
      throw new RemoteInvokeException(("Can not instantiate proxy class : " + clName + "_Proxy") , iexc);
    }
    if(!isServer) {
      //now register it's proxy on server
      Connection con = manager.getConnection(0);
      ProtocolImpl protocol = (ProtocolImpl)con.getProtocolHandler();
      //notify server that object is registered for communication
      Object o = ObjectStore.getObject(name);
      if(o != null) {
        throw new RemoteInvokeException("Allready registered object with this name : '"+ name +"' !");
      }
      try {

        ((Proxy)proxyObject).setObjectRegisteredName(name);
        ObjectStore.storeLocalObject(name , object);
        ((Proxy)proxyObject).setTargetObject(object);
        ((Proxy)proxyObject).setConnectionId(-1);//local proxy object
        ObjectStore.storeObject(name , proxyObject);
        protocol.registerRemoteObjectToServer(proxyClass.getName() , name , con);
      } catch(Exception exc) {

        ObjectStore.removeObject(name);
        ObjectStore.removeLocalObject(name);

        if(exc instanceof RemoteInvokeException) {
          //exception is thrown in remote call
          throw (RemoteInvokeException)exc;
        }
        //in this case exception is in local method
        throw new RemoteInvokeException("Error registering object with name : " + name+" .Local method error!" , exc);
      }
    } else {
      //server check for this object locally
      if(ObjectStore.getObject(name) != null) {
        throw new RemoteInvokeException("Allready registered object with this name : '"+ name +"' !");
      }
      ((Proxy)proxyObject).setObjectRegisteredName(name);
      ObjectStore.storeLocalObject(name , object);
      ((Proxy)proxyObject).setTargetObject(object);
      ((Proxy)proxyObject).setConnectionId(-1);//local proxy object
      ObjectStore.storeObject(name , proxyObject);
    }

    //now notify all listeners
    notifyAllListeners(ProtocolImpl.REGISTER_REQUEST_MESSAGE , name ,proxyClass.getName(), -1);
  }

  /**
   * Before unregister object with specefied name proxy class that
   * responds to this name must be getted.Otherwise object with this name
   * may not be unregistered(in case that this method is called from non server VM)
   * This method unregisters object not name!
   * Example :
   *   String name = "test";
   *   ComunicationContext communication = "factroy_name".getCommunicationContext();
   *   Proxy p = communication.getProxy(name);
   *   communication.unregisterObject(name);
   */
  public synchronized void unregisterObject(String name) throws RemoteInvokeException {
    //ask for unregister object on it's own JVM
    //this method is invoked locally
//System.out.println("-----------+unregioster remote : " + name) ;
    //if this method is called and in some case JVM that holds real object for this name drops
    //(connection problem , in this case close() method of connection is invoked)
    // then method protocol.unregisterObject(name , c); can throw NullPointerExcpetion
    // reason is that there is no responce message and empty one is returned!!!!
    Proxy proxy = (Proxy)ObjectStore.removeObject(name);
    Object object = ObjectStore.removeLocalObject(name);

    if(proxy == null) {
      throw new RemoteInvokeException("No registered object with this name : '"+ name +"' !");
    } else {
      try {
        int connectionId = proxy.getConnectionId();

        if(!isServer) {//unregister remote object
          //send unregister message to the server. Notify server that object have been unregistered.
          Connection c = manager.getConnection(connectionId);
          ProtocolImpl protocol = (ProtocolImpl)c.getProtocolHandler();
          protocol.unregisterObject(name , c);
        }
        //now notify all listeners. In case of call in the client side only local listeners will be notified.
        notifyAllListeners(ProtocolImpl.UNREGISTER_REQUEST_MESSAGE , name ,proxy.getClass().getName(), connectionId);

      } catch(Exception exc) {
        RemoteInvokeException riexc = (exc instanceof RemoteInvokeException) ? (RemoteInvokeException)exc : new RemoteInvokeException("Error unregistering object with name : " + name, exc);
        throw riexc;
      }
    }

  }
  public synchronized void unregisterRemoteObject(String name , int connectionId , byte type) throws RemoteInvokeException {
    //this method is invoked remotely. it is called from ProtocolProcessor in case that message
    // with head "ProtocolImpl.UNREGISTER_REQUEST_MESSAGE" is received. Ordinary this method is invoked on the server side.
    //In this case client send ProtocolImpl.UNREGISTER_REQUEST_MESSAGE message. If the message is send from the server to the client this means
    //that other client have invoked unregister message or unregiter is called on the server
//System.out.println("++++++++unregioster remote : " + name) ;
    try {
      if(isServer) {
        //in this case client have invoked unregister message and it is received from the server.
        //Now the server should unregister the object loacally and send unregister message to the
        //rest of the clients.
        Object object = ObjectStore.removeLocalObject(name);
        if(object != null){//local object
          //unregister locally
          ObjectStore.removeObject(name);
          notifyAllListeners(type , name , object.getClass().getName() + "_Proxy", connectionId);
        } else {
          //In this case unregister is called from the client
          Proxy proxy = (Proxy)ObjectStore.getObject(name);
          if(proxy == null) {
            throw new Exception("Object with name : " + name + " is not registered!");
          } else {
            ObjectStore.removeObject(name);
            notifyAllListeners(ProtocolImpl.UNREGISTER_REQUEST_MESSAGE , name ,proxy.getClass().getName(), connectionId);
          }
        }
      } else {
        ObjectStore.removeObject(name);
        Object obj = ObjectStore.removeLocalObject(name);
        //unregister locally
        //this call is received from the server
        notifyAllListeners(ProtocolImpl.UNREGISTER_REQUEST_MESSAGE , name ,null, connectionId);

      }
    } catch(Exception exc) {
      RemoteInvokeException riexc = (exc instanceof RemoteInvokeException) ? (RemoteInvokeException)exc : new RemoteInvokeException("Error unregistering object with name : " + name, exc);
      throw riexc;// new RemoteInvokeException("Error unregister object : " + name , exc);
    }
  }
//notify server that object is registered on client
  public void registerProxyObject(String name , String proxyClassName , int connectionId) throws Exception {
    if(ObjectStore.getObject(name) != null) {
//System.out.println("ALL READY REGISTERED  REGISTER ON SERVER : " +name+ " PROXY CLAS NAME : " + proxyClassName );
       throw new Exception("Allready registered object with this name : '"+ name +"' !");
    }
    Class proxyClass = load(proxyClassName);
    Object proxyObject = proxyClass.newInstance();
    ((Proxy)proxyObject).setObjectRegisteredName(name);
    ((Proxy)proxyObject).setConnectionId(connectionId);

    ObjectStore.storeObject(name , proxyObject);
    //now notify all listeners

    notifyAllListeners(ProtocolImpl.REGISTER_REQUEST_MESSAGE , name,proxyClassName, connectionId);
//System.out.println(" REGISTER ON SERVER : " +name+ " PROXY CLAS NAME : " + proxyClassName );
  }
  public void registerLocally(String name , Proxy pr) {
    ObjectStore.storeObject(name , pr);
  }

  public void notifyChange(PropertiesWrapper pw) {
     System.out.println(" PROPERTIES HAS CHANGED");
  }

  public Protocol getProtocol() {
    return new ProtocolImpl(this);
  }

  //this method register listener for objects that are registered in CommunicationContext
  public synchronized void registerListener(RegisterListener listener) throws RemoteInvokeException {
    boolean isRegistered = true;
    if(listeners == null) {
      isRegistered = false;
      listeners = new Vector();
    }
    listeners.add(listener);
    if(!isServer && !isRegistered) {
      //now must notify server that listener is registered on client!!!
      Connection con = manager.getConnection(0);
      ProtocolImpl protocol = (ProtocolImpl)con.getProtocolHandler();
      try {
        protocol.registerListenerToServer(con);
      } catch(Exception exc) {
        throw new RemoteInvokeException(exc.getMessage() , exc);
      }
    }
  }
  private void notifyLocalListeners(byte type , String objectName ) {
    if(listeners != null) {
      if(type == ProtocolImpl.REGISTER_REQUEST_MESSAGE ) {
        for (int i = 0 ; i < listeners.size(); i++) {
          try {
          ((RegisterListener)listeners.elementAt(i)).objectRegistered(objectName, null);
          } catch(Exception exc) {
            location.debugT("Error call register of local listener : " + listeners.elementAt(i) + ". Object name : "+objectName+". Error is : " + exc.getMessage());
            location.traceThrowableT(Severity.DEBUG , "",exc);
          }
        }
      } else if (type == ProtocolImpl.UNREGISTER_REQUEST_MESSAGE) {
        for (int i = 0 ; i < listeners.size(); i++) {
          try {
            ((RegisterListener)listeners.elementAt(i)).objectUnregistered(objectName);
          } catch(Exception exc) {
            location.debugT("Error call unregister of local listener : " + listeners.elementAt(i) + ". Object name : "+objectName+". Error is : " + exc.getMessage());
            location.traceThrowableT(Severity.DEBUG , "",exc);

          }
        }
      }
    }
  }

  private void notifyRemoteListeners(byte type , String objectName , String proxyClassName, int connectionId) {
    if(remoteListeners != null) {
      for (int i = 0 ; i < remoteListeners.size(); i++) {
        Connection con = ((Connection)remoteListeners.elementAt(i));
        if(con.getId() == connectionId){
          continue;//server do not send message for registered object to client that register this object!!
        }
        try {
          ProtocolImpl protocol = (ProtocolImpl)con.getProtocolHandler();
          protocol.sendRegisterMessageToClient(con, objectName , proxyClassName, type);
        } catch(Exception exc) {
          location.debugT("Cannot notify remote JVM. connection id : " + connectionId + " class name : " + proxyClassName +" type : " + type );
          location.traceThrowableT(Severity.DEBUG , "",exc);
        }
      }
    }
  }

  public void notifyAllListeners(byte type , String objectName , String proxyClassName, int connectionId) {
    notifyLocalListeners(type , objectName);
    if(isServer) {
      notifyRemoteListeners(type , objectName ,proxyClassName, connectionId);
    }

  }
  public void addListener(Connection con) {
    if(remoteListeners == null) {
      remoteListeners = new Vector();
    }
    if(!remoteListeners.contains(con)) {
      remoteListeners.add(con);
    }
  }

  public void removeListener(Connection con) {
    if(remoteListeners != null) {
      remoteListeners.removeElement(con);
    }
  }

  //unregister registered listener
  public synchronized void unregisterListener(RegisterListener listener) throws RemoteInvokeException {
    listeners.removeElement(listener);
    if(!isServer && listeners.size() == 0) {
      //if the call is made from the client then the server must be notified to remove the listener
      //this should be done in case that there is no other local listeners left

      //now send request for unregister on the server
      //ProtocolImpl p = (ProtocolImpl)getProtocol();
      Connection con = manager.getConnection(0);
      ProtocolImpl p = (ProtocolImpl)con.getProtocolHandler();
      //System.out.println("Unregister listener");
      try {
        p.unregisterListenerInServer(con);
      } catch(Exception exc) {
        throw new RemoteInvokeException(exc.getMessage() , exc);
      }
    }
  }


  //this method is for invoke from client
  public Object invokeMethod(String registeredObjectName , int methodIndex , Object[] params , int connectionId) throws RemoteInvokeException {
    Object o = ObjectStore.getLocalObject(registeredObjectName);
    if(o != null) {
      synchronized(this) {
        //check if object has been unregistered
        if( ObjectStore.getObject(registeredObjectName) == null ) {
          throw new RemoteInvokeException("Object with name '"+registeredObjectName+"' have been unregistered!");
        }
      }
      //local object -----------
      //local call to method
      Method m = null;
      try{
        m = ObjectStore.getMethodLocalFor(registeredObjectName , methodIndex );
      } catch(Throwable thr) {
        throw new RemoteInvokeException("Now such method!" , thr );
      }
      Object res = null;
      try{
        res = m.invoke(o , params);
      } catch(InvocationTargetException itexc) {
         throw new RemoteInvokeException("" , itexc.getTargetException());
      } catch(IllegalAccessException iaexc) {
         throw new RemoteInvokeException("" , iaexc);
      }
      return res;
    } else {
      Connection con = null;
      try{
        con = manager.getConnection(connectionId);
      } catch(Exception exc){
        //now have to unregister object
        throw new RemoteInvokeException("No connection to this object !!");
      }
      Object obj = null;
      ProtocolImpl protocol = (ProtocolImpl)con.getProtocolHandler();
      synchronized(this) {
        //check if object has been unregistered
        if( ObjectStore.getObject(registeredObjectName) == null ) {
          try{
            String[] sss = protocol.askForProxy(registeredObjectName , con);
          } catch(Exception e){
            throw new RemoteInvokeException("Object with name '"+registeredObjectName+"' have been unregistered!" , e);
          }
        }
      }
      //object is not local and now send call to JVM with specefied connection -----------
      try{
        obj = protocol.call(registeredObjectName , methodIndex , params,con);
      } catch(Exception exc) {
        throw new RemoteInvokeException("" , exc);
      }
      if(obj instanceof RemoteInvokeException) {
        Throwable thr = ((RemoteInvokeException)obj).getTargetException();
        throw (RemoteInvokeException)obj;
      }
      return obj;
    }
  }

  public synchronized void notifyConnectionClosed(Connection con) {
    //connection is closed or i/o error occures
    Proxy[] proxies = ObjectStore.getAllProxies();
    int id = con.getId();
    for(int i = 0 ; i < proxies.length ; i++) {
      Proxy proxy = proxies[i];
      if(proxy.getConnectionId() == id) {
        location.debugT("Connection error! Unregistering proxy : " + proxy.getObjectRegisteredName());
        proxy.setConnectionId(ERROR_CONNECTION_ID);
        ObjectStore.removeObject(proxy.getObjectRegisteredName());
        ObjectStore.removeLocalObject(proxy.getObjectRegisteredName());
        notifyAllListeners(ProtocolImpl.UNREGISTER_REQUEST_MESSAGE , proxy.getObjectRegisteredName() ,proxy.getClass().getName(), con.getId());
      }
    }
    removeListener(con);
    manager.unregisterConnection(con);
  }
}
