package com.inqmy.ats.system.communication;

import java.io.*;
import java.net.*;

import com.inqmy.ats.system.communication.protocol.Protocol;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * This class holds tcp/ip connection between two virtual machines. 
 * It has methods for reading and writing to physical connection
 * 
 * @author Tzvetan Georgiev (tsvetan.georgiev@sap.com)
 * @version 1.0
 */
public class Connection implements Runnable {

  private Socket socket;

  private InputStream in;

  private OutputStream out;

  private static Location location = Location.getLocation(Connection.class);

  private Protocol protocol;
  private byte[] b;
  private int length = 0;
  private int id;
  boolean isAlive = true;

  public Connection(Socket socket , InputStream in , OutputStream out) {
    this.socket = socket;
    this.in = in;
    this.out = out;
  }

  public void setId(int id) {
    this.id = id;
  }

  public int getId() {
    return id;
  }
  public boolean isAlive() {
    return isAlive;
  }

  public void setProtocolHandler(Protocol protocol) {
    this.protocol = protocol;
    length = protocol.getHeaderLength();
    b = new byte[length];
  }
  public Protocol getProtocolHandler(){
    return protocol;
  }
  public void run() {
    try {
      while(isAlive) {
        //reads protocol header bytes
        int readed = 0;
        int current = 0;

        while(readed >=0 && current != length) {
          readed = in.read(b , current , length - current);
          current += readed;
        }
        if(readed == -1) {
          //connection is closed before the header of the message is read!
          //finally should be executed in order to close the connection!
          break;
        }
        protocol.processHeader(b , this);
      }
    } catch(Exception exc) {
      //location.debugT("Connection read error : " + exc.getMessage());
      //location.traceThrowableT(Severity.DEBUG , "", exc);
    } finally {
      close();
    }
  }

  public void readBytes(byte[] source) throws Exception {
    int readed = 0;
    int current = 0;
    int l = source.length;
    while(readed >= 0 && current  != l) {
      readed = in.read(source , current , l-current);
      current += readed;
    }
    if(readed == -1) {
      close();
      throw new IOException("Stream unexpected closed!");
    }
  }

  public synchronized void writeMessageAndBody(byte[] m , byte[] body) throws Exception {
    try {
      byte[] send = new byte[m.length + body.length];
      System.arraycopy(m , 0 , send , 0 , m.length);
      System.arraycopy(body , 0 , send , m.length , body.length);
      out.write(send);
    } catch(Exception exc) {
      close();
      throw exc;
    }
  }

  public void close() {
    synchronized(this) {
      if(!isAlive) {
        return;
      }
      isAlive = false;
    }
    protocol.connectionClosed(this);
    try {
      socket.close();
    } catch(Exception exc) {
      exc.printStackTrace();
    }
    try {
      out.close();
    } catch(Exception exc) {
      exc.printStackTrace();
    }
    try {
      in.close();
    } catch(Exception exc) {
      exc.printStackTrace();
    }
  }
}