package com.inqmy.ats.system;

import java.io.*;
import java.util.*;

import com.inqmy.ats.boot.CoreClassLoader;

import com.inqmy.ats.boot.Utility;
import com.inqmy.ats.system.tools.*;
import com.inqmy.ats.system.tools.CommonLog;
import com.inqmy.ats.system.communication.client.CommunicationContext;
import com.inqmy.ats.system.communication.client.CommunicationContextImpl;


/**
 * Main class for loading providers implementation classes
 * it initialize providers with proper ProviderContext instance
 * 
 * @author Tzvetan Georgiev (tsvetan.georgiev@sap.com)
 * @version 1.0
 */            
public class Engine implements ProviderContext  {

  private static Engine mainEngine;
  
  public static final byte WARNING_MESSAGE = 0;
  public static final byte FATAL_MESSAGE = 1;
  public static final byte INFO_MESSAGE = 2;

  public static final String TEST_PROVIDER_PREFIX = "test_provider_class_name";
  public static final String TEST_PROVIDER_JAR_PREFIX = "test_provider_jar_name";
  public static final String PROVIDERS_PROPERTIES = "providers_properties";

  public static final String COMMUNICATION_PREFIX = "communication.";
  public static final String COMMUNICATION_PROPERTIES = "communication_properties";

  public static final String VARIABLE_PREFIX = "%";
  public static final String VARIABLES_PROPERTIES = "variables_properties";

  public static final String LOGSYSTEM_PREFIX = "logsystem.";
  public static final String LOGSYSTEM_PROPERTIES = "logsystem_properties";



  public static final String SYSTEM_PROPS_FILENAME = "./system.properties".replace('/', File.separatorChar);
  
  public static final String JVMName = "MAIN JVM";
 
  
  private CoreClassLoader coreLoader;

  private Hashtable testsProviderStore;
  private Hashtable providerRuntimeStore;

  private String[] environmentParameters;
  private CommunicationContext communication;
  private EngineControl engControl;
  private String[] providerNames;

  public void initialize(String[] env){
    testsProviderStore = new Hashtable();
    providerRuntimeStore = new Hashtable();
    coreLoader = (CoreClassLoader)Engine.class.getClassLoader();
    environmentParameters = env;
    try {
      communication = CoreContext.getCommunicationContext();
      communication.setClassLoader(coreLoader);
      engControl = new EngineControlImpl(this);
      //register engine control interface
      communication.registerObject("ENGINE_CONTROL" , engControl);
    } catch(Exception ex){
      CommonLog.log("Error register engine control interface!\n " +ex.getMessage()  , Engine.FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(ex);
    }
    try {
      communication = CoreContext.getCommunicationContext();
      RemoteFileImpl remoteFile = new RemoteFileImpl();
      //register remote file interface
      communication.registerObject("REMOTE_FILE" , remoteFile);
    } catch(Exception ex){
      CommonLog.log("Error registering remote file transfering logic !\n " +ex.getMessage()  , Engine.FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(ex);
    }
    Properties property = CoreContext.getPropertiesManager().get(PROVIDERS_PROPERTIES).getRuntimeProperties();
    Enumeration enum = property.keys();
    Vector provNames = new Vector();
    while(enum.hasMoreElements()) {
      try {
        String prov = property.getProperty((String)enum.nextElement());
        int index = prov.indexOf(';');
        String providerImplName = prov.substring(0 , index).trim();
        String jar = prov.substring(index+1).trim();

        StringTokenizer tokenizer = new StringTokenizer(jar , ";");
        int tokens = tokenizer.countTokens();
        jar = "";

        for(int i = 0 ; i < tokens ; i++) {
          String t = tokenizer.nextToken();
          if(i == (tokens - 1 )) {
            jar += new File("../lib/".replace('/' , File.separatorChar) + t).getCanonicalPath();
          } else {
            jar += new File("../lib/".replace('/' , File.separatorChar) + t).getCanonicalPath() + File.pathSeparator;
          }
        }

        //initialize provider
        String prName = initializeProvider(providerImplName , jar);
        if(prName == null){
            //todo
        } else {
          provNames.add(prName);
        }
      } catch(Exception ex){
        CommonLog.log("Can not initialize test provider!\n " +ex.getMessage() , FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
        CommonLog.log(ex);
      }
    }
    providerNames = new String[provNames.size()];
    provNames.copyInto(providerNames);
  }
  public String[] getProviderNames() {
    return this.providerNames;
  }
  public CoreClassLoader getProviderLoader(String name) {
    TestProvider provider = (TestProvider)this.testsProviderStore.get(name);
    if (provider == null) {
      return null;
    }
    return (CoreClassLoader)provider.getClass().getClassLoader();
  }
  private void start() {
    //start tests call provider start method
    Enumeration keys = this.testsProviderStore.keys();
    String name = null;
    while(keys.hasMoreElements()) {
      name = (String)keys.nextElement();
      try{
        TestProvider provider = (TestProvider)testsProviderStore.get(name);
        RuntimeControlInterface runtime = provider.getRuntimeControl();
        runtime.start();
      } catch(Throwable thr) {
        CommonLog.log("Error in start method of provider! " + name , FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
        CommonLog.log(thr);
      }
    }
    System.exit(0);
  }
  public String getGUIInterfaceClassName(String nameOfProvider) {
    TestProvider provider = (TestProvider)this.testsProviderStore.get(nameOfProvider);
    if (provider == null) {
      return null;
    }
    return provider.getGUIInterfaceClassName();
  }
 /**
  * This method return proxy class for communication of RuntimeControlInterface
  * for specified TestProvider.
  *
  * @param providerName
  * @return
  */
  public RuntimeControlInterface getProviderRuntimeControl(String providerName) {
    RuntimeControlInterface rci = (RuntimeControlInterface)this.providerRuntimeStore.get(providerName);
    return rci;
  }
  
  public String initializeProvider(String providerName , String jar) throws Exception {
    TestProvider provider;
    try {
      provider = loadTestProvider(providerName , jar);
    } catch(Throwable exc) {
      String excToStr = Utilities.exceptionToString(exc);
      CommonLog.log("Can not load test provider : " + providerName + " error is : \n" + excToStr , FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(exc);
      throw new Exception(excToStr);
    }
    try {
      provider.initialize(this); 
    } catch(Throwable exc) {
      CommonLog.log("Can not initialize test provider : " + providerName , FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(exc);
      throw new Exception(Utilities.exceptionToString(exc));
    }
    try {
      //register runtimecontrol in communication context
      RuntimeControlInterface runtimeControl = provider.getRuntimeControl();
      String prName = runtimeControl.getProviderName();
      String addName = "PROVIDER_CONTROL_INTERFACE_"+ prName;
      communication.registerObject( addName, runtimeControl);
      this.providerRuntimeStore.put( prName ,communication.getProxy(addName));
      this.testsProviderStore.put(prName , provider);
      return prName;
    } catch(Throwable exc) {
      CommonLog.log("Error register provider control interface!" , Engine.FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(exc);
      return null;
    }
    
  }
  
  public String[] getEnvironmentParameters() {
    return environmentParameters;
  }
  
  public TestProvider getProvider(String providerName) {
    return (TestProvider)this.testsProviderStore.get(providerName);
  }
  //return name of current VM using from com.inqmy.ats.system.tools.CommonLog to log with proper tag
   public String getJVMname() {
     return "MAIN ENGINE";
   }
  
   private TestProvider loadTestProvider(String  name , String jar) throws Throwable {
     TestProvider provider = null;
     CoreClassLoader providerLoader = new CoreClassLoader(coreLoader , jar);
     Class classss = providerLoader.loadClass(name);
     provider = (TestProvider)classss.newInstance();
     communication.setClassLoader(providerLoader);
     return provider;
   }
  public static void main(String[] args) throws Exception {
    try{
      initializeEngineStatic();
    } catch(Exception exc){
      CommonLog.log("Error initializing will exit emergency!" , FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      System.exit(0);
    }
    mainEngine = new Engine();
    mainEngine.initialize(args);
    if(!(args.length == 1 && args[0].equals("-s"))) {
      mainEngine.start();
    }
  }
  //loads properties needed for Engine and initializes it
  public static void initializeEngineStatic() throws Exception {
    ConsoleMessageManagerImpl mm = new ConsoleMessageManagerImpl();
    CommonLog.addMessageManager(mm);
    PropertiesManager propsManager = new PropertiesManager();
    PropertiesWrapper providerPr = null;
    PropertiesWrapper temp = null;
    boolean error = false;
    CoreContext.setJvmName(JVMName);
    //providers properties
    PropertiesWrapper varProps = new PropertiesWrapper(VARIABLES_PROPERTIES , new String[]{VARIABLE_PREFIX },null,SYSTEM_PROPS_FILENAME);
    propsManager.add(varProps);
    providerPr = new PropertiesWrapper(PROVIDERS_PROPERTIES , new String[]{TEST_PROVIDER_PREFIX },null,SYSTEM_PROPS_FILENAME);
    propsManager.add(providerPr);
    propsManager.setReference(providerPr , varProps);

    PropertiesWrapper communicationProps = new PropertiesWrapper(COMMUNICATION_PROPERTIES , new String[]{COMMUNICATION_PREFIX},null,SYSTEM_PROPS_FILENAME);
    propsManager.add(communicationProps);
    propsManager.setReference(communicationProps , varProps);
    // Log System properties
    temp = new PropertiesWrapper(LOGSYSTEM_PROPERTIES , new String[]{LOGSYSTEM_PREFIX}, null, SYSTEM_PROPS_FILENAME);
    propsManager.add(temp);
    propsManager.setReference(temp , varProps);


    try {
      //loads settings
       propsManager.loadAll();
    } catch(Exception ee) {
      CommonLog.log("Error loding system properties! Will use default settings for communication" , Engine.WARNING_MESSAGE, Utility.TRACE_MODE);
      CommonLog.log(ee);
      error = true;
    }
    //Properties properties = temp.getRuntimeProperties();
    //OutputStreamLog.setLevels(properties);

    CoreContext.setPropertiesManager(propsManager);
    try {
      Properties p = null;
      if(error){
        p = new Properties();
        p.setProperty("communication.server" , "true");
        p.setProperty("communication.server.port" , "9090");
      } else {
        p = communicationProps.getRuntimeProperties();
        p.setProperty("communication.server" , "true");
      }
      //make communication context
      CommunicationContextImpl communication = new CommunicationContextImpl(p);
      CoreContext.setCommunicationContext(communication);
    } catch(Exception ex){
      CommonLog.log("Can not initialize Communication! Please check port settings!\n Problem is : " + ex.getMessage(), Engine.FATAL_MESSAGE, Utility.LOW_LEVEL_MODE);
      CommonLog.log(ex);
      throw ex;
    }
  }

}