/*
 * Copyright (c) 2000 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_util/java/com/sap/tc/jtools/util/xml/XMLTool.java#4 $
 */

package com.sap.tc.jtools.util.xml;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.Enumeration;
import java.util.Properties;

import com.sap.engine.lib.xml.dom.DocumentImpl;
import com.sap.engine.lib.xml.dom.TextImpl;
import com.sap.engine.lib.xml.parser.DOMParser;
import com.sap.engine.lib.xml.util.DOMSerializer;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.*;

/**
 * XML-Parser that converts instances of the class StructureTree into XML
 * documents and viceversa.
 *
 *
 *
 *
 * Copyright:    Copyright (c) 2001
 * Company:      SAP AG
 * @author       BPL - Tools
 * @version 1.0
 */

public class XMLTool {

	/**
	 * Parser XML -> StructureTree
	 * @param reader XML source.
	 * @return equivalent representation as StructureTree.
	 */
	static public StructureTree parseReader(java.io.Reader reader)
		throws IOException, XMLParseException {
			StructureTree result=null;
		try {
			DOMParser tmp2 =  new DOMParser();
			InputSource source = new InputSource(reader);
			Document doc =  tmp2.parse(source);
			Element tmp1 = (Element)doc.getDocumentElement();
			result = wrapDOM(tmp1);
		} catch (Exception e) {
			e.printStackTrace();
			throw new XMLParseException();
		}
		return result;
	}

	static public StructureTree wrapDOM(Element node) {
		if (node == null)
			return null;
		String tag = node.getNodeName();
		Header header = new Header(tag);
		NamedNodeMap nnm = node.getAttributes();
		if (nnm != null) {
			for (int i = 0; i < nnm.getLength(); i++) {
				Node currentAttr = nnm.item(i);
				header.setParameter(
					currentAttr.getNodeName(),
					currentAttr.getNodeValue());
			}
		}
		StructureTree st = new StructureTree(header);
		NodeList children = node.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node currentNode = children.item(i);
			if (currentNode.getNodeType() == Node.ELEMENT_NODE) {
				st.addNewChild(wrapDOM((Element) currentNode));
			} else if (
				currentNode.getNodeType() == Node.TEXT_NODE && i == 0) {
				st.setText(currentNode.getNodeValue());
			}
		}

		return st;
	}

	static public Document unWrapDOM(StructureTree st) {
		if (st == null)
			return null;
		Document doc = new DocumentImpl();
		Element el = structureTree2Element(st, doc);
		doc.appendChild(el);
		return doc;
	}

	static private Element structureTree2Element(
		StructureTree st,
		Document doc) {

		Element el = doc.createElement(st.getTag());

		Properties params = st.getParameters();
		Enumeration keys = params.keys();
		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();
			el.setAttribute(key, params.getProperty(key));
		}
		String text = st.getText();
		if (text != null) {
			TextImpl t = new TextImpl();
			t.setNodeValue(text);
			el.appendChild(t);
		}
		StructureTree[] children = st.getChildren();
		for (int i = 0; i < children.length; i++) {
			el.appendChild(structureTree2Element(children[i], doc));
		}
		return el;
	}


	/**
	 * Unparser StructureTree array -> XML (generic writer)
	 * @param trees sources.
	 * @param outerTag outermost XML tag
	 * @param writer java.io.Writer instance the XML document is written to
	 */
	static public void writeDocument(
		StructureTree[] trees,
		String outerTag,
		java.io.Writer writer)
		throws IOException {
		if (trees == null)
			return;
		StructureTree documentTree = new StructureTree(new Header(outerTag));
		for (int i = 0; i < trees.length; i++)
			documentTree.addChild(trees[i]);
		writeDocument(documentTree, writer);
		writer.flush();
	}

	/**
	 * Unparser StructureTree -> XML (generic writer)
	 * @param tree source.
	 * @param writer java.io.Writer instance the XML document is written to
	 */
	static public void writeDocument(StructureTree tree, java.io.Writer writer)
		throws IOException {
		if (tree == null)
			return;
		Document doc = unWrapDOM(tree);
		try {
			new DOMSerializer().write(doc, writer);
		} catch (Exception e) {
			throw new IOException();
		}
		writer.flush();
	}

	/**
	 * Unparser StructureTree -> XML (string variant)
	 * @param tree source.
	 * @return string containing the XML document.
	 */
	static public String toString(StructureTree tree) {
		StringWriter sw = new StringWriter();
    BufferedWriter buf = new BufferedWriter(sw);
		Document doc = unWrapDOM(tree);
		try {
			new DOMSerializer().write(doc, buf);
			return sw.toString();
		} catch (Exception e) {
			return null;
		}
	}

}
