/*
 * Copyright (c) 2000 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_util/java/com/sap/tc/jtools/util/ascii/StringBuf.java#2 $
 */

package com.sap.tc.jtools.util.ascii;


import java.util.Vector;

/**
 * Specialized stringbuffer implementation aiming to provide fast
 * methods to modify string contents without bulk object creation orgies.
 *
 * It has an internal buffer which size is >= its actual contents length.
 * All methods are NOT synchronized.
 *
 * Please note that there is no string implementation of indexOf(offset,.. ) /
 * lastIndexOf(offset,..) due its is likely that this methods are called 
 * repeately, causing a conversion (String.toCharArray())) each invocation.
 * It is recommend to create a char array in the caller context instead and 
 * then to use the according char array based method.
 * 
 * @author    BPL Tools
 * @version   $Date: 2002/11/22 $
 */
public final class StringBuf implements java.io.Serializable {
  /** the allocated buffer size */
  private int     fCapacity;
  /** the used buffer size */
  private int     fLength;
  /** the backing character array which is used as buffer */
  private char[]  fCharArray;

  /**
   * @return new instance with a capacity of 16
   */
  public StringBuf() {
    this(16);
  }

  /**
   * @return new instance with specified capacity
   */
  public StringBuf(int pCapacity) {
    if (1 > pCapacity) {
      fCapacity = 16;
    } else {
      fCapacity = pCapacity;
    }
    fCharArray = new char[fCapacity];
    fLength = 0;
  }

  /**
   * @return new instance filled with given char
   */
  public StringBuf(int pLength, char pFillChar) {
    this(pLength);
    while (fLength < pLength) {
      fCharArray[fLength++] = pFillChar;
    }
  }

  /**
   * @return new instance filled with arrays content,
   *         having a capacity of length + 16
   */
  public StringBuf(char[] pCharArray) {
    this(null != pCharArray ? pCharArray.length + 16 : 16);
    if (null != pCharArray) {
      fLength = pCharArray.length;
      System.arraycopy(pCharArray,0,fCharArray,0,fLength);
    }
  }

  /**
   * @return new instance filled with partial arrays content,
   *         having a capacity of length + 16
   */
  public StringBuf(char[] pCharArray, int pOffset, int pLength) {
    this(null != pCharArray ? pLength + 16 : 16);
    if (null != pCharArray) {
      if (0 > pOffset || 1 > pLength || pOffset + pLength > pCharArray.length) {
        throwOutOfBounds(pOffset + pLength);
      } 
      fLength = pLength;
      System.arraycopy(pCharArray,0,fCharArray,pOffset,fLength);
    }
  }

  /**
   * @return new instance filled with string buffer content,
   *         having a capacity of length + 16
   */
  public StringBuf(StringBuf pStrBuf) {
    this(null != pStrBuf ? pStrBuf.fLength + 16 : 16);
    if (null != pStrBuf) {
      fLength = pStrBuf.fLength;
      System.arraycopy(pStrBuf.fCharArray,0,pStrBuf.fCharArray,0,fLength);
    }
  }

  /**
   * @return new instance filled with partial buffers content,
   *         having a capacity of length + 16
   */
  public StringBuf(StringBuf pStrBuf, int pOffset, int pLength) {
    this(0 > pLength ? pLength + 16 : 16);
    if (null != pStrBuf) {
      if (0 > pOffset || 1 > pLength || pOffset + pLength > pStrBuf.fLength) {
        throwOutOfBounds(pOffset+pLength);
      }
      fLength = pLength;
      System.arraycopy(pStrBuf.fCharArray,pOffset,fCharArray,0,fLength);
    }
  }

  /**
   * @return new instance filled with the content of the string,
   *         having a capacity of length + 16
   */
  public StringBuf(String pString) {
    this(null != pString ? pString.length() + 16 : 16);
    if (null != pString) {
      fLength = pString.length();
      pString.getChars(0,fLength,fCharArray,0);
    }
  }

  /**
   * @return new instance filled with parts of the string content,
   *         having a capacity of length + 16
   */
  public StringBuf(String pString, int pOffset, int pLength) {
    this(null != pString ? pString.length() + 16 : 16);
    if (null != pString) {
      if (0 > pOffset || 1 > pLength || pOffset + pLength > pString.length()) {
        throwOutOfBounds(pOffset + pLength);
      }
      fLength = pLength;
      pString.getChars(0,fLength,fCharArray,0);
    }
  }

  /**
   * resizes the buffer if its size is less then the demanded lower limit
   */
  public void ensureCapacity(int pTotalCapacity) {
    if (fCapacity < pTotalCapacity) {
      char[] newCharArray = new char[pTotalCapacity];
      System.arraycopy(fCharArray,0,newCharArray,0,fCapacity);
      fCharArray = newCharArray;
      fCapacity = pTotalCapacity;
    }
  }

  /**
   * resizes the buffer if its free space is less then the demanded lower limit
   */
  public void ensureFreeCapacity(int pFreeCapacity) {
    if (fCapacity < pFreeCapacity + fLength) {
      ensureCapacity(fLength + 4 + (int) 1.5 * pFreeCapacity);
    }
  }

  /**
   * @return new char array
   */
  public char[] createCharArray(int pOffset, int pLength) {
    if (1 > pLength || 0 > pOffset || fLength < pLength + pOffset) {
      throwOutOfBounds(pLength + pOffset);
    }
    char[] newCharArray = new char[pLength];
    System.arraycopy(fCharArray,pOffset,newCharArray,0,pLength);
    return newCharArray;
  }

  /**
   * @return new string buffer
   */
  public StringBuf createStringBuf(int pOffset, int pLength) {
    return new StringBuf(this,pOffset,pLength);
  }

  /**
   * @return new string
   */
  public String createString(int pOffset, int pLength) {
    if (1 > pLength || 0 > pOffset || fLength < pLength + pOffset) {
      throwOutOfBounds(pLength + pOffset);
    }
    return new String(fCharArray, pOffset, pLength);
  }

  /**
   * appends single character
   * @return this
   */
  public StringBuf append(char pNewChar) {
    ensureFreeCapacity(1);
    fCharArray[fLength++] = pNewChar;
    return this;
  }

  /**
   * appends the content of a char array
   * @return this
   */
  public StringBuf append(char[] pSrcArray) {
    int addLength = pSrcArray.length;
    ensureFreeCapacity(addLength);
    System.arraycopy(pSrcArray,0,fCharArray,fLength,addLength);
    fLength += addLength;
    return this;
  }

  /**
   * appends the content of another string buffer
   * @return this
   */
  public StringBuf append(StringBuf pSrcBuf) {
    if (null == pSrcBuf || 0 == pSrcBuf.fLength) {
      return this;
    }
    int addLength = pSrcBuf.fLength;
    ensureFreeCapacity(addLength);
    System.arraycopy(pSrcBuf.fCharArray,0,fCharArray,fLength,addLength);
    fLength += addLength;
    return this;
  }

  /**
   * appends the content of a string
   * @return this
   */
  public StringBuf append(String pSrcString) {
    int addLength = pSrcString.length();
    ensureFreeCapacity(addLength);
    pSrcString.getChars(0,addLength,fCharArray,fLength);
    fLength += addLength;
    return this;
  }

  /**
   * copies content into a character array
   * @param pPos       offset in buffer
   * @param pLength    number of character to copy
   * @param pDstArray  character array to copy to
   * @param pOffset    offset within the character array
   */
  public void copyValue(int pPos, int pLength, char[] pDstArray, int pOffset) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pDstArray.length) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (0 > pPos || pPos + pLength > fLength) {
      throwOutOfBounds(pPos + pLength);
    }
    System.arraycopy(fCharArray,pPos,pDstArray,pOffset,pLength);
  }

  /**
   * copies content into another buffers
   * @param pPos       offset in buffer
   * @param pLength    number of character to copy
   * @param pDstBuf    destination buffer
   * @param pOffset    offset within the destination
   */
  public void copyValue(int pPos, int pLength, StringBuf pDstBuf, int pOffset) {
    if (0 > pOffset || 1 > pLength || pOffset > pDstBuf.fLength) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (0 > pPos || pPos + pLength > fLength) {
      throwOutOfBounds(pPos + pLength);
    }
    if (pOffset + pLength > pDstBuf.fLength) {
      pDstBuf.ensureCapacity(pLength+pOffset);
      pDstBuf.fLength = pLength + pOffset;
      System.arraycopy(fCharArray,pPos,pDstBuf.fCharArray,pOffset,pLength);
    } else {
      System.arraycopy(fCharArray,pPos,pDstBuf.fCharArray,pOffset,pLength);
    }
  }

  /**
   * compares buffer content with char array
   * @return true buffer and char array have identical content
   */
  public boolean equals(char[] pCompareArray) {
    if (pCompareArray.length != fLength) return false;
    for (int ii = 0; ii < fLength; ii++) {
      if(fCharArray[ii] != pCompareArray[ii]) {
        return false;
      }
    }
    return true;
  }
      
  /**
   * compares content of other buffer with own one
   * @return true if both buffers have identical content
   */
  public boolean equals(StringBuf pCompareBuf) {

    if (pCompareBuf.fLength != fLength) return false;
    for (int ii = 0; ii < fLength; ii++) {
      if (fCharArray[ii] != pCompareBuf.fCharArray[ii]) {
        return false;
      }
    }
    return true;
  }

  /**
   * compares buffer content with String
   * @return true buffer and String have identical content
   */
  public boolean equals(String pCompareString) {
    if (pCompareString.length() != fLength) return false;
    char[] compareArray = pCompareString.toCharArray();
    for (int ii=0; ii<fLength; ii++) {
      if (fCharArray[ii] != compareArray[ii]) {
        return false;
      }
    }
    return true;
  }


  /**
   * inserts single character 
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pNewChar   character to insert
   * @return this
   */
  public StringBuf insertAt(int pPos, char pNewChar) {
    return insertAt(pPos,pNewChar,1);
  }


  /**
   * inserts single character multiple times
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pNewChar   character to insert
   * @param pRepetions number of insertions
   * @return this
   */
  public StringBuf insertAt(int pPos, char pNewChar, int pRepetitions) {
    ensureFreeCapacity(pRepetitions);
    if (fLength <= pPos || 0 > pRepetitions) {
      while (0 < pRepetitions) {
        fCharArray[fLength++] = pNewChar;
        pRepetitions--;
      }
    } else {
      // overlapping arrays copy (higher resource need, is looping better ?)
      int moveLength = fLength - pPos;
      int moveOffset = pPos + pRepetitions;
      System.arraycopy(fCharArray,pPos,fCharArray,moveOffset,moveLength);
      while (0 < pRepetitions) {
        fCharArray[pPos++] = pNewChar;
        pRepetitions--;
      }
    }
    return this;
  }

  /**
   * inserts parts of a character array
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pSrcArray  character array to inster
   * @param pOffset    offset within the character array
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf insertAt(
    int pPos, char[] pSrcArray, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcArray.length) {
      throwOutOfBounds(pOffset + pLength);
    }
    ensureFreeCapacity(pLength);
    if (fLength <= pPos ) {
      System.arraycopy(pSrcArray,pOffset,fCharArray,fLength,pLength);
    } else {
      // overlapping arrays copy (higher resource need, is looping better ?)
      int moveLength = fLength - pPos;
      int moveOffset = pPos + pLength;
      System.arraycopy(fCharArray,pPos,fCharArray,moveOffset,moveLength);
      System.arraycopy(pSrcArray,pOffset,fCharArray,pPos,pLength);
    }
    fLength += pLength;
    return this;
  }

  /**
   * inserts parts of another stringbuffer
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pSrcBuf    string buffer to insert
   * @param pOffset    offset within the source string buffer
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf insertAt(
    int pPos, StringBuf pSrcBuf, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcBuf.fLength) {
      throwOutOfBounds(pOffset + pLength);
    }
    ensureFreeCapacity(pLength);
    if (fLength <= pPos ) {
      System.arraycopy(pSrcBuf.fCharArray,pOffset,fCharArray,fLength,pLength);
    } else {
      // overlapping arrays copy (higher resource need, is looping better ?)
      int moveLength = fLength - pPos;
      int moveOffset = pPos + pLength;
      System.arraycopy(fCharArray,pPos,fCharArray,moveOffset,moveLength);
      System.arraycopy(pSrcBuf.fCharArray,pOffset,fCharArray,pPos,pLength);
    }
    fLength += pLength;
    return this;
  }

  /**
   * inserts a string
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pSrcStr    string to insert
   * @return this
   */
  public StringBuf insertAt(int pPos, String pSrcStr) {
    return insertAt(pPos,pSrcStr,0,pSrcStr.length());
  }

  /**
   * inserts parts of a string
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pSrcStr    string to insert
   * @param pOffset    offset within the string
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf insertAt(
    int pPos, String pSrcStr, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcStr.length()) {
      throwOutOfBounds(pOffset + pLength);
    }
    ensureFreeCapacity(pLength);
    if (fLength <= pPos ) {
      pSrcStr.getChars(pOffset, pOffset + pLength,fCharArray,fLength);
    } else {
      // overlapping arrays copy (higher resource need, is looping better ?)
      int moveLength = fLength - pPos;
      int moveOffset = pPos + pLength;
      System.arraycopy(fCharArray,pPos,fCharArray,moveOffset,moveLength);
      pSrcStr.getChars(pOffset,pOffset + pLength,fCharArray,pPos);
    }
    fLength += pLength;
    return this;
  }

  /**
   * modifies content with single character multiple times
   * @param pPos       offset in buffer for modification
   * @param pNewChar   new character value
   * @param pRepetions number of repetions
   * @return this
   */
  public StringBuf modifyAt(int pPos, char pNewChar, int pRepetitions) {
    if (0 > pPos || 1 > pRepetitions || pPos + pRepetitions > fLength) {
      throwOutOfBounds(pPos + pRepetitions);
    }
    while (0 < pRepetitions) {
      fCharArray[pPos++] = pNewChar;
      pRepetitions--;
    }
    return this;
  }

  /**
   * modify buffer from parts of a character array
   * @param pPos       offset in buffer for modification
   * @param pSrcArray  character array to inster
   * @param pOffset    offset within the character array
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf modifyAt(
    int pPos, char[] pSrcArray, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcArray.length) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (0 > pPos || pPos + pLength > fLength) {
      throwOutOfBounds(pPos + pLength);
    }
    System.arraycopy(pSrcArray,pOffset,fCharArray,pPos,pLength);
    return this;
  }

  /**
   * modify buffer contents from partial content of another stringbuffer
   * @param pPos       offset in buffer for modification
   * @param pSrcBuf    string buffer whose content gets copied
   * @param pOffset    offset within the source string buffer
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf modifyAt(
    int pPos, StringBuf pSrcBuf, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcBuf.fLength) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (0 > pPos || pPos + pLength > fLength) {
      throwOutOfBounds(pPos + pLength);
    }
    System.arraycopy(pSrcBuf.fCharArray,pOffset,fCharArray,pPos,pLength);
    return this;
  }

  /**
   * modify buffer contents from parts of a String
   * @param pPos       point of insertion, if out of bounds (e.g -1) append()
   * @param pSrcStr    string to insert
   * @param pOffset    offset within the string
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf modifyAt(
    int pPos, String pSrcStr, int pOffset, int pLength){
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pSrcStr.length()) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (0 > pPos || pPos + pLength > fLength) {
      throwOutOfBounds(pPos + pLength);
    }
    pSrcStr.getChars(pOffset,pOffset + pLength,fCharArray,pPos);
    return this;
  }

  /**
   * removes chars from buffer
   * @param pOffset    offset within the string
   * @param pLength    number of chars to insert
   * @return this
   */
  public StringBuf removeAt(int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > fLength) {
      throwOutOfBounds(pOffset + pLength);
    }
    if (pLength == fLength) {
      fLength = 0;
    } else {
      int posTail = pOffset + pLength;
      if (posTail < fLength) {
        int lenTail = fLength - posTail;
        System.arraycopy(fCharArray,posTail,fCharArray,pOffset,lenTail);
      }  
      fLength -= pLength;
    }
    fCharArray[fLength] = '\0';
    return this;
  }

  /**
   * replace all occurences of one character with another one
   * @return this
   */
  public StringBuf replace(char pOld, char pNew) {
    for (int ii = 0; ii < fLength; ii++) {
      if (pOld == fCharArray[ii]) {
        fCharArray[ii] = pNew;
      }
    }
    return this;
  }

  /**
   * replaces a range of chars with one character for the complete buffer
   * @return this
   */
  public StringBuf replace(char[] pOld, char pNew) {
    for (int ii = 0; ii < fLength; ii++) {
      for (int jj = 0; jj < pOld.length; jj++) {
        if (pOld[jj] == fCharArray[ii]) {
          fCharArray[ii] = pNew;
        }
      }
    }
    return this;
  }

  /**
   * replaces all occurences of an char array with another
   * @return this
   */
  public StringBuf replace(char[] pOldArray, char[] pNewArray) {
    // precheck
    int oldLen = pOldArray.length;
    int newLen = pNewArray.length;
    if (0 == oldLen)  {
      return this;
    }
    // scan all machts
    IVector modPositions = new IVector();
    {
      int curPos = 0;
      int nxtPos = indexOf(curPos,pOldArray,0,oldLen);
      while (0 <= nxtPos) {
        modPositions.add(nxtPos);
        curPos = nxtPos+ oldLen;
        nxtPos = indexOf(curPos,pOldArray,0,oldLen);
      }
      if (0 == curPos) {
        return this;                                      // no match
      }
    }
    // do the modifications
    if (oldLen == newLen) {
      // just replace not tribal movememnt
      for (int ii = 0; ii < modPositions.getLength(); ii++) {
        modifyAt(modPositions.getValue(ii),pNewArray,0,oldLen);
      }
    } else if (oldLen < newLen) {
      // new content is greater resize
      int lenDelta = newLen - oldLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      ensureFreeCapacity(sumDelta);
      // move existing values 
      int posTail = fLength;
      int lenMove, posMove, posOrig, posRead;
      for (int ii = modPositions.getLength()-1; ii >= 0;  ii--) {
        posRead = modPositions.getValue(ii);
        lenMove = posTail - posRead;
        posMove = posRead + lenDelta * (ii + 1);
        posTail = posRead;
        System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
      }
      fLength += sumDelta;
      // copy new ones
      for (int jj= 0; jj< modPositions.getLength(); jj++) {
        posOrig = modPositions.getValue(jj);
        posMove = posOrig + lenDelta * jj;
        System.arraycopy(pNewArray,0,fCharArray,posMove,newLen);
      }
    } else {
      // old content is greater
      int lenDelta = oldLen - newLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      // move existing values 
      modPositions.add(fLength);                          // dirty but good
      int posHead = modPositions.getValue(0);
      int lenMove, posMove, posOrig, posRead;      
      for (int ii = 1; ii < modPositions.getLength();  ii++) {
        posOrig = modPositions.getValue(ii);
        lenMove = posOrig - posHead;
        posRead = posHead + oldLen;
        posMove = posRead - lenDelta * ii;
        if (0 != lenMove) {
          System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
        }
        posHead = posOrig;
      }
      fLength -= sumDelta;
      // copy new ones (be cautious, extra dummy entry fLength in modPositions
      for (int jj = 0; jj< modPositions.getLength()-1; jj++) {
        posOrig = modPositions.getValue(jj);
        posMove = posOrig - lenDelta * jj;
        System.arraycopy(pNewArray,0,fCharArray,posMove,newLen);
      }
    }
    return this;
  }

  /**
   * replaces all occurences of a string buffer with another one
   * @return this
   */
  public StringBuf replace(StringBuf pOldBuf, StringBuf pNewBuf) {
    // precheck
    int oldLen = pOldBuf.fLength;
    int newLen = pNewBuf.fLength;
    if (0 == oldLen)  {
      return this;
    }
    // scan all machts
    IVector modPositions = new IVector();
    {
      int curPos = 0;
      int nxtPos = indexOf(curPos,pOldBuf,0,oldLen);
      while (0 <= nxtPos) {
        modPositions.add(nxtPos);
        curPos = nxtPos+ oldLen;
        nxtPos= indexOf(curPos,pOldBuf,0,oldLen);
      }
      if (0 == curPos) {
        return this;                                      // no match
      }
    }
    // do the modifications
    if (oldLen == newLen) {
      // just replace not tribal movememnt
      for (int ii = 0; ii < modPositions.getLength(); ii++) {
        this.modifyAt(modPositions.getValue(ii),pNewBuf,0,oldLen);
      }
    } else if (oldLen < newLen) {
      // new content is greater resize
      int lenDelta = newLen - oldLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      ensureFreeCapacity(sumDelta);
      int lenMove, posMove, posOrig, posRead;      
      // move existing values 
      int posTail = fLength;
      for (int ii = modPositions.getLength()-1; ii >= 0;  ii--) {
        posRead =  modPositions.getValue(ii);
        lenMove =  posTail - posRead;
        posMove =  posRead + lenDelta * (ii + 1);
        posTail = posRead;
        System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
      }
      fLength += sumDelta;
      // copy new ones
      for (int jj = 0; jj< modPositions.getLength(); jj++) {
        posOrig =  modPositions.getValue(jj);
        posMove =  posOrig + lenDelta * jj;
        System.arraycopy(pNewBuf.fCharArray,0,fCharArray,posMove,newLen);
      }
    } else {
      // old content is greater
      int lenDelta = oldLen - newLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      int lenMove, posMove, posOrig, posRead;      
      // move existing values 
      modPositions.add(fLength);                          // dirty but good
      int posHead = modPositions.getValue(0);
      for (int ii = 1; ii < modPositions.getLength();  ii++) {
        posOrig = modPositions.getValue(ii);
        lenMove = posOrig - posHead;
        posRead = posHead + oldLen;
        posMove = posRead - lenDelta * ii;
        if (0 != lenMove) {
          System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
        }
        posHead = posOrig;
      }
      fLength -= sumDelta;
      // copy new ones (be cautious, extra dummy entry fLength in modPositions
      for (int jj = 0; jj< modPositions.getLength()-1; jj++) {
        posOrig = modPositions.getValue(jj);
        posMove = posOrig - lenDelta * jj;
        System.arraycopy(pNewBuf.fCharArray, 0,fCharArray,posMove,newLen);
      }
    }
    return this;
  }

  /**
   * replaces all occurences of one string with another one
   * @return this
   */
  public StringBuf replace(String pOldStr, String pNewStr) {
    // precheck
    int oldLen = pOldStr.length();
    int newLen = pNewStr.length();
    if (0 == oldLen)  {
      return this;
    }
    char[] scanArray = pOldStr.toCharArray();
    // scan all machts
    IVector modPositions = new IVector();
    {
      int curPos = 0;
      int nxtPos = indexOf(curPos,scanArray,0,oldLen);
      while (0 <= nxtPos) {
        modPositions.add(nxtPos);
        curPos = nxtPos+ oldLen;
        nxtPos = indexOf(curPos,scanArray,0,oldLen);
      }
      if (0 == curPos) {
        return this;                                      // no match
      }
    }
    // do the modifications
    if (oldLen == newLen) {
      // just replace not tribal movememnt
      for (int ii = 0; ii < modPositions.getLength(); ii++) {
        this.modifyAt(modPositions.getValue(ii),pNewStr,0,oldLen);
      }
    } else if (oldLen < newLen) {
      // new content is greater resize
      int lenDelta = newLen - oldLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      ensureFreeCapacity(sumDelta);
      int lenMove, posMove, posRead, posOrig;
      // move existing values 
      int posTail = fLength;
      for (int ii = modPositions.getLength()-1; ii >= 0;  ii--) {
        posRead = modPositions.getValue(ii);
        lenMove = posTail - posRead;
        posMove = posRead + lenDelta * (ii + 1);
        posTail = posRead;
        System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
      }
      fLength += sumDelta;
      // copy new ones
      for (int jj = 0; jj< modPositions.getLength(); jj++) {
        posOrig = modPositions.getValue(jj);
        posMove = posOrig + lenDelta * jj;
        pNewStr.getChars(0,newLen,fCharArray,posMove);
      }
    } else {
      // old content is greater
      int lenDelta = oldLen - newLen;
      int sumDelta = modPositions.getLength() * lenDelta;
      int lenMove, posMove, posRead, posOrig;
      // move existing values 
      modPositions.add(fLength);                          // dirty but good
      int posHead = modPositions.getValue(0);
      for (int ii = 1; ii < modPositions.getLength();  ii++) {
        posOrig = modPositions.getValue(ii);
        lenMove = posOrig - posHead;
        posRead = posHead + oldLen;
        posMove = posRead - lenDelta * ii;
        if (0 != lenMove) {
          System.arraycopy(fCharArray,posRead,fCharArray,posMove,lenMove);
        }
        posHead = posOrig;
      }
      fLength -= sumDelta;
      // copy new ones (be cautious, extra dummy entry fLength in modPositions
      for (int jj = 0; jj< modPositions.getLength()-1; jj++) {
        posOrig = modPositions.getValue(jj);
        posMove = posOrig - lenDelta * jj;
        pNewStr.getChars(0,newLen,fCharArray,posMove);
      }
    }
    return this;
  }

  /**
   * replaces char sequence within buffer with the content of char array
   * @return this
   */
  public StringBuf replaceAt(
    int pDstPos,
    int pOldLen,
    char[] pSrcArray,
    int pSrcPos,
    int pNewLen) {
    if (pOldLen == pNewLen) {
      return this.modifyAt(pDstPos,pSrcArray,pSrcPos,pOldLen);
    }
    if (0 > pSrcPos || 1 > pNewLen || pSrcPos + pNewLen > pSrcArray.length) {
      throwOutOfBounds(pSrcPos + pNewLen);
    }
    if (0 > pDstPos || 1 > pOldLen || pDstPos + pOldLen > fLength) {
      throwOutOfBounds(pDstPos + pOldLen);
    }
    if (pOldLen < pNewLen) {
      int deltaLen = pNewLen - pOldLen;
      int moveLen = fLength - (pDstPos + pOldLen);
      ensureFreeCapacity(deltaLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos+pOldLen,
          fCharArray,pDstPos+pNewLen,
          moveLen);
      }
      fLength += deltaLen;
      System.arraycopy(
        pSrcArray,pSrcPos,
        fCharArray,pDstPos,
        pNewLen);
    } else {
      int deltaLen = pOldLen - pNewLen;
      int moveLen = fLength - (pDstPos + pOldLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos + pOldLen,
          fCharArray,pDstPos + pNewLen,
          moveLen);
      }
      fLength -= deltaLen;
      System.arraycopy(
        pSrcArray,pSrcPos,
        fCharArray,pDstPos,
        pNewLen);
    }
    return this;
  }

  /**
   * replaces char sequence within buffer with the content of another buffer
   * @return this
   */
  public StringBuf replaceAt(
    int pDstPos,
    int pOldLen,
    StringBuf pSrcBuf,
    int pSrcPos,
    int pNewLen) {
    if (pOldLen == pNewLen) {
      return this.modifyAt(pDstPos,pSrcBuf,pSrcPos,pOldLen);
    }
    if (0 > pSrcPos || 1 > pNewLen || pSrcPos + pNewLen > pSrcBuf.fLength) {
      throwOutOfBounds(pSrcPos + pNewLen);
    }
    if (0 > pDstPos || 1 > pOldLen || pDstPos + pOldLen > fLength) {
      throwOutOfBounds(pDstPos + pOldLen);
    }
    if (pOldLen < pNewLen) {
      int deltaLen = pNewLen - pOldLen;
      int moveLen =  fLength - (pDstPos + pOldLen);
      ensureFreeCapacity(deltaLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos+pOldLen,
          fCharArray,pDstPos+pNewLen,
          moveLen);
      }
      fLength += deltaLen;
      System.arraycopy(
        pSrcBuf.fCharArray,pSrcPos,
        fCharArray,pDstPos,
        pNewLen);
    } else {
      int deltaLen = pOldLen - pNewLen;
      int moveLen =  fLength - (pDstPos + pOldLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos+pOldLen,
          fCharArray,pDstPos+pNewLen,
          moveLen);
      }
      fLength -= deltaLen;
      System.arraycopy(
        pSrcBuf.fCharArray,pSrcPos,
        fCharArray,pDstPos,
        pNewLen);
    }
    return this;
  }

  /**
   * replaces char sequence within buffer with the content of a string
   * @return this
   */
  public StringBuf replaceAt(
    int pDstPos,
    int pOldLen,
    String pSrcStr,
    int pSrcPos,
    int pNewLen) {
    if ( pOldLen == pNewLen ) {
      return this.modifyAt(pDstPos,pSrcStr,pSrcPos,pOldLen);
    }
    if (0 > pSrcPos || 1 > pNewLen || pSrcPos + pNewLen > pSrcStr.length()) {
      throwOutOfBounds(pSrcPos + pNewLen);
    }
    if (0 > pDstPos || 1 > pOldLen || pDstPos + pOldLen > fLength) {
      throwOutOfBounds(pDstPos + pOldLen);
    }
     if (pOldLen < pNewLen) {
      int deltaLen = pNewLen - pOldLen;
      int moveLen =  fLength - (pDstPos + pOldLen);
      ensureFreeCapacity(deltaLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos+pOldLen,
          fCharArray,pDstPos+pNewLen,
          moveLen);
      }
      fLength += deltaLen;
      pSrcStr.getChars(pSrcPos,pSrcPos+pNewLen,fCharArray,pDstPos);
    } else {
      int deltaLen = pOldLen - pNewLen;
      int moveLen =  fLength - (pDstPos + pOldLen);
      if (0 < moveLen) {
        System.arraycopy(
          fCharArray,pDstPos+pOldLen,
          fCharArray,pDstPos+pNewLen,
          moveLen);
      }
      fLength -= deltaLen;
      pSrcStr.getChars(pSrcPos,pSrcPos+pNewLen,fCharArray,pDstPos);
    }
    return this;
  }

  /**
   * replaces any whithspaces with a simple space
   * @return this
   */
  public StringBuf replaceWhitespace() {
    for (int ii = 0; ii < fLength; ii++) {
      if (Character.isWhitespace(fCharArray[ii])) {
        fCharArray[ii] = ' ';
      }
    }
    return this;
  }

  /**
   * @return the capacity of this buffer
   */
  public int getCapacity() {
    return fCapacity;
  }

  /**
   * gets a character
   * @param pPos     the postion within the buffer
   * @return the character value or '\0' if pPos is out of bounds
   */
  public char getCharAt(int pPos) {
    if (0 > pPos || pPos >= fLength) {
      return '\0';
    } else {
      return fCharArray[pPos];
    }
  }

  /**
   * @return the acutal number of characters in this buffer
   */
  public int getLength() {
    return fLength;
  }

  /**
   * sets the acutal number of characters in this buffer if the value is
   * increased additional chars are filled with '\0'
   */
  public StringBuf setLength(int pLength) {
    if ( pLength > fLength ) {
      ensureCapacity(pLength);
      while (fLength < pLength) {
        fCharArray[fLength++] = '\0';
      }
    } else {
      if (pLength < 1) {
        fLength = 0;
        fCharArray[0] = '\0';
      } else {
        fLength = pLength;
        fCharArray[fLength] = '\0';
      }
    }
    return this;
  }

  /**
   * scans for the last occurence of a char 
   * @param  pChar the character to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(char pChar) {
    return lastIndexOf(fLength-1,pChar);
  }

  /**
   * scans for the last occurence of a char
   * @param  pUprNdx the index within the buffer to start the backward search
   * @param  pChar the character to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(int pUprNdx, char pChar) {
    if (pUprNdx >= fLength ) {
      pUprNdx = fLength -1;
    }
    for (int ii = pUprNdx; ii >= 0 ; ii--) {
      if (fCharArray[ii] == pChar) {
        return ii;
      }
    }
    return -1;
  }

  /**
   * scans for the last occurence of a char arrays content
   * @param  pArray  the array whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(char[] pArray) {
    return lastIndexOf(0,pArray,0,pArray.length);
  }

  /**
   * scans for the last occurence of a char arrays content
   * @param  pUprNdx the index within the buffer to start the backward search
   * @param  pArray  the array whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(int pUprNdx, char[] pArray) {
    return lastIndexOf(pUprNdx,pArray,0,pArray.length);
  }

  /**
   * scans for the last occurence of a String
   * @param  pString the string whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(String pString) {
    return lastIndexOf(pString.toCharArray());
  }

  /**
   * scans for the last occurence of a char arrays content
   * @param  pUprNdx the index within the buffer to start the backward search
   * @param  pArray  the array whose content is to search
   * @param  pOffset number of leading characters not to compare
   * @param  pLength number of characters to compare
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(int pUprNdx, char[] pArray, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pArray.length) {
      throwOutOfBounds(pOffset+ pLength);
    }
    if (0 > pUprNdx || pUprNdx > fLength) {
      return -1;
    }
    if (pUprNdx + pLength > fLength) {
      pUprNdx = fLength - pLength;
    }
    outer: for (int ii = pUprNdx; ii >= 0; ii--) {
      for (int jj = pOffset, kk = ii; jj < pLength; jj++, kk++) {
        if (pArray[jj] != fCharArray[kk]) continue outer;
      }
      return ii;
    }
    return -1;
  }

  /**
   * scans for the last occurence of a other buffers content
   * @param  pBuf    the buffer whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(StringBuf pBuf){
    return lastIndexOf(this.fLength,pBuf,0,pBuf.fLength);
  }

  /**
   * scans for the last occurence of a other buffers content
   * @param  pUprNdx the index within the buffer to start the backward search
   * @param  pBuf    the buffer whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(int pUprNdx, StringBuf pBuf){
    return lastIndexOf(pUprNdx,pBuf,0,pBuf.fLength);
  }

  /**
   * scans for the last occurence of a other buffers content
   * @param  pUprNdx the index within the buffer to start the backward search
   * @param  pBuf    the buffer whose content is to search
   * @param  pOffset number of leading characters not to compare
   * @param  pLength number of characters to compare
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int lastIndexOf(int pUprNdx, StringBuf pBuf, int pOffset, int pLength){
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pBuf.fLength) {
      throwOutOfBounds(pOffset+ pLength);
    }
    if (0 > pUprNdx || pUprNdx > fLength) {
      return -1;
    }

    if (pUprNdx + pLength > fLength) {
      pUprNdx = fLength - pLength;
    }
    char scanArray[] = pBuf.fCharArray;
    outer: for (int ii = pUprNdx; ii >= 0; ii--) {
      for (int jj = pOffset, kk = ii; jj < pLength; jj++, kk++) {
        if (scanArray[jj] != fCharArray[kk]) continue outer;
      }
      return ii;
    }
    return -1;
  }

  /**
   * scans for the last occurence of a char 
   * @param  pChar    the character to search               
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(char pChar) {
    return indexOf(0,pChar);
  }

  /**
   * scans for the last occurence of a char
   * @param  pLwrNdx the index within the buffer to start the forard search
   * @param  pChar    the character to search               
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(int pLwrNdx, char pChar) {
    if (0 > pLwrNdx) {
      return -1;
    }
    for (int ii = pLwrNdx; ii < fLength ; ii++) {
      if (fCharArray[ii] == pChar) {
        return ii;
      }
    }
    return -1;
  }

  /**
   * scans for the first occurence of a char arrays content
   * @param  pArray  the array whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(char[] pArray) {
    return indexOf(0,pArray,0,pArray.length);
  }

  /**
   * scans for the first occurence of a char arrays content
   * @param  pLwrNdx the index within the buffer to start the forard search
   * @param  pArray  the array whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(int pLwrNdx, char[] pArray) {
    return indexOf(pLwrNdx,pArray,0,pArray.length);
  }

  /**
   * scans for the first occurence of a char arrays content
   * @param  pLwrNdx the index within the buffer to start the forard search
   * @param  pArray  the array whose content is to search
   * @param  pOffset number of leading characters not to compare
   * @param  pLength number of characters to compare
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(int pLwrNdx, char[] pArray, int pOffset, int pLength) {
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pArray.length) {
      throwOutOfBounds(pOffset+ pLength);
    }
    if (0 > pLwrNdx || pLwrNdx > fLength) {
      return -1;
    }
    int lastPos = fLength - pLength;
    outer: for (int ii = pLwrNdx; ii <= lastPos; ii++) {
      for (int jj= pOffset, kk= ii; jj < pLength; jj++, kk++) {
        if (pArray[jj] != fCharArray[kk]) continue outer;
      }
      return ii;
    }
    return -1;
  }

  /**
   * scans for the first occurence of a String
   * @param  pString the string whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(String pString) {
    return indexOf(pString.toCharArray());
  }

  /**
   * scans for the first occurence of a other buffers content
   * @param  pBuf    the buffer whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(StringBuf pBuf){
    return indexOf(0,pBuf,0,pBuf.fLength);
  }

  /**
   * scans for the first occurence of a other buffers content
   * @param  pLwrNdx the index within the buffer to start the forward search
   * @param  pBuf    the buffer whose content is to search
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(int pLwrNdx, StringBuf pBuf){
    return indexOf(pLwrNdx,pBuf,0,pBuf.fLength);
  }

  /**
   * scans for the first occurence of a other buffers content
   * @param  pLwrNdx the index within the buffer to start the forward search
   * @param  pBuf    the buffer whose content is to search
   * @param  pOffset number of leading characters not to compare
   * @param  pLength number of characters to compare
   * @return -1 if no match, else its position/offset within the buffer
   */
  public int indexOf(int pLwrNdx, StringBuf pBuf, int pOffset, int pLength){
    if (0 > pOffset || 1 > pLength || pOffset + pLength > pBuf.fLength) {
      throwOutOfBounds(pOffset+ pLength);
    }
    if (0 > pLwrNdx || pLwrNdx > fLength) {
      return -1;
    }
    char[] cmpArray = pBuf.fCharArray;
    int lastPos = fLength - pLength;
    outer: for (int ii = pLwrNdx; ii <= lastPos; ii++) {
      for (int jj = pOffset, kk = ii; jj < pLength; jj++, kk++) {
        if (cmpArray[jj] != fCharArray[kk]) continue outer;
      }
      return ii;
    }
    return -1;
  }

  /**
   * checks for a specific ending sequence  
   **/
  public boolean endsWith(char [] postfix) {
    int offset = this.fLength - postfix.length;
    if (0 <= offset) {
      int count = 0;
      char[] cmpArray = postfix;
      while (count < postfix.length) {
        if ( cmpArray[count] != this.fCharArray[offset + count] ) {
          return false;
        }
        count++;
      }
      return true;
    }
    return false;
  }

  /**
   * checks for a specific ending sequence  
   **/
  public boolean endsWith(String postfix) {
    int offset = this.fLength - postfix.length();
    if (0 <= offset) {
      int count = 0;
      char[] cmpArray = postfix.toCharArray();
      while (count < postfix.length()) {
        if ( cmpArray[count] != this.fCharArray[offset + count] ) {
          return false;
        }
        count++;
      }
      return true;
    }
    return false;
  }

  /**
   * checks for a specific ending sequence  
   **/
  public boolean endsWith(StringBuf postfix) {
    int offset = this.fLength - postfix.fLength;
    if (0 <= offset) {
      int count = 0;
      char[] cmpArray = postfix.fCharArray;
      while (count < postfix.fLength) {
        if ( cmpArray[count] != this.fCharArray[offset + count] ) {
          return false;
        }
        count++;
      }
      return true;
    }
    return false;
  }


  /**
   * Splits the stringbuffer at char array into several ones.
   * @param  pArray  the array whose content is to search
   * @return array of partial strings
   */
  public String[] splitToString(char[] pSeparator) {
    if (null == pSeparator || 0 == pSeparator.length) {
      throwOutOfBounds(0);
    }    
    int pos_cur = 0;
    int pos_nxt = 0;
    int len_nxt;
    Vector result;
    
    pos_nxt = this.indexOf(pos_cur, pSeparator, 0, pSeparator.length);
    if (0 > pos_nxt) {
      return new String[] { toString() };
    }

    result = new Vector(10);
        
    len_nxt = pos_nxt - pos_cur;
    if ( 0 < len_nxt ) {
      result.add(createString(pos_cur,len_nxt));
    }
    pos_cur = pos_nxt + pSeparator.length;
    
    while ( pos_cur < fLength ) {
      pos_nxt = this.indexOf(pos_cur, pSeparator, 0, pSeparator.length);
      if (0 > pos_nxt) {
        len_nxt = fLength - pos_cur;
        result.add(createString(pos_cur,len_nxt));
        pos_cur = fLength + 1;
      } else {
        len_nxt = pos_nxt - pos_cur;
        if ( 0 < len_nxt ) {
          result.add(createString(pos_cur,len_nxt));
        }
        pos_cur = pos_nxt + pSeparator.length;
      }
    }

    String[] resultArray = new String[result.size()];
    result.copyInto(resultArray);
    return resultArray;
  }


  /**
   * Splits the stringbuffer at char array into several ones.
   * @param  pArray  the array whose content is to search
   * @return array of partial stringbuffers
   */
  public StringBuf[] splitToStringBuf(char[] pSeparator) {
    
    if (null == pSeparator || 0 == pSeparator.length) {
      throwOutOfBounds(0);
    }    
    int pos_cur = 0;
    int pos_nxt = 0;
    int len_nxt;
    Vector result;
    
    pos_nxt = this.indexOf(pos_cur, pSeparator, 0, pSeparator.length);
    if (0 > pos_nxt) {
      return new StringBuf[] { new StringBuf(this) };
    }

    result = new Vector(10);
        
    len_nxt = pos_nxt - pos_cur;
    if ( 0 < len_nxt ) {
      result.add(createStringBuf(pos_cur,len_nxt));
    }
    pos_cur = pos_nxt + pSeparator.length;
    
    while ( pos_cur < fLength ) {
      pos_nxt = this.indexOf(pos_cur, pSeparator, 0, pSeparator.length);
      if (0 > pos_nxt) {
        len_nxt = fLength - pos_cur;
        result.add(createStringBuf(pos_cur,len_nxt));
        pos_cur = fLength + 1;
      } else {
        len_nxt = pos_nxt - pos_cur;
        if ( 0 < len_nxt ) {
          result.add(createStringBuf(pos_cur,len_nxt));
        }
        pos_cur = pos_nxt + pSeparator.length;
      }
    }

    StringBuf[] resultArray = new StringBuf[result.size()];
    result.copyInto(resultArray);
    return resultArray;
  }


  /**
   * checks for a specific starting sequence  
   **/
  public boolean startsWith(char [] prefix) {
    int upperNdx = prefix.length;
    if (this.fLength >= upperNdx) {
      char[] cmpArray = prefix;
      while (upperNdx-- > 0) {
        if ( cmpArray[upperNdx] != this.fCharArray[upperNdx] ) {
          return false;
        }
      }
      return true;
    }
    return false;
  }

  /**
   * checks for a specific starting sequence  
   **/
  public boolean startsWith(String prefix) {
    int upperNdx = prefix.length();
    if (this.fLength >= upperNdx) {
      char[] cmpArray = prefix.toCharArray();
      while (upperNdx-- > 0) {
        if ( cmpArray[upperNdx] != this.fCharArray[upperNdx] ) {
          return false;
        }
      }
      return true;
    }
    return false;
  }
 
  /**
   * checks for a specific starting sequence  
   **/
  public boolean startsWith(StringBuf prefix) {
    int upperNdx = prefix.fLength;
    if (this.fLength >= upperNdx) {
      char[] cmpArray = prefix.fCharArray;
      while (upperNdx-- > 0) {
        if ( cmpArray[upperNdx] != this.fCharArray[upperNdx] ) {
          return false;
        }
      }
      return true;
    }
    return false;
  }


  /**
   * just throws an runtime exception
   */
  private void throwOutOfBounds(int length) {
    throw new StringIndexOutOfBoundsException(length);
  }

  /**
   * converts any uppercase letter to a lower case one
   * @return this
   * @see Character#toLowerCase
   */
  public StringBuf toLower() {
    for (int ii = 0; ii < fLength; ii++) {
      if (Character.isUpperCase(fCharArray[ii])) {
        fCharArray[ii] = Character.toLowerCase(fCharArray[ii]);
      }
    }
    return this;
  }

  /**
   * converts any lowercase letter to an upper case one
   * @return this
   * @see Character#toUpperCase
   */
  public StringBuf toUpper() {
    for (int ii = 0; ii < fLength; ii++) {
      if (Character.isLowerCase(fCharArray[ii])) {
        fCharArray[ii] = Character.toUpperCase(fCharArray[ii]);
      }
    }
    return this;
  }

  /**
   * @return new string
   */
  public String toString() {
    return new String(fCharArray, 0, fLength);
  }

  /**
   * removes any trailing or leading white spaces
   * @return this
   */
  public StringBuf trim() {
    // handle trailing ones
    int jj = fLength-1;
    while (0 < fLength && Character.isWhitespace(fCharArray[jj])) {
      jj--;
      fLength--;
    }
    // handle leading ones
    jj = 0;
    while (jj < fLength && Character.isWhitespace(fCharArray[jj])) {
      jj++;
    }
    if ( 0 != jj ) {
      fLength -= jj;
      System.arraycopy(fCharArray,jj,fCharArray,0,fLength);
    }
    return this;
  }


  /**
   * inner class to handle dynamically growing array of integers
   */
  
  static class IVector {
  
    private int[] fIntArray;
    private int   fLength;
    private int   fCapacity;
    
    protected IVector() {
      fIntArray = new int[10];
      fLength =   0;
      fCapacity = 10;
    }
  
    protected void add(int pValue)  {
      if (fLength == fCapacity) {
        fCapacity += 10;
        int [] newArray = new int[fCapacity];
        System.arraycopy(fIntArray,0,newArray,0,fLength);
        fIntArray = newArray;
      }
      fIntArray[fLength++] = pValue;
    }
    
    protected int getLength() {
      return fLength;
    }
    
    protected int getValue(int pIndex) {
      if (0 > pIndex  || pIndex > fLength) {
        throw new ArrayIndexOutOfBoundsException(pIndex);
      }
      return fIntArray[pIndex];
    }
  }
}
