/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_devel/src/com/sap/tc/jtools/jlint/eclipse/devel/wizards/TestClassMemberCreator.java#3 $
 */
package com.sap.tc.jtools.jlint.eclipse.devel.wizards;

import java.util.ArrayList;
import java.util.Arrays;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.corext.codemanipulation.CodeGenerationSettings;
import org.eclipse.jdt.internal.corext.codemanipulation.StubUtility;
import org.eclipse.jdt.internal.ui.preferences.JavaPreferencesSettings;
import com.sap.tc.jtools.jlint.eclipse.devel.DevelPlugin;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * @author d037913
 * 
 * Helper class for adding members (attributes and methods) to the Jlin test
 * class
 */
public class TestClassMemberCreator {

  private static final String  NAME_JAVADOC          = "\n/* keep in sync with tests.xml! */\n\n";
  private static final String  GET_TEST_NAME_CONTENT = "\npublic String getTestName() {\n return NAME;\n}\n\n";
  private static final String  STRUC_TREE_QUAL_NAME  = StructureTree.class
                                                         .getName();
  private IType                fTestClassType;
  private String               fTestName;
  private ParameterInterface[] fTestParams;

  /**
   * Constructor for TestClassMemberCreator.
   */
  public TestClassMemberCreator(IType type, String testDisplayName,
      ParameterInterface[] testParams) {
    fTestClassType = type;
    fTestName = testDisplayName;
    this.fTestParams = testParams;
  }

  public void createMembers(IProgressMonitor monitor) {
    ArrayList newMethods = new ArrayList();
    ITypeHierarchy hierarchy = null;
    CodeGenerationSettings settings = JavaPreferencesSettings
        .getCodeGenerationSettings();
    try {
      hierarchy = fTestClassType.newSupertypeHierarchy(monitor);
      // override inherited constructors
      IType superclass = hierarchy.getSuperclass(fTestClassType);
      if (superclass != null) {
        String[] constructors = StubUtility.evalConstructors(fTestClassType,
            superclass, settings, null);
        if (constructors != null) {
            newMethods.addAll(Arrays.asList(constructors));
        }
      }
      // create test name member
      fTestClassType.createField(NAME_JAVADOC
          + "private static final String NAME = \"" + fTestName + "\";\n\n",
          null, false, new SubProgressMonitor(monitor, 1));
      // override abstract getName() method
      fTestClassType.createMethod(GET_TEST_NAME_CONTENT, null, false,
          new SubProgressMonitor(monitor, 1));
      // create getter methods for input parameters
      createParamGetterMethods(monitor);
      // create remaining inherited abstract methods
      if (hierarchy != null) {
        String[] unimplemented = StubUtility.evalUnimplementedMethods(
            fTestClassType, hierarchy, false, settings, null, null);
        if (unimplemented != null) {
          newMethods.addAll(Arrays.asList(unimplemented));
        }
      }
      IMethod[] createdMethods = new IMethod[newMethods.size()];
      for (int i = 0; i < newMethods.size(); i++) {
        String content = (String) newMethods.get(i) + '\n';
        // content will be formatted, ok to use \n
        createdMethods[i] = fTestClassType.createMethod(content, null, false,
            new SubProgressMonitor(monitor, 1));
      }
    } catch (JavaModelException e) {
      DevelPlugin.log(e);
    } catch (CoreException e) {
      DevelPlugin.log(e);
    } finally {
      if (monitor != null) {
        monitor.done();
      }
    }
  }

  /**
   * Method createParamGetterMethods.
   * 
   * @param type
   * @param monitor
   */
  private void createParamGetterMethods(IProgressMonitor monitor) {
    for (int i = 0; i < fTestParams.length; i++) {
      try {
        fTestClassType.createMethod(getContents(fTestParams[i]), null, false,
            new SubProgressMonitor(monitor, 1));
        if (fTestParams[i].getType().equals(
            ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {
          fTestClassType.getCompilationUnit().createImport(
              STRUC_TREE_QUAL_NAME, null, null);
        }
      } catch (JavaModelException e) {
        ExceptionHandler.handle(e);
      }
    }
  }

  /**
   * Method getContents.
   * 
   * create method content for parameter getter method
   * 
   * @param parameterInterface
   * @return String
   */
  private String getContents(ParameterInterface param) {
    StringBuffer content = new StringBuffer();
    String type = param.getType();
    String typeName = null;
    if (ParameterTool.isArrayType(type)) {
      typeName = "String[]";
    } else if (ParameterTool.isSimpleType(type)) {
      if (ParameterTool.PAR_TYPE_FILE.equals(param.getType())
          || ParameterTool.PAR_TYPE_DIRECTORY.equals(param.getType())
          || ParameterTool.PAR_TYPE_STRING.equals(param.getType())) {
        typeName = "String";
      } else if (ParameterTool.PAR_TYPE_BOOLEAN.equals(param.getType())) {
        typeName = "Boolean";
      } else if (ParameterTool.PAR_TYPE_INT.equals(param.getType())) {
        typeName = "Integer";
      } else if (ParameterTool.PAR_TYPE_FLOAT.equals(param.getType())) {
        typeName = "Float";
      }
    } else {
      typeName = "StructureTree";
    }
    content.append("\nprivate " + typeName + " get"
        + capitalizeFirstLetter(param.getName()) + "Param() {\n");
    content.append(" return (" + typeName + ") getInputParameter(\""
        + param.getName() + "\");\n");
    content.append("}\n\n");
    return content.toString();
  }

  /**
   * Method capitalizeFirstLetter.
   * 
   * @param string
   * @return String
   */
  private String capitalizeFirstLetter(String string) {
    return string.substring(0, 1).toUpperCase() + string.substring(1);
  }
}
