/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_devel/src/com/sap/tc/jtools/jlint/eclipse/devel/wizards/MainPage.java#1 $
 */
package com.sap.tc.jtools.jlint.eclipse.devel.wizards;

import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.IJavaHelpContextIds;
import org.eclipse.jdt.internal.ui.dialogs.StatusInfo;
import org.eclipse.jdt.internal.ui.util.SWTUtil;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.ComboDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IStringButtonAdapter;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.StringButtonDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.StringDialogField;
import org.eclipse.jdt.ui.wizards.NewTypeWizardPage;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.wizard.IWizardContainer;
import org.eclipse.pde.core.plugin.IPluginImport;
import org.eclipse.pde.internal.PDE;
import org.eclipse.pde.internal.core.ModelEntry;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.internal.dialogs.FileFolderSelectionDialog;

import com.sap.tc.jtools.jlint.TestDescriptionSet;
import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.devel.extension.ExtensionManager;
import com.sap.tc.jtools.jlint.eclipse.devel.extension.ITestGenerator;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * Wizard page to create a new JLin test class.
 * <p>
 * Note: This class is not intended to be subclassed. To implement a different
 * kind of a new class wizard page, extend <code>NewTypeWizardPage</code>.
 * </p>
 *  
 */
public class MainPage extends NewTypeWizardPage {

  private class DescriptorButtonAdapter implements IStringButtonAdapter {

    public void changeControlPressed(DialogField field) {
      handleDescrButtonPressed();
    }
  }
  private class XMLFileFilter extends ViewerFilter {

    public boolean select(Viewer viewer, Object parentElement, Object element) {
      if (element instanceof File) {
        File file = (File) element;
        if ((file.getAbsolutePath().toLowerCase().endsWith(".xml"))
            || file.isDirectory()) //$NON-NLS-1$
          return true;
      }
      return false;
    }
  }
  private final static String     DESCRIPTION           = "field.description";        //$NON-NLS-1$
  private final static String     DESCRIPTOR_FILE       = "field.descriptorfile";     //$NON-NLS-1$
  private final static String     DISPLAY_NAME          = "field.displayname";        //$NON-NLS-1$
  private static ExtensionManager extensionManager      = ExtensionManager
                                                            .getExtensionManager();
  private final static String     PAGE_NAME             = "JlinNewClassWizardPage";   //$NON-NLS-1$
  private final static String     RESPONSIBLE           = "field.responsible";        //$NON-NLS-1$
  private static final String[]   TEST_TYPES            = extensionManager
                                                            .getTestGeneratorNames();
  private IStatus                 fDescriptionStatus    = new StatusInfo();
  private StringButtonDialogField fDescriptorFileDialogField;
  private IStatus                 fDescriptorFileStatus = new StatusInfo();
  private StringDialogField       fResponsibleDialogField;
  private IStatus                 fResponsibleStatus    = new StatusInfo();
  private StringDialogField       fTestDescriptionDialogField;
  private StringDialogField       fTestGroupDialogField;
  /* JLin test features */
  private StringDialogField       fTestNameDialogField;
  private IStatus                 fTestNameStatus       = new StatusInfo();
  private ComboDialogField        fTestTypeDialogField;
  private IStatus                 fProjectStatus        = new StatusInfo();
  private IProject                selectedProject;

  public MainPage() {
    super(true, PAGE_NAME);
    setTitle(TextKeyDevelWizards.NEW_WIZ_PAGETITLE.getText());
    setDescription(TextKeyDevelWizards.NEW_WIZ_PAGEDESCR.getText());
    // initialize GUI elements
    fTestNameDialogField = new StringDialogField();
    fTestNameDialogField.setLabelText(TextKeyDevelWizards.DISP_NAME_LABEL
        .getText());
    fTestNameDialogField.setDialogFieldListener(new IDialogFieldListener() {

      public void dialogFieldChanged(DialogField field) {
        handleFieldChanged(DISPLAY_NAME);
      }
    });
    updateDisplayNameStatus();
    fTestTypeDialogField = new ComboDialogField(SWT.READ_ONLY);
    fTestTypeDialogField.setLabelText(TextKeyDevelWizards.TYPE_LABEL.getText());
    fTestTypeDialogField.setItems(TEST_TYPES);
    fTestTypeDialogField.selectItem(0);
    fResponsibleDialogField = new StringDialogField();
    fResponsibleDialogField.setLabelText(TextKeyDevelWizards.RESP_LABEL
        .getText());
    fResponsibleDialogField.setDialogFieldListener(new IDialogFieldListener() {

      public void dialogFieldChanged(DialogField field) {
        handleFieldChanged(RESPONSIBLE);
      }
    });
    updateResponsibleStatus();
    fTestDescriptionDialogField = new StringDialogField();
    fTestDescriptionDialogField.setLabelText(TextKeyDevelWizards.DESCR_LABEL
        .getText());
    fTestDescriptionDialogField
        .setDialogFieldListener(new IDialogFieldListener() {

          public void dialogFieldChanged(DialogField field) {
            handleFieldChanged(DESCRIPTION);
          }
        });
    updateDescrStatus();
    fTestGroupDialogField = new StringDialogField();
    fTestGroupDialogField.setLabelText(TextKeyDevelWizards.GROUP_LABEL
        .getText());
    fDescriptorFileDialogField = new StringButtonDialogField(
        new DescriptorButtonAdapter());
    fDescriptorFileDialogField
        .setLabelText(TextKeyDevelWizards.DESCR_FILE_LABEL.getText());
    fDescriptorFileDialogField.setButtonLabel(TextKeyDevelWizards.BROWSE_LABEL
        .getText());
    fDescriptorFileDialogField
        .setDialogFieldListener(new IDialogFieldListener() {

          public void dialogFieldChanged(DialogField field) {
            handleFieldChanged(DESCRIPTOR_FILE);
          }
        });
    updateDescrFileStatus();
  }

  // ------ ui --------
  /*
   * @see WizardPage#createControl
   */
  public void createControl(Composite parent) {
    initializeDialogUnits(parent);
    Composite composite = new Composite(parent, SWT.NONE);
    int nColumns = 4;
    GridLayout layout = new GridLayout();
    layout.numColumns = nColumns;
    composite.setLayout(layout);
    // pick & choose the wanted UI components
    createContainerControls(composite, nColumns);
    createPackageControls(composite, nColumns);
    createTypeNameControls(composite, nColumns);
    createSeparator(composite, nColumns);
    createTestNameControls(composite, nColumns);
    createTestTypeControls(composite, nColumns);
    createResponsibleControls(composite, nColumns);
    createTestGroupControls(composite, nColumns);
    createTestDescriptionControls(composite, nColumns);
    createDescriptorFileControls(composite, nColumns);
    WorkbenchHelp.setHelp(composite, IJavaHelpContextIds.NEW_CLASS_WIZARD_PAGE);
    doStatusUpdate();
    setControl(composite);
  }

  private void createDescriptorFileControls(Composite composite, int nColumns) {
    fDescriptorFileDialogField.doFillIntoGrid(composite, nColumns);
    LayoutUtil.setWidthHint(fDescriptorFileDialogField.getTextControl(null),
        getMaxFieldWidth());
  }

  private void createResponsibleControls(Composite composite, int nColumns) {
    fResponsibleDialogField.doFillIntoGrid(composite, nColumns - 1);
    fResponsibleDialogField.setText(System.getProperty("user.name", "")); //$NON-NLS-1$
    DialogField.createEmptySpace(composite);
    LayoutUtil.setWidthHint(fResponsibleDialogField.getTextControl(null),
        getMaxFieldWidth());
  }

  private void createTestDescriptionControls(Composite composite, int nColumns) {
    fTestDescriptionDialogField.doFillIntoGrid(composite, nColumns - 1);
    DialogField.createEmptySpace(composite);
    LayoutUtil.setWidthHint(fTestDescriptionDialogField.getTextControl(null),
        getMaxFieldWidth());
  }

  private void createTestGroupControls(Composite composite, int nColumns) {
    fTestGroupDialogField.doFillIntoGrid(composite, nColumns - 1);
    DialogField.createEmptySpace(composite);
    LayoutUtil.setWidthHint(fTestGroupDialogField.getTextControl(null),
        getMaxFieldWidth());
  }

  private void createTestNameControls(Composite composite, int nColumns) {
    fTestNameDialogField.doFillIntoGrid(composite, nColumns - 1);
    DialogField.createEmptySpace(composite);
    LayoutUtil.setWidthHint(fTestNameDialogField.getTextControl(null),
        getMaxFieldWidth());
  }

  private void createTestTypeControls(Composite composite, int nColumns) {
    fTestTypeDialogField.doFillIntoGrid(composite, nColumns - 1);
    DialogField.createEmptySpace(composite);
    LayoutUtil.setWidthHint(fTestTypeDialogField.getComboControl(null),
        getMaxFieldWidth());
    fTestTypeDialogField.setDialogFieldListener(new IDialogFieldListener() {

      public void dialogFieldChanged(DialogField field) {
        IWizardContainer container = getWizard().getContainer();
        superClassChanged();
        container.updateButtons();
        doStatusUpdate();
      }
    });
  }

  // ---- creation ----------------
  /*
   * @see NewTypeWizardPage#createTypeMembers
   */
  protected void createTypeMembers(IType type, ImportsManager imports,
      IProgressMonitor monitor) throws CoreException {
    getTestGenerator().createTypeMembers(type, imports);
    ParameterInterface[] params = ((NewTestWizard) getWizard())
        .getParametersPage().getTestParameters();
    new TestClassMemberCreator(type, getTestName(), params)
        .createMembers(monitor);
  }

  // ------ validation --------
  private void doStatusUpdate() {
    // status of all used components
    IStatus[] status = new IStatus[]{fContainerStatus, fProjectStatus,
        fPackageStatus, fTypeNameStatus, fModifierStatus, superClassChanged(),
        fSuperInterfacesStatus, fTestNameStatus, fResponsibleStatus,
        fDescriptionStatus, fDescriptorFileStatus};
    // the mode severe status will be displayed and the ok button
    // enabled/disabled.
    updateStatus(status);
  }

  /**
   * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#getEnclosingType()
   */
  public IType getEnclosingType() {
    return null; // no inner classes allowed
  }

  /**
   * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#getModifiers()
   */
  public int getModifiers() {
    return F_PUBLIC; // test class is public
  }

  public IProject getProject() {
    return selectedProject;
  }

  /**
   * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#getSuperClass()
   */
  public String getSuperClass() {
    return getTestGenerator().getSuperClass();
  }

  private ITestGenerator getTestGenerator() {
    return ExtensionManager.getExtensionManager().getTestGenerator(
        getTestType());
  }

  /**
   * @see org.eclipse.jdt.ui.wizards.NewTypeWizardPage#getSuperInterfaces()
   */
  public List getSuperInterfaces() {
    return Collections.EMPTY_LIST; // no superinterfaces
  }

  /**
   * Returns the testDescription.
   * 
   * @return String
   */
  public String getTestDescription() {
    return fTestDescriptionDialogField.getText();
  }

  public File getTestDescriptor() {
    return new File(selectedProject.getLocation().toFile(),
        fDescriptorFileDialogField.getText().trim());
  }

  public IFile getTestDescrIFile() {
    return selectedProject.getFile(fDescriptorFileDialogField.getText().trim());
  }

  public String getTestGroup() {
    return fTestGroupDialogField.getText().trim();
  }

  public String getTestName() {
    return fTestNameDialogField.getText().trim();
  }

  public String getTestResponsible() {
    return fResponsibleDialogField.getText().trim();
  }

  private void handleDescrButtonPressed() {
    FileFolderSelectionDialog dlg = new FileFolderSelectionDialog(getControl()
        .getShell(), false, IResource.FILE);
    dlg.setInput(selectedProject.getLocation().toFile());
    dlg.addFilter(new XMLFileFilter());
    dlg.setValidator(new ISelectionStatusValidator() {

      public IStatus validate(Object[] selection) {
        if (selection.length > 0 && (selection[0] instanceof File)) {
          File descrFile = (File) selection[0];
          if (descrFile.getAbsolutePath().toLowerCase().endsWith(".xml")) { //$NON-NLS-1$
            try {
              new TestDescriptionSet(XMLTool.parseReader(new FileReader(
                  descrFile)));
            } catch (Exception e) {
              return new StatusInfo(IStatus.ERROR,
                  TextKeyDevelWizards.INVALID_DESCRIPTOR.getText());
            }
            return new StatusInfo();
          }
        }
        return new StatusInfo(IStatus.ERROR,
            TextKeyDevelWizards.SELECT_XML_FILE.getText());
      }
    });
    if (Dialog.OK == dlg.open()) {
      Object result = dlg.getFirstResult();
      if (result instanceof File) {
        // file path should be project-relative
        File descrFile = (File) result;
        fDescriptorFileDialogField
            .setText(new Path(descrFile.getAbsolutePath()).removeFirstSegments(
                selectedProject.getLocation().segmentCount()).setDevice(null)
                .toString());
      }
    }
  }

  /*
   * @see NewContainerWizardPage#handleFieldChanged
   */
  protected void handleFieldChanged(String fieldName) {
    super.handleFieldChanged(fieldName);
    if (DISPLAY_NAME.equals(fieldName)) {
      updateDisplayNameStatus();
    } else if (RESPONSIBLE.equals(fieldName)) {
      updateResponsibleStatus();
    } else if (DESCRIPTION.equals(fieldName)) {
      updateDescrStatus();
    } else if (DESCRIPTOR_FILE.equals(fieldName)) {
      updateDescrFileStatus();
    } else if (CONTAINER.equals(fieldName)) {
      updateProjectStatus();
    }
    doStatusUpdate();
  }

  // -------- Initialization ---------
  private void updateProjectStatus() {
    try {
  		String str= getPackageFragmentRootText();
  		if (str.length() == 0) {
  			return;
  		}
  		IPath path= new Path(str);
  		IResource res= ResourcesPlugin.getWorkspace().getRoot().findMember(path);
  		if (res == null) {
  		  return;
  		}
      selectedProject = res.getProject();
      if (selectedProject == null
          || !selectedProject.hasNature(PDE.PLUGIN_NATURE)) {
        fProjectStatus = new StatusInfo(IStatus.ERROR,
            "selected project must be a plugin project");
      } else {
        List missingReuiredPluginsIds = getMissingImports();
        if (missingReuiredPluginsIds.size() == 0) {
          fProjectStatus = new StatusInfo();
        } else {
          StringBuffer buf = new StringBuffer();
          for (Iterator iter = missingReuiredPluginsIds.iterator(); iter
              .hasNext();) {
            buf.append((String) iter.next());
            if (iter.hasNext()) {
              buf.append(", ");
            }
          }
          fProjectStatus = new StatusInfo(IStatus.WARNING,
              "missing required plugin dependencies: " + buf.toString());
        }
      }
    } catch (JavaModelException e) {
      ExceptionHandler.handle(e);
    } catch (CoreException e) {
      ExceptionHandler.handle(e);
    }
  }

  /**
   * The wizard owning this page is responsible for calling this method with
   * the current selection. The selection is used to initialize the fields of
   * the wizard page.
   * 
   * @param selection
   *          used to initialize the fields
   */
  public void init(IStructuredSelection selection) {
    IJavaElement jelem = getInitialJavaElement(selection);
    try {
      if (jelem != null) {
        selectedProject = (IProject) jelem.getJavaProject()
            .getCorrespondingResource();
      }
    } catch (JavaModelException e) {
      ExceptionHandler.handle(e);
    }
    initContainerPage(jelem);
    initTypePage(jelem);
  }

  private void setButtonGridData(Button button) {
    GridData gridData = new GridData();
    button.setLayoutData(gridData);
    SWTUtil.setButtonDimensionHint(button);
  }

  /*
   * @see WizardPage#becomesVisible
   */
  public void setVisible(boolean visible) {
    if (visible) {
      setFocus();
    }
    super.setVisible(visible);
  }

  private void updateDescrFileStatus() {
    String fileName = fDescriptorFileDialogField.getText().trim();
    if (!fileName.endsWith(".xml")) { //$NON-NLS-1$
      fDescriptorFileStatus = new StatusInfo(IStatus.ERROR,
          TextKeyDevelWizards.ENTER_XML_FILE.getText());
    } else {
      fDescriptorFileStatus = new StatusInfo();
    }
  }

  /**
   * Method updateDescrStatus.
   */
  private void updateDescrStatus() {
    if (fTestDescriptionDialogField.getText().trim().equals("")) { //$NON-NLS-1$
      fDescriptionStatus = new StatusInfo(IStatus.WARNING,
          TextKeyDevelWizards.DESCR_EMPTY_MSG.getText());
    } else {
      fDescriptionStatus = new StatusInfo();
    }
  }

  /**
   * Method updateDisplayNameStatus.
   */
  private void updateDisplayNameStatus() {
    if (fTestNameDialogField.getText().trim().equals("")) {
      fTestNameStatus = new StatusInfo(IStatus.ERROR,
          TextKeyDevelWizards.NAME_EMPTY_MSG.getText());
    } else {
      fTestNameStatus = new StatusInfo();
    }
  }

  private List getMissingImports() {
    ModelEntry entry = PDECore.getDefault().getModelManager().findEntry(
        selectedProject);
    IPluginImport[] imports = entry.getActiveModel().getPluginBase()
        .getImports();
    Set importSet = new HashSet();
    for (int i = 0; i < imports.length; i++) {
      importSet.add(imports[i].getId());
    }
    List requiredImportsList = new ArrayList();
    requiredImportsList.add(JlinPlugin.getPluginId());
    String[] reqPluginIds = getTestGenerator().getRequiredPluginIds();
    for (int i = 0; i < reqPluginIds.length; i++) {
      requiredImportsList.add(reqPluginIds[i]);
    }
    requiredImportsList.removeAll(importSet);
    return requiredImportsList;
  }

  /**
   * Method updateResponsibleStatus.
   */
  private void updateResponsibleStatus() {
    if (fResponsibleDialogField.getText().trim().equals("")) { //$NON-NLS-1$
      fResponsibleStatus = new StatusInfo(IStatus.WARNING,
          TextKeyDevelWizards.RESP_EMPTY_MSG.getText());
    } else {
      fResponsibleStatus = new StatusInfo();
    }
  }

  public String getTestType() {
    return fTestTypeDialogField != null
        ? fTestTypeDialogField.getText()
        : TEST_TYPES[0];
  }
}