/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_devel/src/com/sap/tc/jtools/jlint/eclipse/devel/wizards/InputParameterDialog.java#2 $
 */

package com.sap.tc.jtools.jlint.eclipse.devel.wizards;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.ascii.StringBuf;

/**
 * Dialog for getting a parameter (name and type) from the user.
 */

public class InputParameterDialog extends Dialog {

  // child widgets
  private Button fOkButton;

  private Text fNameText;
  private Combo fTypeCombo;

  private Label fMessageLabel;

  private ParameterInterface fResult;
  private ParameterInterface[] fExistingParams;
  
  // mode: "edit" or "add"
  private boolean editMode;
  /**
   * Creates an wildcard dialog with OK and Cancel buttons.
   */
  public InputParameterDialog(
    Shell pParentShell,
    ParameterInterface initialValues,
    ParameterInterface[] existingParameters) {
    super(pParentShell);
    editMode = true;
    fResult = initialValues;
    fExistingParams = existingParameters;
  }

  public InputParameterDialog(
    Shell pParentShell,
    ParameterInterface[] existingParameters) {
    super(pParentShell);
    editMode = false;
    fExistingParams = existingParameters;
  }

  /* (non-Javadoc)
   * Method declared on Dialog.
   */
  protected void buttonPressed(int buttonId) {
    if (buttonId == IDialogConstants.OK_ID) {
      String type = fTypeCombo.getItems()[fTypeCombo.getSelectionIndex()];
      String name = fNameText.getText().trim();
      fResult = ParameterTool.createParameter(name, type, null);
    } else {
      fResult = null;
    }
    super.buttonPressed(buttonId);
  }

  /* (non-Javadoc)
   * Method declared in Window.
   */
  protected void configureShell(Shell shell) {
    super.configureShell(shell);
    shell.setText(TextKeyDevelWizards.PARAM_DLG_TITLE.getText());
  }

  /* (non-Javadoc)
   * Method declared on Dialog.
   */
  protected void createButtonsForButtonBar(Composite parent) {
    // create OK and Cancel buttons by default
    fOkButton =
      createButton(
        parent,
        IDialogConstants.OK_ID,
        IDialogConstants.OK_LABEL,
        true);
    fOkButton.setEnabled(false);

    // cancel button needs not to be observed    
    createButton(
      parent,
      IDialogConstants.CANCEL_ID,
      IDialogConstants.CANCEL_LABEL,
      false);

    //do this here because setting the text will set enablement on the ok button
    fNameText.setFocus();
    validateInput();
  }

  /* (non-Javadoc)
   * Method declared on Dialog.
   */
  protected Control createDialogArea(Composite parent) {
    // create composite
    Composite composite = (Composite) super.createDialogArea(parent);

    fNameText = new Text(composite, SWT.SINGLE | SWT.BORDER);
    fNameText.setLayoutData(
      new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
    fNameText.addModifyListener(new ModifyListener() {
      public void modifyText(ModifyEvent e) {
        validateInput();
      }
    });

    fTypeCombo = new Combo(composite, SWT.DROP_DOWN);
    fTypeCombo.setItems(ParameterTool.getParameterTypes());
    fTypeCombo.setLayoutData(
      new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
    fTypeCombo.select(0);
    if (fResult != null) {
      // set initial values
      String type = fResult.getType();
      String[] items = fTypeCombo.getItems();
      for (int i = 0; i < items.length; i++) {
        if (items[i].equals(type)) {
          fTypeCombo.select(i);
        }
      }
      fNameText.setText(fResult.getName());
    }

    fMessageLabel = new Label(composite, SWT.NONE);
    fMessageLabel.setLayoutData(
      new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
    fMessageLabel.setFont(parent.getFont());
    validateInput();
    return composite;
  }

  /**
   * Returns a ParameterInterface or null
   */
  protected ParameterInterface getResult() {
    return fResult;
  }

  /**
   * Validates the input.
   */
  protected void validateInput() {
    String paramName = fNameText.getText().trim();
    boolean exists = parameterNameExists(paramName);
    if (exists) {
      fMessageLabel.setText(TextKeyDevelWizards.PARAM_EXISTS_MSG.getText());
    }
    boolean nameOk = parameterNameOk(paramName);
    if (!nameOk) {
      fMessageLabel.setText(TextKeyDevelWizards.PARAM_NAME_NOT_OK_MSG.getText());
    }
    boolean empty = paramName.length() < 1;
    if (empty) {
      fMessageLabel.setText(TextKeyDevelWizards.PARAM_EMPTY_MSG.getText());
    }
    boolean okay = !empty && !exists && nameOk;
    // may be called before initializing GUI!
    if (fOkButton != null && fMessageLabel != null) {
      fOkButton.setEnabled(okay);
      fMessageLabel.setVisible(!okay);
    }
  }

  /**
   * Method parameterNameOk.
   * @param paramName
   * @return boolean
   */
  private boolean parameterNameOk(String paramName) {
    char[] name = paramName.toLowerCase().toCharArray();
    StringBuf buf = new StringBuf("abcdefghijklmnopqrstuvwxyz0123456789");
    for (int i = 0; i < name.length; i++) {
      if (buf.indexOf(name[i]) == -1) {
        return false;
      }
    }
    return true;
  }


  private boolean parameterNameExists(String paramName) {
    if (editMode) {
      // in edit mode, the parameter name edited is omitted 
      // from existing params list
      if (paramName.equals(fResult.getName())) {
        return false;
      }
    }
    for (int i = 0; i < fExistingParams.length; i++) {
      if (fExistingParams[i].getName().equals(paramName)) {
        return true;
      }
    }
    return false;
  }
}
