package com.sap.caf.km.ejb.svc.idxsearch;

import java.util.Collection;
import java.util.Map;

import com.sap.caf.km.ejb.svc.idxsearch.common.SearchException;

public interface IIndexSearchSearch {

	/** Search over global indexes */

	/**
	 * Searches for global documents with free text in content or attributes
	 * 
	 * @param queryEntryCollection	Collection of <code>IKMSearchEntry</code> instances 
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForGlobalDoc(Collection queryEntryCollection) throws SearchException;

	/**
	 * Searches for global documents with free text in content or attributes starting from specified folder
	 * 
	 * @param queryEntryCollection			The collection of <code>IKMSearchEntry</code> instances	
	 * @param startFolderRIDForSearch 	The RID as string of a folder to start the search in
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForGlobalDoc(Collection queryEntryCollection, String startFolderRIDForSearch)
		throws SearchException;

	/**
	 * Searches for global tasks with free text in content or attributes by criteria defined in 
	 * <code>queryEntryCollection</code>
	 * 
	 * @param queryEntryCollection 	The collection of <code>IKMSearchEntry</code>
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForGlobalTasks(Collection queryEntryCollection) throws SearchException;

	/**
	 * Searches for global tasks with free text in content or attributes inside specified folder 
	 * by specified query 
	 * 
	 * @param queryEntryCollection	  The collection of <code>IKMSearchEntry</code> instances
	 * @param startFolderRIDForSearch RID as string of a folder to start the search in
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForGlobalTasks(Collection queryEntryCollection, String startFolderRIDForSearch)
		throws SearchException;

	/** Search over CAF-BO indexes */

	/**
	 * Searches for CAF BO with free text in attributes by criteria defined in 
	 * <code>queryEntryCollection</code> and indexes related to <code>BONames</code>
	 *  
	 * @param queryEntryCollection	The collection of <code>IKMSearchEntry</code> instances
	 * @param BONames		
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForCAFBO(Collection queryEntryCollection, Collection BONames) throws SearchException;

	/**
	 * Search for related documents with free text in attribute and/or content by criteria 
	 * <code>queryEntryCollection</code>. Indexes to search in are defined by <code>BONames</code>
	 * collection
	 *  
	 * @param queryEntryCollection The collection of <code>IKMSearchEntry</code> instances
	 * @param BONames The collection of strings representing BONames to get releated indexes to search in 
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForRelatedDocument(Collection queryEntryCollection, Collection BONames) throws SearchException;

	/**
	 * Performs joined search for CAF BO with free text in attributes AND/OR in related documents content 
	 *  
	 * @param queryEntryCollectionCAF	The collection of serach criteria objects <code>IKMSearchEntry</code> for CAF BO
	 * @param queryEntryCollectionKM	The collection of serach criteria objects <code>IKMSearchEntry</code> for KM BO
	 * @param BONames	The collection of BONames 
	 * @param searchKMType	SEARCH_TASK = 1; SEARCH_DOC  = 2; SEARCH_TASKDOC = 3;
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForCAFBOAndRelatedRes(
		Collection queryEntryCollectionCAF,
		Collection queryEntryCollectionKM,
		Collection BONames,
		int searchKMType)
		throws SearchException;

	/**
	 * Searches for related tasks with free text in attributes and/or content
	 *  
	 * @param queryEntryCollection The collection of <code>IKMSearchEntry</code> instances
	 * @param BONames The collection of BO names to find corresponding index
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForRelatedTask(Collection queryEntryCollection, Collection BONames) throws SearchException;

	/**
	 * Searches for all CAF BO´s with free text in attributes
	 *  
	 * @param queryEntryCollection The collection of <code>IKMSearchEntry</code> instances
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForAllCAFBO(Collection queryEntryCollection) throws SearchException;

	/**
	 * Searches for all KM BO´s with free text in attributes and / or in content
	 *  
	 * @param queryEntryCollection The collection of <code>IKMSearchEntry</code> instances
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForAllKMBO(Collection queryEntryCollection) throws SearchException;

	/**
	 * Searches for all BO´s wether they are stored in KM or on CAF side  with free text in attributes and / or in content
	 *  
	 * @param queryEntryCollection	The collection of <code>IKMSearchEntry</code> instances
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForAllBO(Collection queryEntryCollection) throws SearchException;

	/**
	 * Searches for similar CAF resources in KM
	 *  
	 * @param resourceRid The source resource object in the KM used to perform the similar search
	 * @param maxResults
	 * @return
	 * @throws SearchException
	 */
	public Collection searchForSimilarCAF(String resourceRid, int maxResults) throws SearchException;

	/**
	 * Searches for similar KM resources in KM
	 *  
	 * @param resourceRid		The source resource object in the KM used to perform the similar search
	 * @param useGlobalIndex	<code>true</code> if global indexes should be used
	 * @throws SearchException
	 */
	public Collection searchForSimilarKM(String resourceRid, boolean useGlobalIndex) throws SearchException;

	public Collection searchForBO(String attribute, String freeText, String BOName, boolean isAttribute) throws SearchException;

	public Collection searchForBO(Collection collAttributesAsString, Collection collCategoriesAsString, String freeText, String BOName) throws SearchException;
	
	public Collection searchForBO(Map hashMapofAttributesAsStringAndfreeText, Map hashMapofCategoriesAsStringAndfreeText, String BOName) throws SearchException; 

	public Collection searchForRelatedDoc(Collection collAttributesAsString, String freeText, String BOName) throws SearchException;
	
	public Collection searchForBOInRelatedDoc(Collection collAttributesAsString, String freeText, String BOName) throws SearchException;

	public Collection searchForBO(Collection filter, String BOName) throws SearchException;
	
	//TODO: Remove the next three methods if business object template has been updated to the new search method signatures
	public Collection searchForBO(String attribute, String freeText, String BOName) throws SearchException;
	public Collection searchForBO(Collection collAttributesAsString, String freeText, String BOName) throws SearchException;	
	public Collection searchForBO(Map hashMapofAttributesAsStringAndfreeText, String BOName) throws SearchException; 

}
