package com.sap.caf.km.ejb.svc.idxsearch;

import java.util.Collection;

import com.sap.caf.km.ejb.svc.idxsearch.common.ClassifyException;

/**
 *	Business object classification interface 
 */
public interface IIndexSearchClassify {

	/**
	 * Assigns category to business object
	 * @param catGUID	The category GUID
	 * @param BOName	The business object name
	 * @throws KMClassifyException
	 */
  public void assignCategory(String catGUID, String BOName) throws ClassifyException;

  /**
   *	Assigns a collection of categories to business object 
   * @param catGUIDs	A collection of categories GUIDs
   * @param BOName	The business object name
   * @throws KMClassifyException
   */
  public void assignCategory(Collection catGUIDs, String BOName) throws ClassifyException;
  
  /**
	 * Deassigns category from business object
	 * @param catGUID	The category GUID
	 * @param BOName	The business object name
   * @throws KMClassifyException
   */
  public void deassignCategory(String catGUID, String BOName) throws ClassifyException;
  
  /**
	 * Deassigns category from all business objects
	 * @param catGUID	The category GUID
	 * @param BOName	The business object name
   * @throws KMClassifyException
   */
  public void deassignCategory(String catGUID) throws ClassifyException;
  
  /**
	 * Deassigns a collection category from business object
	 * @param catGUIDs	The collection of categories GUIDs
	 * @param BOName	The business object name
   * @throws KMClassifyException
   */
  public void deassignCategory(Collection catGUIDs, String BOName) throws ClassifyException;
  
  /**
   * Gets collection of categories assigned to a business object  
   * @param BOName	The business object name
   * @return	A collection of assigned categories as instances of <code>Category</code> interface
   * @throws KMClassifyException
   */
  public Collection getCategories(String BOName) throws ClassifyException;

	/**
	 * Classifies an instance of business object. Links an instance of  
	 * <code>ValuesSet</code> object with an instance of business object by their GUIDs
	 * @see BOCategory 
	 * @param valueGUID	The <code>ValueSet</code> instance GUID
	 * @param BOGUID	The business object instance GUID
	 * @throws KMClassifyException
	 */
  public void addClassification(String valueGUID, String BOGUID,  String categoryGUID) throws ClassifyException;
  
  /**
	 * Classifies an instance of business object. Links a collection of <code>ValuesSet</code> instances  
	 * with an instance of business object by their GUIDs
   * @param valueGUIDs The collection of <code>ValueSet</code> instances GUIDs
   * @param BOGUID	The business object instance GUID
   * @throws KMClassifyException
   */
  public void addClassification(Collection valueGUIDs, String BOGUID, String categoryGUID) throws ClassifyException;

  /**
	 * Declassifies an instance of business object. Unlinks a <code>ValuesSet</code> instance  
	 * from an instance of business object by their GUIDs
   * @param valueGUIDs The <code>ValueSet</code> instance GUID
   * @param BOGUID	The business object instance GUID
   * @throws KMClassifyException
   */
  public void removeClassification(String valueGUID, String BOGUID) throws ClassifyException;

  /**
	 * Declassifies an instance of business object. Unlinks a collection of <code>ValuesSet</code> instances  
	 * from an instance of business object by their GUIDs
   * @param valueGUIDs The collection of <code>ValueSet</code> instances GUIDs
   * @param BOGUID	The business object instance GUID
   * @throws KMClassifyException
   */
  public void removeClassification(Collection valueGUID, String BOGUID) throws ClassifyException;

  /**
   * Tests if valueGUID is used in classification, i.e. there is an BO linked with 
   * the <code>valueGUID</code>
   * @param valueGUID	A GUID of the <code>ValueSet</code> instance
   * @throws KMClassifyException
   */
  public boolean isUsedInClassification(String valueGUID) throws ClassifyException;
  
  /**
   * Removes all links between the <code>valueGUID</code> and any BO instance
   * @param valueGUID	A GUID of the <code>ValueSet</code> instance
   * @throws KMClassifyException
   */
  public void removeClassification(String valueGUID) throws ClassifyException;
  
  /**
   * Removes all links between each value GUID from a collection <code>valueGUIDs</code> 
   * and any BO instance
   * @param valueGUIDs	A collection of values GUIDs represented as <code>String</code>
   * @throws KMClassifyException
   */
  public void removeClassification(Collection valueGUIDs) throws ClassifyException;
  
  /**
   * Gets a collection of category values linked with business object  
   * @param BOGUID	The business object GUID
   * @return	The collection of <code>CategoryValueSetVO</code> instances
   * @throws KMClassifyException
   */
  public Collection getClassification(String BOGUID) throws ClassifyException;

  /**
   * Gets a collection of category values linked with business object  
   * @param BOGUID	The business object GUID
   * @return	The collection of <code>CategoryValueSetVO</code> instances
   * @throws KMClassifyException
   */
  public Collection getClassification(String BOGUID, String categoryGUID) throws ClassifyException;

  /**
	 * Gets a collection of category values linked with corresponding boName business objects  
	 * @param boNames	The full BO name
	 * @return	The collection of arrays of String (String[]{boGuid, catGuid, catValGuid})
	 * @throws KMClassifyException
	 */
  public Collection getAssignedClassification(String BOName) throws ClassifyException;
	
	/**
	 * Gets prefix of KM taxonomies repository
	 * @return	taxonomies repository prefix 
	 * @throws KMClassifyException
	 */
  
  public String getTaxonomyRepositoryPrefix() throws ClassifyException;
}


