/*
 * SAP Copyright (c) 2003
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_pdfobject/java/com/sap/tc/webdynpro/pdfobject/api/IWDPDFObject.java#6 $
 */
package com.sap.tc.webdynpro.pdfobject.api;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Locale;

/**
 * Interface IWDPDFObject describes the public interface of the PDFObject API.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDPDFObject {
  
  
  /**
   * Method getData returns the data from a PDF File
   * Preconditions: method setPDF must have been called
   * @return XFD
   */
  public InputStream getData();
  
  /**
   * Method getData returns the data from a PDF File. The result is written to the given
   * url. Method returns <code>true</code> when call was successful.
   * Preconditions: method setPDF must have been called
   * @param url destination where data is saved respectively to which data is posted 
   * @return 
   */
  public boolean getData(String url);
  
  /**
   * Method createPDF creates a PDFDocument and the result is written to the 
   * given url. Method returns <code>true</code> when call was successful.
   * Preconditions: setData and setTemplate must have been called
   * @param url
   * @return 
   */
  public boolean createPDF(String url);
  
  /**
   * Method createPDF creates a PDFDocument and the result is returned as InputStream
   * Preconditions: setData and setTemplate must have been called
   * @return
   */
  public InputStream createPDF();
  
  /**
	 * Method setData sets a URL pointing to the document containing the form data(XFD) that
   * is used for creating the PDF
	 * @param url
	 */
	public void setData(String url);
  
  /**
	 * Method setData sets an OutputStream that contains data that is used for
   * creating the PDF. 
	 * @param xfd. If xfd is an ByteArrayOutputStream it must be UTF-8 encoded!
	 */
	public void setData(OutputStream xfd);
  
  /**
	 * Method setPDF sets the PDF document that should be examined i.e. from that 
   * data should be extracted, digital signatures should be verified
	 * @param url
	 */
	public void setPDF(String url); 

  /**
   * Method setPDF sets the PDF document that should be examined i.e. from that 
   * data should be extracted, digital signatures should be verified
   * @param url
   */  
  public void setPDF(ByteArrayOutputStream pdf);
  
  /**
   * Method setLocale can be called to set the locale for the PDF document. Required
   * for decimal digits when the localized version of the template is not available 
   * @param locale
   */
  public void setLocale(Locale locale);
  
  public void setSecurity(List certificates, boolean printable, boolean fillable, boolean changeable, boolean extractable);
  
  /**
   * Method setTemplate sets a URL that points to the Template used for
   * creating the PDF.    
   * @param url
   */
  public void setTemplate(String url); 
  
  /**
   * Method setTemplate sets an OutputStream that contains the Template used for
   * creating the PDF.
   * @param template. If template is an ByteArrayOutputStream it must be UTF-8 encoded!
   */
  public void setTemplate(OutputStream template);
  
  /**
   * Method setInteractive creates an interactive form
   * @param b
   */
  public void setInteractive(boolean b);
  
  /**
   * Method setSecureCommunication. Default is not secure. Method will be removed
   * and the security level will be defined by the concrete call requirements
   * @param b
   * @deprecated This method may be withdrawn with the first new NetWeaver release in 2006.
   */
  public void setSecureCommunication(boolean b);
  
  /**
   * Method setBaseURI.
   * @param url
   */
  public void setBaseURI(String url);
  
  /**
   * Method setXsltURI can be used to set a XSLT that is applied to the data
   * before it is returned.
   * 
   * @param url
   */
  public void setXsltURI(String url);
  
  
  /**
   * Method getColumnData.
   * @param url
   * @return
   */
  public boolean getColumnData(String url);
  
  /**
   * Method getColumnData.
   * @return
   */
  public InputStream getColumnData();
  
  /**
   * Method getResultString returns the return code of the last call
   * @return
   */
  public String getResultString();
      
  
  /**
   * Method setUsageRights allows to set usage rights. UsageRight are predefined
   * e.g. "WDFormRights.ADD". 
   * @param credential if "null" default "ReaderRights" is used
   * @param rights
   */
  public void setUsageRights(String credential, WDUsageRight[] rights);
  
  /**
   * Method setUserName sets the user name that is used for applying rights
   * or digital signatures.
   * @param userName
   */
  public void setUserName(String userName);
  
  //public void setCertification(String key, String fieldName, String legatAttestation);
  
  
//  EXECUTE                 Instan   Aufruf ADS                             
                      
//  GET_PDF                 Instan   Get PDF or PDL                         
//  GET_PDL                 Instan   PDL holen                              
//  SET_DATA                Instan   Determine Data                         
//  SET_DOCUMENT            Instan   PDF Dokument festlegen                 
//  SET_OUTPUTPDF           Instan   PDF Ausgabe auswählen                  
//  SET_OUTPUTPDL           Instan   PDL Ausgabe auswählen                  
//  SET_TEMPLATE            Instan   Determine Template                     
//  SET_EXTRACTDATA         Instan   Datenextraktion auswählen              
//  SET_PASSWORD            Instan   Set password                           
//  SET_MASTER_PASSWORD     Instan   Master Paßwort setzen                  
//  GET_ENCRYPTION          Instan   Verschlüsselung lesen                  
//  GET_PRINTABLE           Instan   Berechtigung "Drucken" lesen           
//  GET_CHANGEABLE          Instan   Berechtigung "Ändern" lesen            
//  GET_EXTRACTABLE         Instan   Berechtigung "Datenextraktion" lesen   
//  GET_FILLABLE            Instan   Berechtigung "Ausfüllen" lesen         

	/**
	 * Get the application name within which the PDFObject is runnning.
	 */
	public String getApplicationName();

	/**
	 * Set the application name within which the PDFObject is runnning.
	 */
	public void setApplicationName(String applicationName);
 
	/**
	 * Get the Client ID who wants the PDFObject service.
	 */
	public String getClientID();

	/**
	 * Set the Client ID who wants the PDFObject service.
	 */
	public void setClientID(String clientID);

	/**
	 * Get the PDF form name for which the PDFObject is being processed.
	 */
	public String getFormName();

	/**
	 * Set the PDF form name for which the PDFObject is being processed.
	 */
	public void setFormName(String formName);  
  
  /**
   * Method getReviewCopy gets the Review Copy from the ADS in a read only mode.
   * Result will be written to the url specified.
   * Preconditions : setPDF/setTemplate and setData should be called before 
   * calling this method.
   * 
   * @param   url : String
   * @return  boolean
   */
  public boolean getReviewCopy(String url);
  
  /**
   * Method getReviewCopy gets the Review Copy from the ADS in a read only mode.
   * Result is returned as InputStream.
   * Preconditions : setPDF/setTemplate and setData should be called before 
   * calling this method.
   * 
   * @return InputStream : String 
   */
  public InputStream getReviewCopy();
  
  /**
   * Method certify puts a request to apply certificate on the pdf document 
   * under consideration.
   * 
   * @param  credential : Optional. Name of the private key. Default is 
   * "DocumentCertification".
   *                
   * @param  field name : Name of the filed to which certification will 
   * be applied. If this is not set then the certification will be added 
   * to the document itself, but will not be shown on the document itself. 
   * The user will see it as certified in a corner of the Acrobat UI.
   * 
   * @param  reason : Optional. If not set the default <b>I attest to the 
   * accuracy and integrity of this document</b> will be set.
   * 
   * @param  location : Optional. If not set the default will be empty.
   * @param  contact info : Optional. If not set the default will be empty.
   * @param  legal attestations : Optional.
   * @param  permission mode : Optional. Mode of permission,allowing the user 
   * to change such as formFields or Comments, that won't break the certificate 
   * attached with the document. Currently available permission modes are :
   *  <li>None</li>
   *  <li>FormFields</li>
   *  <li>FormFieldsAndComments</li>
   * Default permission mode is <b>FormFields</b>
   *   
   * @return nil
   */  
   public void certify(String credential,
                      String fieldName,
                      String reason,
                      String location,
                      String contactInfo,
                      String legalAttestations,
                      WDInteractiveFormPermissionModes permissionMode);
  
  /**
   * Method getCertification returns the status of the certificate for the pdf
   * document under consideration. True if the certificate for the document is 
   * intact and false if it has been broken.
   * 
   * @return boolean
   */
  public boolean getCertification();
  
  /**
   * Method suppressPrint.
   * 
   * @param  boolean
   * @return nil  
   */
  public void suppressPrint(boolean suppress);
}
