/*
 * @(#) $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/Test.java#3 $
 *
 * Copyright(c) 2002, SAP AG, All Rights Reserved.
 * 
 * Last Modification $Author: perforce $ $Date: 2004/04/29 $
 */

package com.sap.tc.jtools.jtci;

import java.util.HashMap;
import java.util.Map;

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ITestMessageDescription;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Instances of this class represent the Name of an Test as well as its 
 * input parameter and their settings.
 * @version   $Date: 2004/04/29 $
 * @author    BPL Tools
 */

public class Test {

	private final String name;
	private ParameterInterface[] parameters;
	private ITestMessageDescription[] messages;
	private boolean active = true;

	private String[] codeClassificationStrings = ALL;

	public static final String APPLIES_TO_TAG = "APPLIES_TO"; //$NON-NLS-1$
	public static final String NAME_TAG = "NAME"; //$NON-NLS-1$
	public static final String ACTIVE_TAG = "ACTIVE"; //$NON-NLS-1$

	private static final String[] ALL =
		new String[] { CodeClassificationConstants.ALL_CODE_STR };

	public Test(String testName, ParameterInterface[] parameters) {
		this(testName, parameters, new ITestMessageDescription[0], ALL,true);
	}
	
	public Test(String testName, ParameterInterface[] parameters, boolean active) {
		this(testName, parameters, new ITestMessageDescription[0], ALL,active);
	}

	public Test(
		String testName,
		ParameterInterface[] parameters,
		ITestMessageDescription[] messages,
		String[] appliesTo,
		boolean active) {
		name = testName;
		this.parameters = parameters;
		this.messages = messages;
		this.active = active;
		this.codeClassificationStrings = appliesTo;
	}

	public Test(StructureTree tree) throws BadTreeException {
		name = tree.getParameter(NAME_TAG);
		// backwards compatibility: tests need not have an "active" attribute,
		// default is true
		String active = tree.getParameter(ACTIVE_TAG);
		if (active != null) {
			this.active = Boolean.valueOf(active).booleanValue();
		} else {
			this.active = true;
		}

		StructureTree[] params =
			tree.getChildren(ParameterInterface.INPUT_PARAMETER_TAG);
		parameters = new ParameterInterface[params.length];
		for (int i = 0; i < params.length; i++) {
			parameters[i] = ParameterTool.createParameter(params[i]);
		}
		StructureTree msgsTree = tree.getOnlyChild(ITestMessageDescription.TAG_MESSAGES);
		if (msgsTree != null) {
		  StructureTree[] msgTrees = msgsTree.getChildren(ITestMessageDescription.TAG_MESSAGE);
		  this.messages = new ITestMessageDescription[msgTrees.length];
		  for (int i = 0; i < msgTrees.length; i++) {
        this.messages[i] = TestMessageDescription.fromStructureTree(msgTrees[i]);
      }
		} else {
		  messages = new ITestMessageDescription[0];
		}

		StructureTree[] appliesToTrees = tree.getChildren(APPLIES_TO_TAG);
		if (appliesToTrees.length > 0) {
			codeClassificationStrings = new String[appliesToTrees.length];
			for (int i = 0; i < appliesToTrees.length; i++) {
				codeClassificationStrings[i] = appliesToTrees[i].getText();
			}
		}

	}

	public String[] getCodeClassification() {
		return codeClassificationStrings;
	}

	public String getName() {
		return name;
	}

	public ParameterInterface[] getParameters() {
		return parameters;
	}

	
	public void setMessages(ITestMessageDescription[] messages)  {
	  this.messages = messages;
	}
	
	public ITestMessageDescription[] getMessages() {
	  return messages;
	}
	

	public ParameterInterface getParameter(String name) {
		if (name == null)
			return null;
		for (int i = 0; i < parameters.length; i++) {
			if (name.equals(parameters[i].getName()))
				return parameters[i];
		}
		return null;
	}

	public void setParameters(ParameterInterface[] newParameters) {
		this.parameters = newParameters;
	}

	public boolean paramsOkay() {
		if (null == parameters)
			return true;
		boolean okay = true;
		for (int i = 0; okay && i < parameters.length; i++) {
			Object value = parameters[i].getValue();
			if (value instanceof String) {
				okay = 0 != ((String) value).length();
			} else {
				okay = null != value;
			}
		}
		return okay;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
	}

	public StructureTree toStructureTree() {
		Header header = new Header(TestDescriptionInterface.PARAMETER_TAG_TEST);
		header.setParameter(NAME_TAG, name);
		header.setParameter(ACTIVE_TAG, String.valueOf(active));
		StructureTree rootTree = new StructureTree(header);

		for (int i = 0; i < parameters.length; i++) {
			rootTree.addNewChild(parameters[i].toStructureTree());
		}

		if (messages != null && messages.length > 0) {
		  StructureTree msgsTree = rootTree.addNewChild(new Header(ITestMessageDescription.TAG_MESSAGES));
		  for (int i = 0; i < messages.length; i++) {
        msgsTree.addNewChild(messages[i].toStructureTree());
      }
		}
		for (int i = 0; i < codeClassificationStrings.length; i++) {
			StructureTree newTree =
				new StructureTree(new Header(APPLIES_TO_TAG));
			newTree.setText(codeClassificationStrings[i]);
			rootTree.addNewChild(newTree);
		}
		return rootTree;
	}

  /* merge messages from test definition and request (request overrides testdefinition) */
  public static ITestMessageDescription[] mergeMessages(TestDescriptionInterface testDefinition, Test requestTest) {
    ITestMessageDescription[] originalMsgs = testDefinition.getMessages();
    Map msgMap = new HashMap(originalMsgs.length);
    for (int i = 0; i < originalMsgs.length; i++) {
      msgMap.put(originalMsgs[i].getKey(), originalMsgs[i]);
    }
    // override with request messages
    ITestMessageDescription[] requestMsgs = requestTest.getMessages();
    for (int i = 0; i < requestMsgs.length; i++) {
      msgMap.put(requestMsgs[i].getKey(), requestMsgs[i]);
    }
    return (ITestMessageDescription[]) 
      msgMap.values().toArray(new ITestMessageDescription[0]);
  }


}