/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/StringParameter.java#2 $
 */

package com.sap.tc.jtools.jtci;

/**
 * Implementation of a container for string like parameters.
 * @version   $Date: 2004/03/22 $
 * @author    BPL Tools
 */

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.ascii.StringBuf;
import com.sap.tc.jtools.util.structures.*;

final public class StringParameter implements ParameterInterface {

 

  protected final String name;
  private String value;
  private String type = ParameterTool.PAR_TYPE_STRING; //default

  /** 
   * Constructor based on a String as value.
   */
  StringParameter(String name, String value) {
    this(name, value, null);
  }

  StringParameter(String name, String value, String type) {
    this.name = name;
    this.value = value;
    if (type != null
      && (type.equals(ParameterTool.PAR_TYPE_STRING)
        || type.equals(ParameterTool.PAR_TYPE_FILE)
        || type.equals(ParameterTool.PAR_TYPE_DIRECTORY))) {
      this.type = type;
    } else {
      throw new IllegalArgumentException("type " + type + " is invalid");
    }
  }

  /**
   * Returns the name of this Parameter.
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the Type name.
   */
  public String getType() {
    return type;
  }

  /**
   * Returns the value as Serializable.
   */
  public java.io.Serializable getValue() {
    return value;
  }

  /**
   * Returns the value in its own Type.
   */
  public String getValueTrue() {
    return value;
  }

  /**
   * Returns a String Representation of the value.
   */
  public String valueToString() {
    if (null == value)
      return ""; //$NON-NLS-1$
    return value;
  }

  /**
   * Returns a String Representation.
   */
  public String toString() {
    String valString = valueToString();
    StringBuf buf = new StringBuf(name.length() + valString.length() + 10);
    return buf.append(name).append(':').append(valString).toString();
  }

  /**
   * Allows to change the value of this argument.
   * @see ParameterInterface#getValue()
   **/
  public void setValue(java.io.Serializable newValue) {
    if (newValue != null) {
      value = newValue.toString();
    } else {
      value = null;
    }
  }

  /**
   * Returns a StructureTree representation of this instance.
   */
  public StructureTree toStructureTree() {
    Header paramHeader = new Header(INPUT_PARAMETER_TAG);
    paramHeader.setParameter(PARAMETER_NAME, name);
    paramHeader.setParameter(PARAMETER_TYPE, type);

    StructureTree paramTree = new StructureTree(paramHeader);
    if (null != value && 0 != value.length()) {
      Header contentHeader = new Header(PARAMETER_VALUE);
      StructureTree contentTree = new StructureTree(contentHeader);
      paramTree.addNewChild(contentTree);
      contentTree.setText(value);
    }
    return paramTree;
  }

  /**
   * Internal method to create an instance out of according StructureTree.
   */
  static StringParameter createFromStructureTree(
    String nameInfo,
    StructureTree treeInfo)
    throws BadTreeException {

    StructureTree contents[] =
      treeInfo.getChildren(PARAMETER_VALUE);
    if (null == contents || 0 == contents.length) {
      return new StringParameter(
        nameInfo,
        null,
        treeInfo.getParameter(PARAMETER_TYPE));
    }
    if (1 != contents.length) {
      throw new BadTreeException("illegal value content"); //$NON-NLS-1$
    }
    return new StringParameter(
      nameInfo,
      contents[0].getText(),
      treeInfo.getParameter(PARAMETER_TYPE));
  }

}