/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/StringArrayParameter.java#2 $
 */

package com.sap.tc.jtools.jtci;

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.ascii.StringBuf;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Implementation of a container for parameters which are a set of strings.
 * @version   $Date: 2004/03/22 $
 * @author    BPL Tools
 */

final public class StringArrayParameter implements ParameterInterface {

  private static final String PAR_TYPE_STRING_ARRAY_ITEM = "ITEM"; //$NON-NLS-1$ 

  protected final String name;
  private String[] value;
  private String type = ParameterTool.PAR_TYPE_STRING_ARRAY; //default

  /** 
   * Constructor based on a String as value.
   */
  StringArrayParameter(String name, String[] value) {
    this.name = name;
    this.value = value;
  }

  StringArrayParameter(String name, String[] value, String type) {
    this.name = name;
    this.value = value;
    if (type != null
      && (type.equals(ParameterTool.PAR_TYPE_STRING_ARRAY)
        || type.equals(ParameterTool.PAR_TYPE_FILE_ARRAY)
        || type.equals(ParameterTool.PAR_TYPE_DIRECTORY_ARRAY))) {
      this.type = type;
    } else {
      throw new IllegalArgumentException("type " + type + " is invalid");
    }
  }
  /**
   * Returns the name of this Parameter.
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the Type name.
   */
  public String getType() {
    return type;
  }

  /**
   * Returns the value as Serializable.
   */
  public java.io.Serializable getValue() {
    return value;
  }

  /**
   * Returns the value in its own Type.
   */
  public String[] getValueTrue() {
    return value;
  }

  /**
   * Returns a String Representation of the value.
   */
  public String valueToString() {

    StringBuf buf = new StringBuf(200).append('{');

    if (null != value) {
      if (0 != value.length) {
        buf.append(value[0]);
        for (int i = 1; i < value.length; i++) {
          buf.append(',');
          buf.append(value[i]);
        }
      }
    }
    return buf.append('}').toString();
  }

  /**
   * Returns a String Representation.
   */
  public String toString() {
    String valString = valueToString();
    StringBuf buf = new StringBuf(name.length() + valString.length() + 10);
    return buf.append(name).append(':').append(valString).toString();
  }

  /**
   * Allows to change the value of this argument.
   * @param newValue  must be an instanceof String[]
   * @see ParameterInterface#getValue()
   **/
  public void setValue(java.io.Serializable newValue) {
    if (newValue == null) {
      value = null;
    } else if (newValue instanceof String[]) {
      value = (String[]) newValue;
    } else {
      value = new String[] { newValue.toString()};
    }
  }

  /**
   * Returns a StructureTree representation of this instance.
   */
  public StructureTree toStructureTree() {
    Header paramHeader = new Header(INPUT_PARAMETER_TAG);
    paramHeader.setParameter(PARAMETER_NAME, name);
    paramHeader.setParameter(PARAMETER_TYPE, type);
    StructureTree paramTree = new StructureTree(paramHeader);

    if (null != value && 0 != value.length) {
      Header contentHeader = new Header(PARAMETER_VALUE);
      StructureTree contentTree = new StructureTree(contentHeader);
      paramTree.addNewChild(contentTree);

      for (int i = 0; i < value.length; i++) {
        if (null != value[i]) {
          StructureTree itemTree =
            new StructureTree(
              new Header(PAR_TYPE_STRING_ARRAY_ITEM));
          itemTree.setText(value[i]);
          contentTree.addChild(itemTree);
        }
      }
    }
    return paramTree;
  }

  /**
   * Internal method to create an instance out of according StructureTree.
   */
  static StringArrayParameter createFromStructureTree(
    String nameInfo,
    StructureTree treeInfo)
    throws BadTreeException {

    StructureTree contents[] =
      treeInfo.getChildren(PARAMETER_VALUE);
    if (null == contents || 0 == contents.length) {
      return new StringArrayParameter(
        nameInfo,
        null,
        treeInfo.getParameter(PARAMETER_TYPE));
    }
    if (1 != contents.length) {
      throw new BadTreeException("illegal value content"); //$NON-NLS-1$
    }

    StructureTree[] items =
      contents[0].getChildren(PAR_TYPE_STRING_ARRAY_ITEM);

    if (null != items && 0 != items.length) {
      String[] values = new String[items.length];
      for (int i = 0; i < items.length; i++) {
        values[i] = items[i].getText();
        if (null == values[i]) {
          throw new BadTreeException("Empty StringArray Element"); //$NON-NLS-1$
        }
      }
      return new StringArrayParameter(
        nameInfo,
        values,
        treeInfo.getParameter(PARAMETER_TYPE));
    } else {
      return new StringArrayParameter(
        nameInfo,
        null,
        treeInfo.getParameter(PARAMETER_TYPE));
    }
  }

}