/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/PerformResponse.java#3 $
 */

package com.sap.tc.jtools.jtci;

import com.sap.tc.jtools.util.structures.*;
import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.extension.TestProvider;
import com.sap.tc.jtools.jtci.exceptions.*;
import com.sap.tc.jtools.jtci.interfaces.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

/**
 * Perform response that corresponds to a PerformRequest object and contains a
 * set of results (objects that implement the interface ResultInterface).
 *
 * <p>
 * Copyright (c) 2001, SAP AG
 */

public class PerformResponse {
  protected String name;
  protected String tool;
  private List results = new ArrayList();
  private Properties properties;

  /**
   * @param name name of the corresponding request
   *
   */
  public PerformResponse(String name) {
    this.name = name;
  }

  /**
   * @param tree response in StructureTree format
   * @throws BadTreeException if the tree does not correspond to a valid request
   *
   */
  public PerformResponse(StructureTree tree) throws BadTreeException {

    name = tree.getParameter("NAME");
    properties = tree.getParameters();
    properties.remove("NAME");

    StructureTree[] sourceTrees = tree.getChildren("SOURCE");
    for (int i = 0; i < sourceTrees.length; i++) {
      ResultInterface[] tempResults = getResults(sourceTrees[i]);
      for (int ii = 0; ii < tempResults.length; ii++) {
        results.add(tempResults[ii]);
      }
    }

  }

  /**
   * Returns name.
   *
   */
  public String getName() {
    return name;
  }

  /**
   * Returns tool name.
   *
   */
  public String getTool() {
    return tool;
  }

  /**
   * Adds result to the internal list.
   * @param newResult result to be added.
   *
   */
  public void addResult(ResultInterface newResult) {
    results.add(newResult);
  }

  /**
   * Adds the result array result to the internal list.
   * @param newResults results to be added.
   *
   */
  public void addResults(ResultInterface[] newResults) {
    results.addAll(Arrays.asList(newResults));
  }

  /**
   * Returns the results (as an array).
   *
   */

  public ResultInterface[] getResults() {
    return (ResultInterface[]) results.toArray(
      new ResultInterface[results.size()]);
  }

  /**
   * Returns all the results for a given test object.
   * @param testObjectID test object ID
   */
  public ResultInterface[] getResults(String testObjectID) {
    Vector outVector = new Vector();

    Iterator iterator = results.iterator();
    while (iterator.hasNext()) {
      ResultInterface current = (ResultInterface) iterator.next();
      if (current.getTestObjectID().equals(testObjectID))
        outVector.add(current);
    }
    return (ResultInterface[]) outVector.toArray(
      new ResultInterface[outVector.size()]);
  }

  /**
   * Returns all the results for a given target and test.
   * @param testObjectID test object ID
   * @param testName test name
   */
  public ResultInterface[] getResults(String testObjectID, String testName) {
    Vector outVector = new Vector();

    Iterator iterator = results.iterator();
    while (iterator.hasNext()) {
      ResultInterface current = (ResultInterface) iterator.next();
      if (current.getTestObjectID().equals(testObjectID)
        && current.getTestName().equals(testName))
        outVector.add(current);
    }
    return (ResultInterface[]) outVector.toArray(
      new ResultInterface[outVector.size()]);

  }

  /**
   * Returns  the response in StructureTree format
   */
  public StructureTree toStructureTree() {

    Header rootHeader = new Header("RESPONSE");

    rootHeader.setParameter("NAME", name);
    if (properties != null) {
      for (Iterator iter = properties.keySet().iterator(); iter.hasNext();) {
        String key = (String) iter.next();
        rootHeader.setParameter(key, properties.getProperty(key));
      }
    }
    StructureTree rootTree = new StructureTree(rootHeader);

    if (results.size() == 0)
      return rootTree;

    Iterator iterator = results.iterator();

    while (iterator.hasNext()) {
      StructureTree resultTree =
        ((ResultInterface) iterator.next()).toStructureTree();
      rootTree.addNewChild(resultTree);
    }

    return rootTree;
  }

  private static ResultInterface[] getResults(StructureTree resultTree) {
    String testObjectType = resultTree.getParameter("TYPE");
    String testObjectID = resultTree.getParameter("ID");

    Vector resultVector = new Vector();

    StructureTree[] testTrees = resultTree.getChildren("TEST");
    // file-independent messages
    for (int i3 = 0; i3 < testTrees.length; i3++) {
      String testName = testTrees[i3].getParameter("NAME");
      StructureTree[] messageTrees = testTrees[i3].getChildren("MESSAGE");
      for (int i4 = 0; i4 < messageTrees.length; i4++) {
        int priority = Integer.parseInt(messageTrees[i4].getParameter("PR"));
        String errorMessage = messageTrees[i4].getText();
        Properties errorMessageParameters = messageTrees[i4].getParameters();

        TestObjectPosition position = null;
        StructureTree[] toPositionTrees = messageTrees[i4].getChildren();
        if (toPositionTrees.length == 1) {
          String type = toPositionTrees[0].getTag();
          try {
            if (type.equals("FILE")) {
              position = new FilePosition(toPositionTrees[0]);
            } else {
              position = new GenericPosition(toPositionTrees[0]);
            }
          } catch (Exception e) {
            position = null;
          }
        }

        resultVector.add(
          new Result(
            testName,
            testObjectType,
            testObjectID,
            position,
            priority,
            errorMessage,
            errorMessageParameters));
      }
    }
    return (ResultInterface[]) resultVector.toArray(
      new ResultInterface[resultVector.size()]);

  }
  /**
   * @return
   */
  public Properties getProperties() {
    return properties;
  }

  /**
   * @param properties
   */
  public void setProperties(Properties properties) {
    this.properties = properties;
  }

}