/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/PerformRequest.java#3 $
 */
 
package com.sap.tc.jtools.jtci;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.sap.tc.jtools.jlint.extension.TestProvider;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ITestMessageDescription;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Represents a job request. A PerformRequest instance consists of a set of
 * request parameters and a set of perform units (combinations of tests and targets).
 * If no perform units are specified, all tests and targets are combined into a
 * single perform unit. Request parameters can be defined, which are accessible to all
 * tests.
 *
 * <p>
 * Copyright (c) 2001, SAP AG
 */

public class PerformRequest {

	private final static String TASK_PERFORM_TEST = "TEST"; //$NON-NLS-1$
	private final static String TASK = "TASK"; //$NON-NLS-1$
	private final static String TASK_NAME = "NAME"; //$NON-NLS-1$
	private final static String TASK_PERFORM = "PERFORM"; //$NON-NLS-1$
	private final static String TASK_PERFORM_UNIT = "PERFORM_UNIT"; //$NON-NLS-1$
	private final static String TASK_PERFORM_SOURCE_SET = "SOURCE_SET"; //$NON-NLS-1$
	private final static String TASK_PERFORM_TEST_SET = "TEST_SET"; //$NON-NLS-1$

	protected String name;
	private String task;
	private PerformUnit[] performUnits = null;
	private ParameterInterface[] requestParameters = new ParameterInterface[0];
	private List tests = new ArrayList();
	private List testObjects = new ArrayList();

	/**
	 * @param name request name
	 */

	public PerformRequest(String name) {
		this.name = name;
		this.task = TASK_PERFORM;
	}

	/**
	 * @param name request name
	 * @param tool request tool name(e.g. JLin)
	 * @deprecated use PerformRequest(String name) instead
	 */

	public PerformRequest(String name, String tool) {
		this(name);
	}

	/**
	 * 
	 * 
	 * @param tree request in StructureTree format
	 * @throws BadTreeException if the tree does not correspond to a valid request
	 * 
	 */
	public PerformRequest(StructureTree tree) throws BadTreeException {
		if (!tree.getParameter(TASK).equals(TASK_PERFORM)) {
			throw new BadTreeException();
		}
		name = tree.getParameter(TASK_NAME);

		this.task = TASK_PERFORM;
		StructureTree[] requestParameterTrees =
			tree.getChildren(ParameterInterface.INPUT_PARAMETER_TAG);
		requestParameters =
			new ParameterInterface[requestParameterTrees.length];
		for (int i = 0; i < requestParameters.length; i++) {
			requestParameters[i] =
				ParameterTool.createParameter(requestParameterTrees[i]);
		}
		StructureTree[] performUnitTrees = tree.getChildren(TASK_PERFORM_UNIT);
		performUnits = new PerformUnit[performUnitTrees.length];
		for (int i = 0; i < performUnits.length; i++) {
			StructureTree[] sourceSetTrees =
				performUnitTrees[i].getChildren(TASK_PERFORM_SOURCE_SET);
			TestObject[] sources = new TestObject[0];
			if (sourceSetTrees.length > 0) {
				StructureTree[] sourceTrees =
					sourceSetTrees[0].getChildren(TestObject.TEST_OBJECT);
				sources = new TestObject[sourceTrees.length];
				try {
					for (int k = 0; k < sources.length; k++) {
						sources[k] =
							TestProvider.getTestProvider().getTestObject(
								sourceTrees[k]);
					}
				} catch (IllegalStateException ise) {
					throw ise;
				} catch (Exception e) {
					throw new BadTreeException(e);
				}
			}
			StructureTree testSetTree =
				performUnitTrees[i].getOnlyChild(TASK_PERFORM_TEST_SET);
			Test[] tests = new TestSet(testSetTree).getActiveTests();
			performUnits[i] = new PerformUnit(sources, tests);
		}
	}

	/**
	 * Returns name.
	 *
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the perform units.
	 *
	 */
	public PerformUnit[] getPerformUnits() {
		if (performUnits == null) {
			computePerformUnits();
		}
		return performUnits;
	}

	/**
	 * 
	 */
	private void computePerformUnits() {
		if (performUnits != null)
			return; // nothing to do 

		//group tests according to code
		Map testMap = new HashMap(); // map code --> test list
		for (Iterator iterator = tests.iterator(); iterator.hasNext();) {
			Test currentTest = (Test) iterator.next();
			long code =
				CodeClassificationConstants.getValue(currentTest.getCodeClassification());
			List testList = (List) testMap.get(new Long(code));
			if (testList == null) {
				testList = new ArrayList();
				testMap.put(new Long(code), testList);
			}
			testList.add(currentTest);
		}

		//group test objects according to code
		Map testObjectMap = new HashMap(); // map code --> test list
		for (Iterator iterator = testObjects.iterator(); iterator.hasNext();) {
			TestObject currentTestObject = (TestObject) iterator.next();
			long code =
				CodeClassificationConstants.getValue(
					currentTestObject.getCodeClassification());
			List testObjectList = (List) testObjectMap.get(new Long(code));
			if (testObjectList == null) {
				testObjectList = new ArrayList();
				testObjectMap.put(new Long(code), testObjectList);
			}
			testObjectList.add(currentTestObject);
		}

		List performUnitList = new ArrayList();

		for (Iterator iterator = testObjectMap.keySet().iterator();
			iterator.hasNext();
			) {
			Long currentTestObjectCode = (Long) iterator.next();
			long testObjectCode = currentTestObjectCode.longValue();
			List allTestList = new ArrayList();
			List testObjectList =
				(List) testObjectMap.get(currentTestObjectCode);
			TestObject[] testObjectArray =
				(TestObject[]) testObjectList.toArray(
					new TestObject[testObjectList.size()]);
			for (Iterator iter = testMap.keySet().iterator();
				iter.hasNext();
				) {
				Long currentTestCode = (Long) iter.next();
				long testCode = currentTestCode.longValue();
				List testList = (List) testMap.get(currentTestCode);
				if ((testObjectCode & testCode) != 0) {
					allTestList.addAll(testList);
				}
			}
			Test[] testArray =
				(Test[]) allTestList.toArray(new Test[allTestList.size()]);
			
			performUnitList.add(
				new PerformUnit(testObjectArray, testArray));

		}
		performUnits =
			(PerformUnit[]) performUnitList.toArray(new PerformUnit[0]);
	}

	/**
	 * Adds request parameter.
	 * @param parameter the parameter to be added.
	 *
	 */
	public void addParameter(ParameterInterface parameter) {
		ParameterInterface newParameter = parameter;
		ParameterInterface[] tempParameters =
			new ParameterInterface[requestParameters.length + 1];
		for (int i = 0; i < requestParameters.length; i++)
			tempParameters[i] = requestParameters[i];
		tempParameters[requestParameters.length] = newParameter;
		requestParameters = tempParameters;
	}

	/**
	 * Returns the request parameters.
	 * @return request parameters.
	 *
	 */
	public ParameterInterface[] getParameters() {
		return requestParameters;
	}

	/**
	 * Adds test target to default perform unit.
	 * @param newSource target to be added.
	 *
	 */
	public void addSource(TestObject newSource) {
		if (performUnits != null)
			throw new IllegalStateException("sources cannot be added to this perform request");
		testObjects.add(newSource);
	}

	/**
	 * Adds test (name + parameters) to default perform unit.
	 * @param newTest test to be added.
	 *
	 */
	public void addTest(Test newTest) {
		if (performUnits != null)
			throw new IllegalStateException("tests cannot be added to this perform request");
		tests.add(newTest);
	}

	/**
	 * Adds test (name + parameters) to default perform unit.
	 * @param test name of the test to be added.
	 * @param parameters parameters of the test to be added.
	 */
	public void addTest(
		String testName,
		ParameterInterface[] parameters,
		String[] codeClassification) {
		addTest(
			new Test(
				testName,
				parameters,
				new ITestMessageDescription[0],
				codeClassification,
				true));
	}

	/**
	 * Return the perform request in StructureTree format.
	 */
	public StructureTree toStructureTree() {
		computePerformUnits();
		Header topHeader = new Header("REQUEST");
		topHeader.setParameter("NAME", name);
		topHeader.setParameter("TASK", task);
		StructureTree rootTree = new StructureTree(topHeader);

		for (int i = 0; i < requestParameters.length; i++) {
			StructureTree parameterTree =
				requestParameters[i].toStructureTree();
			rootTree.addNewChild(parameterTree);
		}

		for (int i = 0; i < performUnits.length; i++) {
			StructureTree unitTree =
				new StructureTree(new Header(TASK_PERFORM_UNIT));
			TestObject[] sources = performUnits[i].getSources();
			StructureTree testSetTree =
				new TestSet(TASK_PERFORM_TEST_SET, performUnits[i].getTests()).toStructureTree();
			unitTree.addChild(testSetTree);
			StructureTree sourceSetTree =
				new StructureTree(new Header(TASK_PERFORM_SOURCE_SET));
			for (int j = 0; j < sources.length; j++) {
				StructureTree testObjectTree = sources[j].toStructureTree();
				sourceSetTree.addChild(testObjectTree);
			}
			unitTree.addChild(sourceSetTree);
			rootTree.addNewChild(unitTree);
		}
		return rootTree;
	}


}