/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/ParameterTool.java#2 $
 */

package com.sap.tc.jtools.jtci;

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Tool methods to convert paramaters to a Structure tree and vice versa
 * @version   1.0, $Date: 2004/03/22 $
 * @author    BPL Tools
 */

public class ParameterTool {

  /* parameter types */
  public static final String PAR_TYPE_STRING = "STRING"; //$NON-NLS-1$
  public static final String PAR_TYPE_STRING_ARRAY = "STRING[]"; //$NON-NLS-1$
  public static final String PAR_TYPE_INT = "INT"; //$NON-NLS-1$
  public static final String PAR_TYPE_FLOAT = "FLOAT"; //$NON-NLS-1$
  public static final String PAR_TYPE_BOOLEAN = "BOOLEAN"; //$NON-NLS-1$
  public static final String PAR_TYPE_FILE = "FILE"; //$NON-NLS-1$
  public static final String PAR_TYPE_FILE_ARRAY = "FILE[]"; //$NON-NLS-1$
  public static final String PAR_TYPE_DIRECTORY = "DIRECTORY"; //$NON-NLS-1$
  public static final String PAR_TYPE_DIRECTORY_ARRAY = "DIRECTORY[]"; //$NON-NLS-1$
  public static final String PAR_TYPE_DEEP_STRUCTURE = "DEEP STRUCTURE"; //$NON-NLS-1$
	public static final String PAR_MSG_KEY = "__MSG_KEY"; //$NON-NLS-1$
  public static final String PAR_LINK = "__LINK"; //$NON-NLS-1$
  public static final String PAR_EXCEPTION = "EXCEPTION"; //$NON-NLS-1$

  private static final String[] TYPE_NAMES =
    new String[] {
      PAR_TYPE_STRING,
      PAR_TYPE_STRING_ARRAY,
      PAR_TYPE_INT,
      PAR_TYPE_FLOAT,
      PAR_TYPE_BOOLEAN,
      PAR_TYPE_FILE,
      PAR_TYPE_FILE_ARRAY,
      PAR_TYPE_DIRECTORY,
      PAR_TYPE_DIRECTORY_ARRAY,
      PAR_TYPE_DEEP_STRUCTURE };

  /**
   * returns  names of all supported parameter types
   * 
   * @return supported parameters
   * 
   **/
  static public String[] getParameterTypes() {
    return TYPE_NAMES;
  }

  /**
   * convert the basic implementations of ParameterInterface to a StructureTree
   * @see StringParameter
   * @see StringArrayParameter
   * @see DeepStructureParameter
   **/
  public static ParameterInterface createParameter(StructureTree pStructureTree)
    throws BadTreeException {

    Header header = pStructureTree.getHeader();

    if (null == header) {
      throw new BadTreeException("no header"); //$NON-NLS-1$
    }

    String name = header.getParameter(ParameterInterface.PARAMETER_NAME);
    String type = header.getParameter(ParameterInterface.PARAMETER_TYPE);
    if (null == name || null == type) {
      throw new BadTreeException("illegal header format"); //$NON-NLS-1$
    }
    if (type.equals(PAR_TYPE_INT)) {
      return IntParameter.createFromStructureTree(pStructureTree);
    } else if (type.equals(PAR_TYPE_FLOAT)) {
      return FloatParameter.createFromStructureTree(pStructureTree);
    } else if (type.equals(PAR_TYPE_BOOLEAN)) {
      return BooleanParameter.createFromStructureTree(pStructureTree);
    } else if (
      type.equals(PAR_TYPE_STRING)
        || type.equals(PAR_TYPE_FILE)
        || type.equals(PAR_TYPE_DIRECTORY)) {
      return StringParameter.createFromStructureTree(name, pStructureTree);
    } else if (
      type.equals(PAR_TYPE_STRING_ARRAY)
        || type.equals(PAR_TYPE_DIRECTORY_ARRAY)
        || type.equals(PAR_TYPE_FILE_ARRAY)) {
      return StringArrayParameter.createFromStructureTree(name, pStructureTree);
    } else if (type.equals(ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {
      return DeepStructureParameter.createFromStructureTree(
        name,
        pStructureTree);
    } else {
      throw new BadTreeException("unknown type: " + type); //$NON-NLS-1$
    }
  }

  /**
   * creates a parameter with given name, type and value. The value class must 
   * be compatible with the chosen type
   * 
   * @param name parameter name
   * @param type parameter type
   * @param value parameter value
   * 
   * @return created parameter
   */
  public static ParameterInterface createParameter(
    String name,
    String type,
    Object value) {
    if (isSimpleType(type)) {
      if ((value != null) && !(value instanceof String)) { //  null is allowed
        throw new IllegalArgumentException("value must be of type String"); //$NON-NLS-1$
      } else {
        if (type.equals(PAR_TYPE_BOOLEAN)) {
          return new BooleanParameter(
            name,
            value == null
              ? false
              : Boolean.valueOf((String) value).booleanValue());
        } else if (type.equals(PAR_TYPE_INT)) {
          return new IntParameter(name, (String) value);
        } else if (type.equals(PAR_TYPE_FLOAT)) {
          return new FloatParameter(name, (String) value);
        } else if (
          type.equals(PAR_TYPE_STRING)
            || type.equals(PAR_TYPE_FILE)
            || type.equals(PAR_TYPE_DIRECTORY)) {
          return new StringParameter(name, (String) value, type);
        } else {
          throw new IllegalArgumentException("unknown type: " + type); //$NON-NLS-1$
        }
      }
    } else if (isArrayType(type)) {
      if ((value != null) && !(value instanceof String[])) {
        throw new IllegalArgumentException("value must be of type String[]"); //$NON-NLS-1$
      }
      return new StringArrayParameter(name, (String[]) value, type);
    } else if (type.equals(ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {
      if ((value != null) && !(value instanceof StructureTree)) {
        throw new IllegalArgumentException("value must be of type StructureTree"); //$NON-NLS-1$
      }
      return new DeepStructureParameter(name, (StructureTree) value);
    } else {
      throw new IllegalArgumentException("unknown type: " + type); //$NON-NLS-1$
    }
  }
  /**
          * checks if a given type is a simple type
          * 
          * @param type type name
          * 
          * @return <code>true</code> if type is a simple type
          */
  public static boolean isSimpleType(String type) {
    return PAR_TYPE_STRING.equals(type)
      || PAR_TYPE_FILE.equals(type)
      || PAR_TYPE_BOOLEAN.equals(type)
      || PAR_TYPE_FLOAT.equals(type)
      || PAR_TYPE_INT.equals(type)
      || PAR_TYPE_DIRECTORY.equals(type);
  }

  /**
    * checks if a given type is an array type
    * 
    * @param type type name
    * 
    * @return <code>true</code> if type is an array type
    */
  public static boolean isArrayType(String type) {
    return PAR_TYPE_STRING_ARRAY.equals(type)
      || PAR_TYPE_DIRECTORY_ARRAY.equals(type)
      || PAR_TYPE_FILE_ARRAY.equals(type);
  }

  /**
   * checks if two parameters are equal
   * 
   * @param par1 first parameter
   * @param par2 second parameter
   * 
   * @return <code>true</code> if parameters have the same type, name, and value
   */
  public static boolean eq(
    ParameterInterface[] par1,
    ParameterInterface[] par2) {
    if (par1.length != par2.length)
      return false;
    for (int i = 0; i < par1.length; i++) {
      if ((!par1[i].getName().equals(par2[i].getName()))
        || (!par1[i].getType().equals(par2[i].getType())))
        return false;
      if ((ParameterTool.isSimpleType(par1[i].getType()))
        && (!(par1[i].valueToString()).equals(par2[i].valueToString())))
        return false;
      if (ParameterTool.isArrayType(par1[i].getType())) {
        String[] a1 = (String[]) par1[i].getValue();
        String[] a2 = (String[]) par2[i].getValue();
        if (a1.length != a2.length)
          return false;
        for (int j = 0; j < a1.length; j++) {
          if (!a1[j].equals(a2[j]))
            return false;
        }
      }
      if ((par1[i].getType().equals(PAR_TYPE_DEEP_STRUCTURE))
        && (!((StructureTree) par1[i].getValue())
          .equals((StructureTree) par2[i].getValue())))
        return false;
    }
    return true;
  }
}