/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/Dispatcher.java#4 $
 */

package com.sap.tc.jtools.jtci;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import com.sap.tc.jtools.jlint.ProcessorUnitFactory;
import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.TestComponentInterface;
import com.sap.tc.jtools.jlint.TestConflictException;
import com.sap.tc.jtools.jlint.TestManager;
import com.sap.tc.jtools.jlint.exceptions.InvalidDataProviderException;
import com.sap.tc.jtools.jlint.extension.TestProviderInterface;
import com.sap.tc.jtools.jlint.i18n.ResultLocalizer;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestSpecificationException;
import com.sap.tc.jtools.jtci.exceptions.PerformException;
import com.sap.tc.jtools.jtci.interfaces.JtciDataInterface;
import com.sap.tc.jtools.jtci.interfaces.Listener;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.cache.CacheInterface;

/**
 * This class is used to process a test requests.
 * 
 * <p>
 * Copyright (c) 2003, SAP AG
 * 
 */

public class Dispatcher {

  private static final JtciDataInterface[] EMPTY_EXTERNAL_DATA = new JtciDataInterface[0];
  private static TestManager testManager;
  private static ResultLocalizer localizer;
  private TestProviderInterface testProvider;
  private Listener listener = null;

  /**
   * constructor
   * 
   * @param  provider test provider that provides available tests
   * and relative libraries
   * 
   */
  public Dispatcher(TestProviderInterface provider, CacheInterface cache) {
    testProvider = provider;
    ProcessorUnitFactory.setTestProvider(provider);
    try {
      testManager = new TestManager(provider.getTestDescriptionSet(), cache);
      localizer = new ResultLocalizer(provider);
    } catch (TestConflictException e) {
      e.printStackTrace();
    }
  }

  /**
   * constructor
   * 
   * @param  provider test provider that provides available tests
   * and relative libraries
   * 
   */
  public Dispatcher(TestProviderInterface provider) {
    this(provider, null);
  }

  /**
   * returns array of available tests  
   * 
   * @return available tests 
   * 
   */
  public TestDescriptionInterface[] getTests() {
    return (TestDescriptionInterface[]) testManager
      .getAllTests()
      .getAllLeaves();
  }

  /**
   * returns the available tests, organized hierarchically according to their category  
   * 
   * @return available tests 
   * 
   */
  public TestTree getTestTree() {
    return testManager.getAllTests();
  }

  /**
   * Sets listener
   * 
   * @param l listener
   *
   */
  public void setListener(Listener l) {
    listener = l;
    testManager.setListener(l);
  }

  /**
   * Processes a request.
   *
   * @param request the request
   * @return the corresponding response
   * @throws PerformException if a test job couldn't be performed
   * @throws InvalidTestSpecificationException if a test specification (test name
   * + parameters) is not valid
   */
  public PerformResponse processRequest(PerformRequest request)
    throws PerformException, InvalidTestSpecificationException {
    return processRequest(request, EMPTY_EXTERNAL_DATA);
  }

  /**
   * Processes a request.
   *
   * @param request the request
   * @param externalData external data structures
   * @return the corresponding response
   * @throws PerformException if a test job couldn't be performed
   * @throws InvalidTestSpecificationException if a test specification (test name
   * + parameters) is not valid
   */
  public PerformResponse processRequest(
    PerformRequest request,
    JtciDataInterface[] externalData)
    throws PerformException, InvalidTestSpecificationException {

    List rawResultList  = new ArrayList();
    try {

      ParameterInterface[] requestParameters = request.getParameters();

      //loop over performUnits
      PerformUnit[] performUnits = request.getPerformUnits();
      TestDescriptionInterface[] allTests =
        testManager.getAllTests().getAllLeaves();
      for (int j = 0; j < performUnits.length; j++) {
        // get Tests and Sources
        TestObject[] sources = performUnits[j].getSources();
        Test[] tests = performUnits[j].getTests();

        for (int l = 0; l < sources.length; l++) {
          testManager.reset();
          testManager.addExternalData(externalData);
          testloop : for (int k = 0; k < tests.length; k++) {

            String testName = tests[k].getName();
            // is this a legitimate test?
            TestDescriptionInterface test = null;
            boolean found = false;
            for (int ll = 0; ll < allTests.length; ll++) {
              if (allTests[ll].getName().equals(testName)) {
                test = allTests[ll];
                found = true;
              }
            }
            if (!found)
              continue testloop;

            ParameterInterface[] signatureParameters =
              test.getInputParameters();
            ParameterInterface[] parameters =
              new ParameterInterface[signatureParameters.length];
            for (int ll = 0; ll < signatureParameters.length; ll++) {
              parameters[ll] = signatureParameters[ll];
              Serializable value = signatureParameters[ll].getValue();
              if (value != null)
                parameters[ll].setValue(value); //default
            }
            ParameterInterface[] explicitParameters = tests[k].getParameters();
            if (null == explicitParameters) {
              explicitParameters = new ParameterInterface[0];
            }
            for (int ll = 0; ll < explicitParameters.length; ll++) {
              for (int m = 0; m < parameters.length; m++) {
                if (parameters[m]
                  .getName()
                  .equals(explicitParameters[ll].getName())) {
                  parameters[m].setValue(explicitParameters[ll].getValue());
                }

              }

            }
            // merge parameters and requestParameters
            ParameterInterface[] totalParameters =
              new ParameterInterface[parameters.length
                + requestParameters.length];
            for (int ii = 0; ii < parameters.length; ii++)
              totalParameters[ii] = parameters[ii];
            for (int ii = 0; ii < requestParameters.length; ii++) {
              totalParameters[parameters.length + ii] = requestParameters[ii];

            }
            testManager.addTest(testName, totalParameters, Test.mergeMessages(test,tests[k]), sources[l]);
          }

          ResultInterface[] results = testManager.performTests();
          rawResultList.addAll(Arrays.asList(results));
          if (listener != null) {
            listener.processedTestObject(results);
            if (listener.isCanceled()) {
              return null;
            }
          }
        }
      }

    } catch (Exception e) {
      throw new PerformException(e);
    }
    PerformResponse response = new PerformResponse(request.getName());
    response.addResults(processResults((ResultInterface[])rawResultList.toArray(new ResultInterface[0])));
    return response;
  }

  /**
   * this method does the post-processing (e.g. compacting) of results
   * that belong to _one_ test
   * 
   * @param testName name of the test that produced the results
   * @param results _MUST_ only be results of the test specified above
   * @return processed results (or null if not implemented)
   */
  private ResultInterface[] processResults(
    ResultInterface[] results,
    String testName)
    throws InvalidDataProviderException, InvalidTestSpecificationException {

    // parameter sanity checking
    if (testName == null || testName.length() == 0) {
      throw new IllegalArgumentException("invalid test name: " + testName); //$NON-NLS-1$
    }
    for (int i = 0; i < results.length; i++) {
      if (!testName.equals(results[i].getTestName())) {
        throw new IllegalArgumentException("one or more results are not generated by " + testName); //$NON-NLS-1$
      }
    }

    TestDescriptionInterface[] tests = getTests();
    TestDescriptionInterface testDescr = null;
    boolean found = false;
    for (int i = 0; i < tests.length && !found; i++) {
      if (testName.equals(tests[i].getName())) {
        testDescr = tests[i];
        found = true;
      }
    }

    if (!found) {
      throw new InvalidTestSpecificationException("test " + testName + " not found!"); //$NON-NLS-1$ //$NON-NLS-2$
    }
    TestComponentInterface test =
      ProcessorUnitFactory.createTestComponent(testDescr);
    return test.processErrors(results);
  }

  /**
   * this method does the post-processing (e.g. compacting) 
   * as well as localizing of results
   * that belong to an arbitrary collection of tests
   * 
   * @param results results to be processed
   * @return processed results  
   */
  public ResultInterface[] processResults(
    ResultInterface[] results) {
    HashMap rawResults = new HashMap();
    List processedResults = new ArrayList();
    // order results by test
    for (int i = 0; i < results.length; i++) {
      if (results[i].getPriority() == Result.SEVERITY_INTERNAL_ERROR) {
        // internal errors are not processed
        processedResults.add(results[i]);
      } else {
        String testName = results[i].getTestName();
        Set oldResults = (Set) rawResults.get(testName);
        if (oldResults == null) {
          Set resultSet = new HashSet();
          resultSet.add(results[i]);
          rawResults.put(testName, resultSet);
        } else {
          oldResults.add(results[i]);
        }
      }
    }

    // process results of each test
    for (Iterator it = rawResults.keySet().iterator(); it.hasNext();) {
      String testName = (String) it.next();
      Set resultSet = (Set) rawResults.get(testName);
      ResultInterface[] oldResults =
        (ResultInterface[]) resultSet.toArray(
          new ResultInterface[resultSet.size()]);
      ResultInterface[] newResults;
      try {
        newResults = this.processResults(oldResults, testName);
      } catch (Exception e) {
        e.printStackTrace();
        newResults = null;
      }
      // if not implemented, keep the original results
      if (newResults == null) {
        newResults = oldResults;
      }
      processedResults.addAll(Arrays.asList(newResults));
    }

    return localizer.localize(
      (ResultInterface[]) processedResults.toArray(
        new ResultInterface[processedResults.size()]));
  }
  
  public TestProviderInterface getTestProvider() {
    return testProvider;
  }

}