/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jtci/DeepStructureParameter.java#2 $
 */


package com.sap.tc.jtools.jtci;

import com.sap.tc.jtools.util.structures.*;
import com.sap.tc.jtools.util.xml.XMLTool;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.ascii.StringBuf;
import com.sap.tc.jtools.util.persistable.*;

/**
 * Implementation of a container for hierachical parameters.
 * @version   1.0, $Date: 2004/03/22 $
 * @author    BPL Tools
 */

final public class DeepStructureParameter implements Persistable, ParameterInterface {

 
 

  public static final String EMPTY_STRING = ""; //$NON-NLS-1$
  
  private StructureTree value;
  final protected String name;

  /** 
   * Constructor which builds a parameter from a StructureTree.
   */
  DeepStructureParameter(String name, StructureTree value) {
    this.name = name;
    this.value = value;
  }


  /** 
   * Constructor which builds a parameter from a Persistable which is internally 
   * converted to a StructureTree.
   */
  DeepStructureParameter(String name, Persistable object) {
    this.name = name;
    this.value = object.toStructureTree();
  }


  /**
   * Returns the name of this Parameter.
   */
  public String getName() {
    return name;
  }


  /**
   * Returns the Type name.
   */
  public String getType() {
    return ParameterTool.PAR_TYPE_DEEP_STRUCTURE;
  }


  /**
   * Returns the value as Serializable.
   */
  public java.io.Serializable getValue() {
    return value;
  }

  /**
   * Returns the value in its own Type.
   */
  public StructureTree getValueTrue() {
    return value;
  }
  
  /**
   * Returns a String Representation of the value.
   */
  public String valueToString() {
    if (null != value) {
      return XMLTool.toString(value);
    }
    return EMPTY_STRING;
  }

  /**
   * Returns a String Representation
   */
  public String toString() {
    String valString = valueToString();
    StringBuf buf = new StringBuf(name.length() + valString.length() + 10);
    return buf.append(name).append(':').append(valString).toString();
  }


  /**
   * Allows to change the value of this argument.
   * @param newValue eighter a instance of StructureTree or Persistable
   * @see ParameterInterface#getValue()
   **/
  public void setValue(java.io.Serializable newValue) {
    if (newValue == null) {
      value = null;
    } else if (newValue instanceof StructureTree) {
      value = (StructureTree) newValue;
    } else if (newValue instanceof Persistable) {
      value = ((Persistable)newValue).toStructureTree();
    } else {
      throw new IllegalArgumentException ("arg must be StructureTree or Persistable"); //$NON-NLS-1$
    }
  }


  /**
   * Converts the parameter to an StructureTree representation.
   **/
  // keep in synch with ParameterTool
  public StructureTree toStructureTree() {
    Header paramHeader = new Header(INPUT_PARAMETER_TAG);
    paramHeader.setParameter(PARAMETER_NAME, name);
    paramHeader.setParameter(PARAMETER_TYPE, ParameterTool.PAR_TYPE_DEEP_STRUCTURE);
    StructureTree paramTree = new StructureTree(paramHeader);
    if (null != value) {
      Header contentHeader =
        new Header(PARAMETER_VALUE);
      StructureTree contentTree = new StructureTree(contentHeader);
        paramTree.addNewChild(contentTree);
      contentTree.addNewChild(value);
    }
    return paramTree;
  }

  /**
   * Internal method to create an instance out of according StructureTree.
   */
  static DeepStructureParameter createFromStructureTree(
    String nameInfo,
    StructureTree treeInfo) throws BadTreeException {

    StructureTree contents[] =
      treeInfo.getChildren(PARAMETER_VALUE);
    if (null == contents || 0 == contents.length) {
      return new DeepStructureParameter(nameInfo, null);
    }
    if (1 != contents.length) {
      throw new BadTreeException("illegal value content"); //$NON-NLS-1$
    }

    StructureTree[] values = contents[0].getChildren();

    if (null == values || 0 == values.length) {
      return new DeepStructureParameter(nameInfo, null);
    } else if (1 == values.length) {
      return new DeepStructureParameter(nameInfo, values[0]);
    } else {
      throw new BadTreeException("too much children: " + values.length);  //$NON-NLS-1$
    }
  }

}