/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/statistics/TestSummary.java#2 $
 */

package com.sap.tc.jtools.jlint.statistics;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.persistable.Persistable;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * @author SAP AG d034036
 *
 */
public class TestSummary implements Persistable {

	private static final int ERROR_INDEX = 0;
	private static final int WARNING_INDEX = 1;
	private static final int INFO_INDEX = 2;

	private Map test2Results; // test --> map(msgKey--> int[3] count)
	private int[] totals;

	public TestSummary(StructureTree tree) {
		test2Results = new HashMap();
		totals = new int[] { 0, 0, 0 };
		StructureTree[] tests = tree.getChildren("TEST");
		for (int i = 0; i < tests.length; i++) {

			StructureTree[] msgKeyTrees = tests[i].getChildren("MSG_KEY");
			String testName = tests[i].getParameter("TEST_NAME");
			Map key2Messages = new HashMap();
			for (int j = 0; j < msgKeyTrees.length; j++) {
				String msgKey = msgKeyTrees[j].getParameter("NAME");
				int[] count = new int[3];
				count[ERROR_INDEX] = Integer.parseInt(msgKeyTrees[j].getParameter("ERRORS"));
				count[WARNING_INDEX] = Integer.parseInt(msgKeyTrees[j].getParameter("WARNINGS"));
				count[INFO_INDEX] = Integer.parseInt(msgKeyTrees[j].getParameter("INFOS"));								
				for (int k = 0; k < count.length; k++) {
					totals[k] += count[k];
				}
				key2Messages.put(msgKey,count);
			}
			test2Results.put(testName, key2Messages);
		}
	}

	public TestSummary(ResultInterface[] results) {
		test2Results = new HashMap();
		totals = new int[] { 0, 0, 0 };

		for (int i = 0; i < results.length; i++) {
			int prio = results[i].getPriority();
			if (prio > Result.SEVERITY_INFO || prio < Result.SEVERITY_ERROR)
				continue;
			if (MetricResult.isGlobalMetricResult(results[i]))
				continue;
			String testName = results[i].getTestName();
			String msgKey = results[i].getErrorParameters().getProperty("__MSG_KEY");
			if ( msgKey == null ) {
				msgKey = "";
			}
			
			Map key2Messages = (Map) test2Results.get(testName);
			if ( key2Messages == null ) {
				key2Messages = new HashMap();
				test2Results.put(testName,key2Messages);
			}
			
			
			int[] previousResultCount = (int[]) key2Messages.get(msgKey);
			if (previousResultCount == null) {
				previousResultCount = new int[3];
				key2Messages.put(msgKey, previousResultCount);
			}
			
			switch (prio) {
				case Result.SEVERITY_ERROR :
					previousResultCount[ERROR_INDEX]++;
					totals[ERROR_INDEX]++;
					break;
				case Result.SEVERITY_WARNING :
					previousResultCount[WARNING_INDEX]++;
					totals[WARNING_INDEX]++;
					break;
				case Result.SEVERITY_INFO :
					previousResultCount[INFO_INDEX]++;
					totals[INFO_INDEX]++;
					break;
				default :
					break;
			}
		}
	}

	private TestSummary(Map test2Results, int[] totals) {
		this.test2Results = test2Results;
		this.totals = totals;
	}

	public String[] getTestNames() {
		return (String[]) test2Results.keySet().toArray(new String[0]);
	}

	public int getErrorCountForTest(String testId) {
		Map key2Messages = (Map) test2Results.get(testId);
	 	if ( key2Messages == null ) {
	 		return 0;
	 	}
		int errors = 0;
	 	for (Iterator iter = key2Messages.values().iterator(); iter.hasNext();) {
			int[] partialResults = (int[]) iter.next();
			errors+= partialResults[ERROR_INDEX];
		}
		return errors;
	}
	
	public int getErrorCountForTest(String testId, String msgId) {
		Map key2Messages = (Map) test2Results.get(testId);
		if ( key2Messages == null ) {
			return 0;
		}
		int errors = 0;
		
		int[] messages = (int[]) key2Messages.get(msgId);
		if ( messages == null )
			return 0;
		return messages[ERROR_INDEX];
	}
	

	public int getWarningCountForTest(String testId) {
		Map key2Messages = (Map) test2Results.get(testId);
		if ( key2Messages == null ) {
			return 0;
		}
		int warnings = 0;
		for (Iterator iter = key2Messages.values().iterator(); iter.hasNext();) {
			int[] partialResults = (int[]) iter.next();
			warnings+= partialResults[WARNING_INDEX];
		}
		return warnings;
	}
	
	public int getWarningCountForTest(String testId, String msgId) {
		Map key2Messages = (Map) test2Results.get(testId);
		if ( key2Messages == null ) {
			return 0;
		}
		int errors = 0;
		
		int[] messages = (int[]) key2Messages.get(msgId);
		if ( messages == null )
			return 0;
		return messages[WARNING_INDEX];
	}
	
	
	public int getInfoCountForTest(String testId) {
		Map key2Messages = (Map) test2Results.get(testId);
		if ( key2Messages == null ) {
			return 0;
		}
		int infos = 0;
		for (Iterator iter = key2Messages.values().iterator(); iter.hasNext();) {
			int[] partialResults = (int[]) iter.next();
			infos+= partialResults[INFO_INDEX];
		}
		return infos;
	}

	public int getInfoCountForTest(String testId, String msgId) {
		Map key2Messages = (Map) test2Results.get(testId);
		if ( key2Messages == null ) {
			return 0;
		}
		int errors = 0;
		
		int[] messages = (int[]) key2Messages.get(msgId);
		if ( messages == null )
			return 0;
		return messages[INFO_INDEX];
	}
		
	public String[] getMessageKeys(String testName,boolean ignoreWarnings,boolean ignoreInfos) {
		Set out = new HashSet();
		Map key2Messages = (Map) test2Results.get(testName);
		if ( key2Messages == null ) {
			return new String[0];
		}
		for (Iterator iter = key2Messages.keySet().iterator(); iter.hasNext();) {
			String key = (String) iter.next();
			if ( getMessageCount(testName,key,ignoreWarnings,ignoreInfos) > 0 )
				out.add(key);
		}
		
		return (String[]) out.toArray(new String[0]);
	}
	
	public int getMessageCount(String testName, String msgKey,boolean ignoreWarnings,boolean ignoreInfos) {
		Map key2Messages = (Map) test2Results.get(testName);
		if ( key2Messages == null ) {
			return 0;
		}
		int[] messages = (int[]) key2Messages.get(msgKey);
		if ( messages == null )
			return 0;
		int total = messages[ERROR_INDEX];
		if (!ignoreWarnings)
			total += messages[WARNING_INDEX];
		if (!ignoreInfos)
			total += messages[INFO_INDEX];
		return total;	
	}

	public int getTotalErrorCount() {
		return totals[ERROR_INDEX];
	}

	public int getTotalWarningCount() {
		return totals[WARNING_INDEX];
	}

	public int getTotalInfoCount() {
		return totals[INFO_INDEX];
	}

	public static TestSummary merge(TestSummary[] toBeAdded) {
		Map test2Messages = new HashMap();
		int[] totals = new int[] { 0, 0, 0 };
		for (int i = 0; i < toBeAdded.length; i++) {
			String[] testNames = toBeAdded[i].getTestNames();

			for (int j = 0; j < testNames.length; j++) {
				Map key2Messages = (Map) test2Messages.get(testNames[j]);
				if ( key2Messages == null ) {
					key2Messages = new HashMap();
					test2Messages.put(testNames[j],key2Messages);
				}
				Map mapToBeAdded = (Map) toBeAdded[i].test2Results.get(testNames[j]);

				for (Iterator iter = mapToBeAdded.keySet().iterator(); iter.hasNext();) {
					String msgKey = (String) iter.next();
					int[] count = (int[]) key2Messages.get(msgKey);
					if (count == null) {
						count = new int[] { 0, 0, 0 };
						key2Messages.put(msgKey, count);
					}
					int[] messages = (int[]) mapToBeAdded.get(msgKey);
					for (int k = 0; k < messages.length; k++) {
						count[k]+=messages[k];
						totals[k]+=messages[k];
					}
				}
			}
		}

		return new TestSummary(test2Messages, totals);
	}

	public StructureTree toStructureTree() {
		StructureTree tree = new StructureTree(new Header("TESTS"));
		for (Iterator iter = test2Results.keySet().iterator(); iter.hasNext();) {
			String testName = (String) iter.next();
			Header testHeader = new Header("TEST");
			testHeader.setParameter("TEST_NAME", testName);
			StructureTree testTree = new StructureTree(testHeader);
			tree.addNewChild(testTree);			
			Map key2Messages = (Map) test2Results.get(testName);
			for (Iterator iterator = key2Messages.keySet().iterator(); iterator.hasNext();) {
				String key = (String) iterator.next();
				int[] resultCount = (int[]) key2Messages.get(key);
				Header keyHeader = new Header("MSG_KEY");
				keyHeader.setParameter("NAME",key);
				keyHeader.setParameter("ERRORS", Integer.toString(resultCount[ERROR_INDEX]));
				keyHeader.setParameter("WARNINGS", Integer.toString(resultCount[WARNING_INDEX]));
				keyHeader.setParameter("INFOS", Integer.toString(resultCount[INFO_INDEX]));		
				testTree.addNewChild(keyHeader);		
			}			


		}
		return tree;
	}

}
