/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/statistics/MetricResult.java#2 $
 */

package com.sap.tc.jtools.jlint.statistics;

import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.persistable.Persistable;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * @author SAP AG d034036
 *
 */
public class MetricResult implements Persistable {
 

	private static final int SUM = 0x1;
	private static final int AVERAGE = 0x2;
	private static final int MAX = 0x4;
	private static final int MIN = 0x8;
	private static final int STD_DEV = 0x10;
	
	private String testName;
	private String metricName;
	private int metricCode;
	private int count;
	private float sum;
	private float quadraticSum;
	private float max;
	private float min;

	public MetricResult(
		String testName,
		String metricName,
		int metricCode,
		int count,
		float sum,
		float quadraticSum,
		float max,
		float min) {
		this.testName = testName;
		this.metricCode = metricCode;
		this.metricName = metricName;
		this.count = count;
		this.sum = sum;
		this.quadraticSum = quadraticSum;
		this.max = max;
		this.min = min;
	}
	
	public MetricResult(StructureTree tree) {
		testName = tree.getParameter("TEST_NAME");
		metricName = tree.getParameter("METRIC_NAME");
		metricCode = Integer.parseInt(tree.getParameter("METRIC_CODE"));
		count = Integer.parseInt(tree.getParameter("COUNT"));
		sum = Float.parseFloat(tree.getParameter("SUM"));
		quadraticSum = Float.parseFloat(tree.getParameter("Q_SUM"));
		max = Float.parseFloat(tree.getParameter("MAX"));
		min = Float.parseFloat(tree.getParameter("MIN"));
	}

	public MetricResult(ResultInterface result) {
		testName = result.getTestName();
		metricName = result.getErrorParameters().getProperty("METRIC_NAME");
		metricCode =
			Integer.parseInt(result.getErrorParameters().getProperty("__MSG_KEY").substring("metric.".length()));
		count = Integer.parseInt(result.getErrorParameters().getProperty("COUNT"));
		sum = Float.parseFloat(result.getErrorParameters().getProperty("SUM"));
		quadraticSum = Float.parseFloat(result.getErrorParameters().getProperty("QUADRATIC_SUM"));
		max = Float.parseFloat(result.getErrorParameters().getProperty("MAX"));
		min = Float.parseFloat(result.getErrorParameters().getProperty("MIN"));
	}

	public boolean isSumDefined() {
		return (metricCode & SUM) == SUM;
	}

	public boolean isAverageDefined() {
		return (metricCode & AVERAGE) == AVERAGE;
	}

	public boolean isMaxDefined() {
		return (metricCode & MAX) == MAX;
	}

	public boolean isMinDefined() {
		return (metricCode & MIN) == MIN;
	}

	public boolean isStdDevDefined() {
		return (metricCode & STD_DEV) == STD_DEV;
	}

	/**
	 * @return
	 */
	public int getCount() {
		return count;
	}

	/**
	 * @return
	 */
	public float getMax() {
		return max;
	}

	public String getTestName() {
		return testName;
	}
	
	/**
	 * @return
	 */
	public String getMetricName() {
		return metricName;
	}

	/**
	 * @return
	 */
	public float getMin() {
		return min;
	}

	/**
	 * @return
	 */
	public float getQuadraticSum() {
		return quadraticSum;
	}

	/**
	 * @return
	 */
	public float getSum() {
		return sum;
	}

	/**
	 * @return
	 */
	public float getAverage() {
		return sum / count;
	}

	/**
	 * @return
	 */
	public float getStdDev() {
		return (float) Math.sqrt(quadraticSum / count - Math.pow(sum / count, 2));
	}

	static MetricResult merge(MetricResult[] addenda) {

		if (addenda.length == 0) {
			throw new IllegalArgumentException("nothing to merge");
		}
		String testName = addenda[0].testName;
		String metricName = addenda[0].metricName;
		int metricCode = addenda[0].metricCode;
		int count = 0;
		float sum = 0;
		float quadraticSum = 0;
		float max = Float.MIN_VALUE;
		float min = Float.MAX_VALUE;
		for (int i = 0; i < addenda.length; i++) {
			count += addenda[i].count;
			sum += addenda[i].sum;
			quadraticSum += addenda[i].quadraticSum;
			if (max < addenda[i].max)
				max = addenda[i].max;
			if (min > addenda[i].min)
				min = addenda[i].min;
		}
		return new MetricResult(testName, metricName, metricCode, count, sum, quadraticSum, max, min);
	}

	public static boolean isMetricResult(ResultInterface result) {
		return result.getErrorParameters().getProperty("METRIC_NAME") != null;
	}

	public static boolean isGlobalMetricResult(ResultInterface result) {
		return isMetricResult(result)
			&& result.getErrorParameters().getProperty("INTERNAL_DATA") != null
			&& result.getErrorParameters().getProperty("INTERNAL_DATA").equals("true");
	}

	public StructureTree toStructureTree() {
		Header header = new Header("METRIC_RESULT");
		header.setParameter("TEST_NAME",testName);
		header.setParameter("METRIC_NAME",metricName);
		header.setParameter("METRIC_CODE",Integer.toString(metricCode));
		header.setParameter("COUNT",Integer.toString(count));
		header.setParameter("SUM",Float.toString(sum));
		header.setParameter("Q_SUM",Float.toString(quadraticSum));
		header.setParameter("MAX",Float.toString(max));
		header.setParameter("MIN",Float.toString(min));
		StructureTree tree = new StructureTree(header);
		return tree;
	}

}
