/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/i18n/ResultLocalizer.java#2 $
 */


package com.sap.tc.jtools.jlint.i18n;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import com.sap.tc.jtools.jlint.extension.Module;
import com.sap.tc.jtools.jlint.extension.TestSet;
import com.sap.tc.jtools.jlint.extension.TestProviderInterface;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestTree;
import com.sap.tc.jtools.jtci.TestTreeNode;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;

/**
 * TODO: description of this class/interface 
 * @author D037913
 */
public class ResultLocalizer {

  private Map testName2BundleMap = new HashMap();

  private static final String PARAMETER_PREFIX = "{";
  private static final String PARAMETER_SUFFIX = "}";
  private static final String MISSING_RESOURCEBUNDLE_MARKER = "!!";
  private static final String MISSING_RESOURCE_MARKER = "!";

  public ResultLocalizer(TestProviderInterface testProvider) {
    Module[] modules = testProvider.getModules();
    for (int i = 0; i < modules.length; i++) {
      TestSet[] testSets = modules[i].getTestSets();
      for (int j = 0; j < testSets.length; j++) {
        TestTree testTree = testSets[j].getTestDescriptionSet().getTestTree();
        descendTree(testTree, testSets[j].getLoader(), null);
      }
    }
  }

  private void descendTree(
    TestTree testTree,
    ClassLoader loader,
    String parentResBundle) {
    TestTreeNode[] children = testTree.getNodes();
    for (int i = 0; i < children.length; i++) {
      if (children[i].getNodeType() == TestTreeNode.TREE) {
        String resBundle =
          readResourceBundle((TestTree) children[i], loader, parentResBundle);
        descendTree((TestTree) children[i], loader, resBundle);
      }
    }
  }

  private String readResourceBundle(
    TestTree tree,
    ClassLoader loader,
    String parentResBundle) {
    String descrResBundle = tree.getResourceBundle();
    if (descrResBundle == null) {
      // inherit from parent
      descrResBundle = parentResBundle;
    }
    TestDescriptionInterface[] tests = tree.getAllLeaves();
    for (int j = 0; j < tests.length; j++) {
      String testName = tests[j].getName();
      String testResBundle = tests[j].getResourceBundle();
      String bundleName = null;
      // precedence : first test-specific, then decriptor-specific 
      // resource bundle
      if (testResBundle != null) {
        bundleName = testResBundle;
      } else if (descrResBundle != null) {
        bundleName = descrResBundle;
      }
      if (bundleName == null)
        continue;
      ResourceBundle resBundle = null;
      try {
        resBundle =
          ResourceBundle.getBundle(bundleName, Locale.getDefault(), loader);
      } catch (MissingResourceException e) {
        //$JL-EXC$ no bundle present, ignore
      } catch (NullPointerException npe) {
        //$JL-EXC$ no bundle present, ignore
      }
      if (resBundle != null)
        testName2BundleMap.put(testName, resBundle);
    }
    return descrResBundle;
  }

  public ResultInterface[] localize(ResultInterface[] results) {
    ResourceBundle bundle;
    Properties msgProps;
    StringBuffer buf;
    String key;
    String errorText;
    String value = null;
    for (int i = 0; i < results.length; i++) {
      msgProps = results[i].getErrorParameters();
      key = msgProps.getProperty(ParameterTool.PAR_MSG_KEY);
      errorText = results[i].getErrorText().trim();
      if (key == null) {
        key = errorText;
      }
      bundle =
        (ResourceBundle) testName2BundleMap.get(results[i].getTestName());
      if (bundle == null) {
        if ("".equals(errorText)) {
          results[i].setErrorText(
            MISSING_RESOURCEBUNDLE_MARKER + key + MISSING_RESOURCEBUNDLE_MARKER);
        }
        continue;
      }
      try {
        value = bundle.getString(key);
      } catch (MissingResourceException e) {
        if ("".equals(errorText)) {
          results[i].setErrorText(
            MISSING_RESOURCE_MARKER + key + MISSING_RESOURCE_MARKER);
        }
        continue;
      } catch (NullPointerException npe) {
        if ("".equals(errorText)) {
          results[i].setErrorText(
            MISSING_RESOURCE_MARKER + key + MISSING_RESOURCE_MARKER);
        }
        continue;
      }
      if (msgProps.size() == 1) {
        //optimized: no parameter replacement necessary
        results[i].setErrorText(value);
        continue;
      }
      buf = new StringBuffer(value);
      Enumeration en = msgProps.keys();
      while (en.hasMoreElements()) {
        String propKey = (String) en.nextElement();
        replaceOccurences(
          buf,
          PARAMETER_PREFIX + propKey + PARAMETER_SUFFIX,
          msgProps.getProperty(propKey));
      }
      results[i].setErrorText(buf.toString());
    }
    return results;
  }

  /**
   * replaces all occurences of original in buf by replacement
   */
  private static void replaceOccurences(
    StringBuffer buf,
    String original,
    String replacement) {
    String str = buf.toString();
    int origLen = original.length();
    int diff = replacement.length() - origLen;
    // find all occurences
    int curIndex = 0;
    for (int i = 0; curIndex != -1; i++) {
      curIndex = str.indexOf(original, i == 0 ? 0 : curIndex + origLen);
      if (curIndex != -1) {
        int correctedPos = curIndex + i * diff;
        buf.replace(correctedPos, correctedPos + origLen, replacement);
      }
    }
  }

}
