/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/extension/TestProvider.java#2 $
 */

package com.sap.tc.jtools.jlint.extension;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import com.sap.tc.jtools.jlint.TestConflictException;
import com.sap.tc.jtools.jlint.TestDescriptionSet;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.TestTree;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestObjectSpecification;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * @author $Author: p4trans $
 */
public class TestProvider implements TestProviderInterface {

  private static TestProvider provider = null;
  private static final String CONFIG_TAG = "configuration";
  private static final String LIB_TAG = "library";
  private static final String MODULE_TAG = "module";
  private static final String PATH_ATTR = "path";
  private static boolean initialized = false;

  private Map modules = new HashMap();

  public static TestProvider initialize(File configFile)
    throws BadTreeException, FileNotFoundException, IOException, XMLParseException {
    if (provider != null)
      throw new IllegalStateException("already initialized!");
    StructureTree tree = XMLTool.parseReader(new FileReader(configFile));
    Properties p = new Properties();
    p.setProperty("CURRENT_DIR", configFile.getParentFile().getAbsolutePath());
    tree.resolveMacros(p);
    initialized = true;
    return provider = new TestProvider(tree);
  }

  public static TestProvider initialize(Module[] modules) {
    if (provider != null)
      throw new IllegalStateException("already initialized!");
    provider = new TestProvider(modules);
    initialized = true;
    return provider;
  }

  public static TestProvider getTestProvider() {
    if (provider == null)
      throw new IllegalStateException("not initialized!");
    return provider;
  }

  private TestProvider(StructureTree testProviderTree)
    throws BadTreeException, IOException, XMLParseException {
    if (!CONFIG_TAG.equals(testProviderTree.getTag()))
      throw new BadTreeException();
    // libs for root class loader
    StructureTree[] libTrees = testProviderTree.getChildren(LIB_TAG);
    List libs = new ArrayList();
    for (int i = 0; i < libTrees.length; i++) {
      libs.add(new File(libTrees[i].getParameter(PATH_ATTR).trim()));
    }
    ClassLoader rootClassLoader =
      new JarClassLoader(
        this.getClass().getClassLoader(),
        (File[]) libs.toArray(new File[0]));
    // modules
    StructureTree[] moduleTrees = testProviderTree.getChildren(MODULE_TAG);
    Map id2ModuleMap = new HashMap();
    for (int i = 0; i < moduleTrees.length; i++) {
      Module mod = new Module(moduleTrees[i], rootClassLoader, id2ModuleMap);
      modules.put(mod.getId(), mod);
      id2ModuleMap.put(mod.getId(), mod);
    }
  }

  private TestProvider(Module[] modules) {
    for (int i = 0; i < modules.length; i++) {
      this.modules.put(modules[i].getId(), modules[i]);
    }
  }

  public Class loadTestClass(String className) throws ClassNotFoundException {
    Collection mods = modules.values();
    for (Iterator iter = mods.iterator(); iter.hasNext();) {
      Module mod = (Module) iter.next();
      try {
        Class c = mod.loadTestClass(className);
        return c;
      } catch (ClassNotFoundException cnfe) {
        //$JL-EXC$ try next loader
      }
    }
    // should not be reached 
    throw new ClassNotFoundException(className);
  }

  public TestDescriptionSet getTestDescriptionSet()
    throws TestConflictException {
    TestTree tt = new TestTree();
    tt.setName("tests");
    TestDescriptionSet out = new TestDescriptionSet("global+local", 0, tt);
    Collection mods = modules.values();
    for (Iterator iter = mods.iterator(); iter.hasNext();) {
      Module mod = (Module) iter.next();
      TestSet[] testSets = mod.getTestSets();
      for (int i = 0; i < testSets.length; i++) {
        TestTree currentTree =
          testSets[i].getTestDescriptionSet().getTestTree();
        if ("".equals(currentTree.getName())) {
          currentTree.setName(testSets[i].getTestDescriptionSet().getName());
        }
        tt.addTree(currentTree);
      }
    }
    return out;
  }

  public Module[] getModules() {
    return (Module[]) modules.values().toArray(new Module[0]);
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.extension.TestProviderInterface#getTestObject(com.sap.tc.jtools.util.structures.StructureTree)
   */
  public TestObject getTestObject(StructureTree testObjTree)
    throws InvalidTestObjectSpecification {
    Collection mods = modules.values();
    for (Iterator iter = mods.iterator(); iter.hasNext();) {
      Module mod = (Module) iter.next();
      TestObject testObj = mod.getTestObject(testObjTree);
      if (testObj != null)
        return testObj;
    }
    throw new InvalidTestObjectSpecification("testobject does not exist");
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.extension.TestProviderInterface#getTestObject(com.sap.tc.jtools.util.structures.StructureTree)
   */
  public TestObjectPosition getTestObjectPosition(String testObjectType, StructureTree positionTree)
	throws InvalidTestObjectSpecification {
    Collection mods = modules.values();
	for (Iterator iter = mods.iterator(); iter.hasNext();) {
	  Module mod = (Module) iter.next();
	  TestObjectPosition testObjPosition = mod.getTestObjectPosition(testObjectType, positionTree);
	  if (testObjPosition != null)
		return testObjPosition;
	}
	throw new InvalidTestObjectSpecification("test object position does not exist");
  }
  
  public static boolean isInitialized() {
    return initialized;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.extension.TestProviderInterface#getModule(java.lang.String)
   */
  public Module getModule(String id) {
    return (Module) modules.get(id);
  }
}
