/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/extension/Module.java#2 $
 */

package com.sap.tc.jtools.jlint.extension;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestObjectSpecification;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLParseException;

/**
 * @author D037913
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class Module {

	private static final String MODULE_TAG = "module";
	private static final String ID_ATTR = "id";
	private static final String DEPENDS_ATTR = "depends";
	private static final String PATH_ATTR = "path";
	private static final String NAME_ATTR = "name";
	private static final String CLASS_ATTR = "class";
	private static final String LIB_TAG = "library";
	private static final String TESTSET_TAG = "testSet";
	private static final String TESTOBJECT_TYPE_TAG = "testObjectType";

	private String id;
	private ClassLoader moduleClassLoader;
	private List testSets = new ArrayList();
	private Map name2testObjectClassnameMap = new HashMap();

	public Module(
		StructureTree moduleTree,
		ClassLoader parent,
		Map id2ModuleMap)
		throws BadTreeException, IOException, XMLParseException {
		if (!MODULE_TAG.equals(moduleTree.getTag()))
			throw new BadTreeException();
		id = moduleTree.getParameter(ID_ATTR);
		// required modules
		String reqModIds = moduleTree.getParameter(DEPENDS_ATTR);
		List reqClassLoaders = new ArrayList();
		if (reqModIds != null) {
			StringTokenizer tok = new StringTokenizer(reqModIds, ",", false); //$NON-NLS-1$
			while (tok.hasMoreElements()) {
				Module mod = (Module) id2ModuleMap.get(tok.nextToken().trim());
				reqClassLoaders.add(mod.moduleClassLoader);
			}
		}
		// lib jars
		StructureTree[] libTrees = moduleTree.getChildren(LIB_TAG);
		List libs = new ArrayList();
		for (int i = 0; i < libTrees.length; i++) {
			libs.add(new File(libTrees[i].getParameter(PATH_ATTR)));
		}
		moduleClassLoader =
			new ModuleClassLoader(
				parent,
				(File[]) libs.toArray(new File[0]),
				(ClassLoader[]) reqClassLoaders.toArray(new ClassLoader[0]));

		// testobject types
		StructureTree[] testObjTypes =
			moduleTree.getChildren(TESTOBJECT_TYPE_TAG);
		for (int i = 0; i < testObjTypes.length; i++) {
			String name = testObjTypes[i].getParameter(NAME_ATTR);
			String className = testObjTypes[i].getParameter(CLASS_ATTR);
			name2testObjectClassnameMap.put(name, className);
		}
		// testSets
		StructureTree[] testSetTrees = moduleTree.getChildren(TESTSET_TAG);
		for (int i = 0; i < testSetTrees.length; i++) {
			testSets.add(new TestSet(testSetTrees[i], moduleClassLoader));
		}
	}

	protected Module(
		String id,
		ClassLoader moduleLoader,
		Map name2testObjClassName) {
		this.id = id;
		this.moduleClassLoader = moduleLoader;
		this.name2testObjectClassnameMap = name2testObjClassName;
	}

	public Class loadTestClass(String className)
		throws ClassNotFoundException {
		for (Iterator iter = testSets.iterator(); iter.hasNext();) {
			TestSet testSet = (TestSet) iter.next();
			try {
				Class c = testSet.getLoader().loadClass(className);
				return c;
			} catch (ClassNotFoundException cnfe) {
				//$JL-EXC$ try next loader 
			}
		}
		throw new ClassNotFoundException(className);
	}

	public TestSet[] getTestSets() {
		TestSet[] out = new TestSet[testSets.size()];
		for (int i = 0; i < out.length; i++) {
			out[i] = (TestSet) testSets.get(i);
		}
		return out;
	}

	public TestObject getTestObject(StructureTree tree)
		throws InvalidTestObjectSpecification {
		String name = tree.getParameter(ParameterInterface.PARAMETER_TYPE);
		Class c = getTestObjectClass(name);
		if (c == null) {
			return null;
		}
		Constructor constructor;
		try {
			constructor =
				c.getConstructor(
					new Class[] {
						moduleClassLoader.loadClass(
							"com.sap.tc.jtools.util.structures.StructureTree")});
			if (constructor == null) {
				throw new InvalidTestObjectSpecification(
					"missing constructor: " + c.getName());
			}
			return (TestObject) constructor.newInstance(new Object[] { tree });
		} catch (Exception e) {
			throw new InvalidTestObjectSpecification(e);
		}

	}

	public String getId() {
		return id;
	}

	public boolean hasTest(String testName) {
		for (Iterator iter = testSets.iterator(); iter.hasNext();) {
			TestSet testSet = (TestSet) iter.next();
			TestDescriptionInterface[] tests =
				testSet.getTestDescriptionSet().getTestTree().getAllLeaves();
			for (int j = 0; j < tests.length; j++) {
				if (testName.equals(tests[j].getName()))
					return true;
			}
		}
		return false;
	}

	public void addTestSet(TestSet testSet) {
		this.testSets.add(testSet);
	}

	/**
	 * @param testObjectType
	 * @param positionTree
	 * @return
	 */
	public TestObjectPosition getTestObjectPosition(
		String testObjectType,
		StructureTree positionTree)
		throws InvalidTestObjectSpecification {

		try {
			Class c = getTestObjectClass(testObjectType);
			if (c == null) {
				return null;
			}
			return (TestObjectPosition) c
				.getMethod(
					"getPosition",
					new Class[] {
						moduleClassLoader.loadClass(
							"com.sap.tc.jtools.util.structures.StructureTree")})
				.invoke(null, new Object[] { positionTree });

		} catch (Exception e) {
			throw new InvalidTestObjectSpecification(e);
		}
	}

	private Class getTestObjectClass(String testObjectType)
		throws InvalidTestObjectSpecification {
		String className =
			(String) name2testObjectClassnameMap.get(testObjectType);
		if (className == null)
			return null;
		Class c = null;
		try {
			c = moduleClassLoader.loadClass(className);
		} catch (ClassNotFoundException e) {
			return null;
		}
		if (c == null) {
			throw new InvalidTestObjectSpecification(
				"class " + className + " does not exist");
		}
		return c;
	}
}