/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/extension/JarClassLoader.java#2 $
 */

package com.sap.tc.jtools.jlint.extension;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

public class JarClassLoader extends ClassLoader {

  private static final String JAR_SUFFIX = ".jar";
  private static final String ZIP_SUFFIX = ".zip";
  private static final String CLASS_SUFFIX = ".class";

  private static final String FILE_URL_PREFIX = "file:";
  private static final String JAR_URL_PREFIX = "jar:file:";
  private static final String JAR_URL_SEPARATOR = "!/";

  private static final char SLASH = '/';
  private static final char DOT = '.';

  private static final int BUFFER_SIZE = 512;

  private List classDirs = new ArrayList();
  private List jarFiles = new ArrayList();
  private Map classes = new HashMap();
  private ByteArrayOutputStream baos = new ByteArrayOutputStream(BUFFER_SIZE);

  private class URLEnumeration implements Enumeration {
    private Iterator iterator;

    public URLEnumeration(URL[] urls) {
      if (urls == null)
        throw new IllegalArgumentException();
      iterator = Arrays.asList(urls).iterator();
    }

    public boolean hasMoreElements() {
      return iterator.hasNext();
    }

    public Object nextElement() {
      return iterator.next();
    }
  }

  public JarClassLoader(ClassLoader parent, File[] libs) throws IOException {
    super(parent);
    for (int i = 0; i < libs.length; i++) {
      if (libs[i].isDirectory()) {
        classDirs.add(libs[i]);
      } else if ((libs[i].isFile())) {
        String fileName = libs[i].getName().toLowerCase();
        if (fileName.endsWith(JAR_SUFFIX) || fileName.endsWith(ZIP_SUFFIX)) {
          jarFiles.add(new ZipFile(libs[i]));
        }
      }
    }
  }

  protected Class findClass(String name) throws ClassNotFoundException {
    // check if class is already cached
    Class c;
    c = (Class) classes.get(name);
    if (c != null) {
      return c;
    }
    try {
      byte[] f =
        getBytes(name.replace(DOT, File.separatorChar).concat(CLASS_SUFFIX));
      c = defineClass(name, f, 0, f.length);
      // store loaded class in cache
      classes.put(name, c);
    } catch (IOException ex) {
      c = super.findClass(name);
    }
    return c;
  }

  private byte[] getBytes(String fileName) throws IOException {
    if (fileName == null) {
      return null;
    }
    byte[] resultBytes = null;
    InputStream is = null;
    File file = null;
    File classDir = null;
    for (Iterator iter = classDirs.iterator(); iter.hasNext();) {
      classDir = (File) iter.next();
      file = new File(classDir, fileName);
      if (file.isFile()) {
        is = new FileInputStream(file);
        resultBytes = getEntryBytes(is);
        is.close();
        return resultBytes;
      }
    }
    ZipFile zip = null;
    ZipEntry entry = null;
    for (Iterator iter = jarFiles.iterator(); iter.hasNext();) {
      zip = (ZipFile) iter.next();
      entry = zip.getEntry(fileName.replace(File.separatorChar, SLASH));
      if (entry != null) {
        is = zip.getInputStream(entry);
        resultBytes = getEntryBytes(is);
        is.close();
        return resultBytes;
      }
    }
    throw new FileNotFoundException(fileName);
  }

  private byte[] getEntryBytes(InputStream in) throws IOException {
    byte[] buffer = new byte[BUFFER_SIZE];
    baos.reset();
    int length = 0;
    while ((length = in.read(buffer)) != -1) {
      baos.write(buffer, 0, length);
    }
    return baos.toByteArray();
  }

  protected URL findResource(String name) {
    URL result = super.findResource(name);
    if (result != null) {
      return result;
    }
    try {
      File classDir = null;
      File file = null;
      for (Iterator iter = classDirs.iterator(); iter.hasNext();) {
        classDir = (File) iter.next();
        file = new File(classDir, name);
        if (file.exists()) {
          return new URL(FILE_URL_PREFIX + file.getCanonicalPath());
        }
      }
      ZipFile zip = null;
      ZipEntry entry = null;
      for (Iterator iter = jarFiles.iterator(); iter.hasNext();) {
        zip = (ZipFile) iter.next();
        entry = zip.getEntry(name.replace(File.separatorChar, SLASH));
        if (entry != null) {
          return new URL(
            JAR_URL_PREFIX.concat(zip.getName()).concat(
              JAR_URL_SEPARATOR).concat(
              name));
        }
      }
    } catch (IOException e) {
      return null;
    }
    return null;
  }

  /* (non-Javadoc)
   * @see java.lang.ClassLoader#findResources(java.lang.String)
   */
  protected Enumeration findResources(String name) throws IOException {
    URL url = findResource(name);
    if (url != null) {
      return new URLEnumeration(new URL[] { url });
    } else {
      return new URLEnumeration(new URL[0]);
    }
  }

}
