/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/TestManager.java#4 $
 */

package com.sap.tc.jtools.jlint;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import com.sap.tc.jtools.jlint.exceptions.ExecutionException;
import com.sap.tc.jtools.jlint.exceptions.InvalidDataProviderException;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.TestTree;
import com.sap.tc.jtools.jtci.cache.CacheEntry;
import com.sap.tc.jtools.jtci.exceptions.CancelJobException;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestSpecificationException;
import com.sap.tc.jtools.jtci.exceptions.PerformException;
import com.sap.tc.jtools.jtci.interfaces.ITestMessageDescription;
import com.sap.tc.jtools.jtci.interfaces.JtciDataInterface;
import com.sap.tc.jtools.jtci.interfaces.Listener;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.cache.CacheDependency;
import com.sap.tc.jtools.util.cache.CacheInterface;

/**
 * represents a test manager.
 * 
 * The test manager:
 */
public class TestManager {

  /**
   * name of the default configuration file (in the directory com/sap/tc/jtools/jlint)
   */
  public static final String DEFAULT_CONFIG_FILE = "tool_descriptor.xml"; //$NON-NLS-1$

  private static TestDescriptionSet toolDescriptor;
  private TestVariant[] testVariants = new TestVariant[0];
  private DataEntryInterface[] data = new DataEntry[0];
  private ProcessorEntry[] processors = new ProcessorEntry[0];
  private Listener listener = null;

  private CacheInterface cache;

  public TestManager(TestDescriptionSet toolDescriptor, CacheInterface cache) {
    TestManager.toolDescriptor = toolDescriptor;
    this.cache = cache;
  }

  /**
   * returns the tool descriptor
   * 
   * @return tool descriptor 
   */
  public TestDescriptionSet getTestDescriptionSet() {
    return toolDescriptor;
  }

  /**
   * returns the tool version
   * 
   * @return tool version 
   */
  public int getVersion() {
    return toolDescriptor.getVersion();
  }

  /**
   * returns a hierarchical view of the available tests
   * 
   * @return available tests
   */
  public TestTree getAllTests() {
    return toolDescriptor.getTestTree();
  }

  /**
   * resets the tool manager (to be called before each new test job)
   * 
   */
  public void reset() {
    testVariants = new TestVariant[0];
    data = new DataEntry[0];
    processors = new ProcessorEntry[0];
  }

  /**
   * adds a new fully qualified test (test name + test parameters + test object) 
   * to the current job
   * 
   * @param test test name
   * @param parameters test parameters
   * @param testObject object to be tested
   * @throws InvalidTestSpecificationException if the test does not exist, 
   * the parameters are not valid, or the test object is not valid
   * 
   */

  public void addTest(
    String test,
    ParameterInterface[] parameters,
    ITestMessageDescription[] messages,
    com.sap.tc.jtools.jtci.TestObject testObject)
    throws InvalidTestSpecificationException {
    //construct variant
    TestVariant testVariant = null;
    TestDescriptionInterface[] tests = getAllTests().getAllLeaves();
    boolean done = false;
    for (int i = 0; i < tests.length && !done; i++) {
      if (tests[i].getName().equals(test)) {
        try {
          testVariant =
            new TestVariant((TestDefinition) tests[i], parameters, messages, testObject);
        } catch (Exception e) {
          throw new InvalidTestSpecificationException(e);
        }
        done = true;
      }
    }
    TestVariant[] tempArray = new TestVariant[testVariants.length + 1];
    for (int i = 0; i < testVariants.length; i++)
      tempArray[i] = testVariants[i];
    try {
      tempArray[testVariants.length] = testVariant;
      // get test processor   !!new!!
      TestComponentInterface processorUnit =
        ProcessorUnitFactory.createTestComponent(
          testVariant.getTestDefinition());
      processorUnit.setMessages(testVariant.getMessages());
      processorUnit.setParameters(
        testVariant.getParameters(),
        testVariant.getTestObject());
      ProcessorInterface newTestProcessor = null;
      RequirementInterface requiredProcessor =
        processorUnit.getProcessorDefinition();
      if (requiredProcessor == null) {
        newTestProcessor = (ProcessorInterface) processorUnit;
        addProcessor(
          newTestProcessor,
          newTestProcessor instanceof TestProcessorInterface);
      } else {
        RequirementInterface processorDefinition =
          processorUnit.getProcessorDefinition();
        addProcessor(processorDefinition).getProcessor().addProcessorUnit(
          processorUnit,
          processorUnit.getRole());
      }

    } catch (InvalidDataProviderException e) {
      throw new InvalidTestSpecificationException(e);
    }
    testVariants = tempArray;
  }

  /**
   * registers external data that can be used by the tests to short-circuit some parts of 
   * the test execution
   * 
   * @param externalData external data
   * 
   */
  public void addExternalData(JtciDataInterface[] externalData) {
    Vector totalData = new Vector();

    for (int i = 0; i < externalData.length; i++) {
      totalData.add(
        new ExternalDataEntry(
          externalData[i].getName(),
          externalData[i].getParameters(),
          externalData[i].getData()));
    }

    for (int i = 0; i < data.length; i++) {
      boolean found = false;
      for (int j = 0; j < externalData.length && !found; j++) {
        if (data[i].getName().equals(externalData[j].getName())
          && ParameterTool.eq(
            data[i].getParameters(),
            externalData[j].getParameters()))
          found = true;
      }
      if (!found)
        totalData.add(data[i]);
    }
    data =
      (DataEntryInterface[]) totalData.toArray(
        new DataEntryInterface[totalData.size()]);
  }

  /**
   * executes the current job and returns its results
   * 
   * @return job results
   */
  public Result[] performTests() throws PerformException {
    // loop over tests, collect all needed processors and data

    boolean somethingWasDone = true;
    B : while (somethingWasDone) {
      somethingWasDone = false;
      A : for (int i = 0; i < data.length; i++) {
        if (listener != null) {
          if (listener.isCanceled()) {
            return new Result[0];
          }
        }
        if (data[i].getStatus() == DataEntryInterface.COMPLETED)
          continue A;
        if (cache != null) {
          Object cachedEntry = cache.getEntryValue(computeKey(data[i]));
          if (cachedEntry != null) {
            data[i] =
              new CachedDataEntry(
                data[i].getName(),
                data[i].getParameters(),
                cachedEntry);
            continue A;
          }
        }

        // it must be computed -> we can safely cast to DataEntry

        DataEntry dataEntry = (DataEntry) data[i];
        ProcessorInterface currentProcessor = dataEntry.getProcessor();

        //if ready, add to Processor Array
        if (readyToStart(currentProcessor)) {
          RequirementInterface[] requirements = dataEntry.getRequirements();
          Object[] values = new Object[requirements.length];
          for (int j = 0; j < values.length; j++) {
            if (!isExternalFile(requirements[j])) {
              if (listener != null && listener.isCanceled()) {
                return new Result[0];
              }
              values[j] = getDataEntry(requirements[j]).getData();
            }
          }
          currentProcessor.setPrerequisites(values);
          try {
            currentProcessor.execute();
            dataEntry.setStatus(DataEntryInterface.COMPLETED);
            if (cache != null) {
              List dependencies = new ArrayList();
              for (int j = 0; j < requirements.length; j++) {
                boolean isFile = isExternalFile(requirements[j]);
                String key;
                if (isFile) {
                  key = requirements[j].getParameters()[0].valueToString();

                } else {
                  key = computeKey(requirements[j]);

                }
                dependencies.add(new CacheDependency(key, isFile));

              }

              CacheEntry newEntry =
                new CacheEntry(
                  computeKey(data[i]),
                  data[i].getData(),
                  (CacheDependency[]) dependencies.toArray(
                    new CacheDependency[dependencies.size()]),
                  System.currentTimeMillis());
              cache.addEntry(newEntry);
            }
            somethingWasDone = true;
          } catch (CancelJobException cje) {
            return new Result[0];
          } catch (Exception e) {
            ((DataEntry) data[i]).setStatus(DataEntryInterface.PROBLEM);
          }
        }
      }

    }
    List jobResult = new ArrayList();

    for (int i = 0; i < processors.length; i++) {
      if (processors[i].isTest()) {
        TestProcessorInterface processor =
          (TestProcessorInterface) processors[i].getProcessor();
        processor.setListener(listener);
        RequirementInterface[] requirements = processor.getRequirements();
        Object[] values = new Object[requirements.length];
        boolean missingData = false;
        RequirementInterface missingReqirement = null;
        for (int j = 0; j < values.length && !missingData; j++) {
          if (!requirements[j]
            .getClassName()
            .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE))
            values[j] = getDataEntry(requirements[j]).getData();
          if (values[j] == null)
            missingData = true;
          missingReqirement = requirements[j];
        }

        if (missingData) {
          TestObject testObject = processor.getTestObject();
          Properties p = new Properties();
          p.setProperty(
            ParameterTool.PAR_MSG_KEY,
            Result.ERROR_MESSAGE_MISSING_DATA);
          if (missingReqirement != null) {
            p.setProperty(
              "REQUIRED_CLASSNAME",
              missingReqirement.getClassName());
            p.setProperty(
              "PROCESSOR_NAME",
              missingReqirement.getProcessor().getName());
            ParameterInterface[] params = missingReqirement.getParameters();
            for (int j = 0; j < params.length; j++) {
              p.setProperty("PARAMETER_" + j, params[j].toString());
            }
          }
          jobResult.add(
            new Result(
              processors[i].getName(),
              testObject.getType(),
              testObject.getID(),
              null,
              Result.SEVERITY_INTERNAL_ERROR,
              Result.ERROR_MESSAGE_MISSING_DATA,
              p));
        } else {
          processor.setPrerequisites(values);
          try {
            processor.execute();
          } catch (CancelJobException cje) {
            TestObject testObject = processor.getTestObject();
            jobResult.add(
              new Result(
                processors[i].getName(),
                testObject.getType(),
                testObject.getID(),
                null,
                Result.SEVERITY_INTERNAL_ERROR,
                Result.ERROR_MESSAGE_JOB_CANCELED,
                null));
            break; // exit for loop
          } catch (Exception e) {
            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw));
            Properties parameters = new Properties();
            parameters.setProperty(ParameterTool.PAR_EXCEPTION, sw.toString());
            parameters.setProperty(
              ParameterTool.PAR_MSG_KEY,
              Result.ERROR_MESSAGE_COULDNT_PERFORM);
            TestObject testObject = processor.getTestObject();
            jobResult.add(
              new Result(
                processors[i].getName(),
                testObject.getType(),
                testObject.getID(),
                null,
                Result.SEVERITY_INTERNAL_ERROR,
                Result.ERROR_MESSAGE_COULDNT_PERFORM,
                parameters));
          }

          ResultInterface[] results = processor.getErrors();
          for (int k = 0; k < results.length; k++)
            jobResult.add(results[k]);
        }
      }
    }

    testVariants = new TestVariant[0];
    // todo: reset everything else

    return (Result[]) jobResult.toArray(new Result[jobResult.size()]);
  } // performTests

  /**
   * @param interface1
   * @return
   */
  private String computeKey(RequirementInterface data) {
    String key = data.getClassName() + "@";
    ParameterInterface[] params = data.getParameters();
    for (int i = 0; i < params.length; i++) {
      key += params[i].valueToString().hashCode() + ";";
    }
    return key;
  }

  /**
   * Sets listener
   *
   */
  public void setListener(Listener l) {
    listener = l;
  }

  private boolean readyToStart(Requirement req) {
    DataEntryInterface dataEntry = getDataEntry(req);
    if (dataEntry == null
      || dataEntry.getStatus() != DataEntryInterface.INITIAL)
      return false;
    // status == initial -> we can safely cast
    RequirementInterface[] reqs = ((DataEntry) dataEntry).getRequirements();
    for (int i = 0; i < reqs.length; i++) {
      if (!reqs[i]
        .getClassName()
        .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE)
        && !getDataEntry(reqs[i]).isCompleted())
        return false;
    }
    return true;
  }

  private boolean readyToStart(String name, ParameterInterface[] parameters) {
    DataEntryInterface dataEntry = getDataEntry(name, parameters);
    if (dataEntry == null
      || dataEntry.getStatus() != DataEntryInterface.INITIAL)
      return false;
    // status == initial -> we can safely cast
    RequirementInterface[] reqs = ((DataEntry) dataEntry).getRequirements();
    for (int i = 0; i < reqs.length; i++) {
      if (!reqs[i]
        .getClassName()
        .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE)
        && !getDataEntry(reqs[i]).isCompleted())
        return false;
    }
    return true;
  }

  private boolean readyToStart(ProcessorInterface processor) {
    RequirementInterface[] reqs = processor.getRequirements();
    for (int i = 0; i < reqs.length; i++) {
      if (!reqs[i]
        .getClassName()
        .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE)
        && !getDataEntry(reqs[i]).isCompleted())
        return false;
    }
    return true;
  }

  private DataEntryInterface getDataEntry(RequirementInterface req) {
    for (int i = 0; i < data.length; i++) {
      if (data[i].getName().equals(req.getClassName())
        && ParameterTool.eq(data[i].getParameters(), req.getParameters()))
        return data[i];
    }
    return null;
  }

  private DataEntryInterface getDataEntry(
    String name,
    ParameterInterface[] parameters) {
    for (int i = 0; i < data.length; i++) {
      if (data[i].getName().equals(name)
        && ParameterTool.eq(parameters, data[i].getParameters()))
        return data[i];
    }
    return null;
  }

  private ProcessorEntry getTestProcessor(RequirementInterface processor) {
    for (int i = 0; i < processors.length; i++) {
      if (processor.getClassName().equals(processors[i].getName())
        && ParameterTool.eq(
          processor.getParameters(),
          processors[i].getProcessor().getParameters()))
        return processors[i];
    }
    return null;
  }

  private ProcessorEntry addProcessor(
    ProcessorInterface newProcessor,
    boolean isTest) throws InvalidDataProviderException {

    for (int i = 0; i < processors.length; i++) {
      // check if already there
      if (newProcessor.getName().equals(processors[i].getName())
        && newProcessor.getParameters().length
          == processors[i].getProcessor().getParameters().length) {
        boolean sameParameters = true;
        for (int j = 0;
          j < newProcessor.getParameters().length && sameParameters;
          j++) {
          ParameterInterface par1 = newProcessor.getParameters()[j];
          ParameterInterface par2 =
            processors[i].getProcessor().getParameters()[j];
          if (!par1.getName().equals(par2.getName())
            || !par1.getValue().equals(par2.getValue()))
            sameParameters = false;
        }
        if (sameParameters) {
          if (isTest)
            processors[i].setIsTest(true);
          return processors[i];
        }
      }

    }
    ProcessorEntry newEntry = new ProcessorEntry(newProcessor);
    if (isTest)
      newEntry.setIsTest(true);
    ProcessorEntry[] tempArray = new ProcessorEntry[processors.length + 1];
    for (int i = 0; i < processors.length; i++)
      tempArray[i] = processors[i];
    tempArray[processors.length] = newEntry;
    processors = tempArray;
    RequirementInterface[] reqs = newEntry.getProcessor().getRequirements();
    for (int j = 0; j < reqs.length; j++) {
      if (!reqs[j]
        .getClassName()
        .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE))
        try {
          addData(reqs[j]);
        } catch (Exception e) {
          throw new InvalidDataProviderException(e);
        }
    }
    return newEntry;
  }

  private ProcessorEntry addProcessor(RequirementInterface processorDefinition) throws InvalidDataProviderException {

    for (int i = 0; i < processors.length; i++) {
      // check if already there
      if (processorDefinition.getClassName().equals(processors[i].getName())
        && processorDefinition.getParameters().length
          == processors[i].getProcessor().getParameters().length) {
        boolean sameParameters = true;
        for (int j = 0;
          j < processorDefinition.getParameters().length && sameParameters;
          j++) {
          ParameterInterface par1 = processorDefinition.getParameters()[j];
          ParameterInterface par2 =
            processors[i].getProcessor().getParameters()[j];
          if (!par1.getName().equals(par2.getName())
            || !par1.getValue().equals(par2.getValue()))
            sameParameters = false;
        }
        if (sameParameters) {
          return processors[i];
        }
      }

    }
    ProcessorEntry newEntry =
      new ProcessorEntry(processorDefinition.getProcessor());
    if (processorDefinition.getProcessor() instanceof TestProcessorInterface)
      newEntry.setIsTest(true);
    ProcessorEntry[] tempArray = new ProcessorEntry[processors.length + 1];
    for (int i = 0; i < processors.length; i++)
      tempArray[i] = processors[i];
    tempArray[processors.length] = newEntry;
    processors = tempArray;
    RequirementInterface[] reqs = newEntry.getProcessor().getRequirements();
    for (int j = 0; j < reqs.length; j++) {
      if (!reqs[j]
        .getClassName()
        .equals(RequirementInterface.REQUIREMENT_EXTERNAL_FILE))
        try {
          addData(reqs[j]);
        } catch (Exception e) {
          throw new InvalidDataProviderException(e);
        }
    }
    return newEntry;
  }

  private void addData(RequirementInterface newData)
    throws InvalidDataProviderException {

    for (int i = 0; i < data.length; i++) {
      if (newData.getClassName().equals(data[i].getName())
        && ParameterTool.eq(data[i].getParameters(), newData.getParameters()))
        return;
    }

    DataEntryInterface[] tempArray = new DataEntryInterface[data.length + 1];
    for (int i = 0; i < data.length; i++)
      tempArray[i] = data[i];

    try {
      data = tempArray;
      DataEntry newEntry = new DataEntry(newData);
      data[data.length - 1] = newEntry;

      ProcessorInterface nextProcessor = newEntry.getProcessor();
      if (nextProcessor != null)
        addProcessor(nextProcessor, false);
      return;
    } catch (Exception e) {
      throw new InvalidDataProviderException(e);
    }

  }

  private interface DataEntryInterface extends JtciDataInterface {
    public int INITIAL = 0;
    public int READY = 1;
    public int STARTED = 2;
    public int COMPLETED = 3;
    public int PROBLEM = 4;
    public int getStatus();
    public boolean isCompleted();
  }

  private class ExternalDataEntry implements DataEntryInterface {
    private String name;
    private ParameterInterface[] parameters;
    private Object value;

    public ExternalDataEntry(
      String name,
      ParameterInterface[] parameters,
      Object value) {
      this.name = name;
      this.parameters = parameters;
      this.value = value;
    }
    public int getStatus() {
      return COMPLETED;
    }
    public String getName() {
      return name;
    }
    public ParameterInterface[] getParameters() {
      return parameters;
    }
    public boolean isCompleted() {
      return true;
    }
    public Object getData() {
      return value;
    }
  }

  private class CachedDataEntry implements DataEntryInterface {
    private String name;
    private ParameterInterface[] parameters;
    private Object value;

    public CachedDataEntry(
      String name,
      ParameterInterface[] parameters,
      Object value) {
      this.name = name;
      this.parameters = parameters;
      this.value = value;
    }
    public int getStatus() {
      return COMPLETED;
    }
    public String getName() {
      return name;
    }
    public ParameterInterface[] getParameters() {
      return parameters;
    }
    public boolean isCompleted() {
      return true;
    }
    public Object getData() {
      return value;
    }
  }

  private class DataEntry implements DataEntryInterface {

    private RequirementInterface data;
    private int status = INITIAL;

    public DataEntry(RequirementInterface req) {
      data = req;
    }

    public int getStatus() {
      return status;
    }

    public void setStatus(int status) {
      this.status = status;
    }

    public ProcessorInterface getProcessor() {
      return data.getProcessor();
    }

    public boolean isCompleted() {
      return (status == COMPLETED);
    }

    public RequirementInterface[] getRequirements() {
      RequirementInterface[] reqs = data.getProcessor().getRequirements();
      if (reqs.length == 0)
        return new Requirement[0];
      return reqs;
    }

    public ResultInterface[] getResults() {
      if (status == COMPLETED)
        return (ResultInterface[]) data.getProcessor().getData();
      return null;
    }

    public String getName() {
      return data.getClassName();
    }

    public ParameterInterface[] getParameters() {
      return data.getParameters();
    }
    public Object getData() {
      return data.getProcessor().getData();
    }

  }

  private class ProcessorEntry {
    private ProcessorInterface processor;
    private boolean isTestProcessor = false;

    public ProcessorEntry(ProcessorInterface processor) {
      this.processor = processor;
    }

    public ProcessorInterface getProcessor() {
      return processor;
    }

    public String getName() {
      return processor.getName();
    }

    public void execute() throws ExecutionException {
      processor.execute();
    }

    public void setIsTest(boolean isTest) {
      this.isTestProcessor = isTest;
    }

    public boolean isTest() {
      return isTestProcessor;
    }
  }

  private String computeKey(DataEntryInterface data) {
    String key = data.getName() + "@";
    ParameterInterface[] params = data.getParameters();
    for (int i = 0; i < params.length; i++) {
      key += params[i].valueToString().hashCode() + ";";
    }
    return key;
  }

  private boolean isExternalFile(RequirementInterface req) {
    return req.getClassName().equals(
      RequirementInterface.REQUIREMENT_EXTERNAL_FILE);
  }
}
