/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/TestDefinition.java#3 $
 */

package com.sap.tc.jtools.jlint;

import java.util.ArrayList;
import java.util.List;

import com.sap.tc.jtools.util.structures.*;
import com.sap.tc.jtools.jtci.interfaces.*;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestMessageDescription;
import com.sap.tc.jtools.jtci.exceptions.*;

/**
 * each instance represents a test template, that in general has to
 * be supplemented by parameters.
 * 
 * @see TestDescriptionInterface
 *
 *
 */

public class TestDefinition implements TestDescriptionInterface {

  private static final String TEST_RESPONSIBLE = "RESPONSIBLE"; //$NON-NLS-1$
  private static final String TEST_DESCRIPTION = "DESCRIPTION"; //$NON-NLS-1$
  private static final String TEST_SIGNATURE = "SIGNATURE"; //$NON-NLS-1$
  private static final String TEST_INTERNAL_PARAMETERS = "INTERNAL_PARAMETERS"; //$NON-NLS-1$
  private static final String TEST_TEST_OBJECT_TYPE = "TEST_OBJECT_TYPE"; //$NON-NLS-1$ 
  private static final String TEST_LABEL = "LABEL"; //$NON-NLS-1$
  private static final String TEST_VERSION = "VERSION"; //$NON-NLS-1$
  public static final String PARAMETER_TEST_CLASS = "TEST_CLASS"; //$NON-NLS-1$
  public static final String PARAMETER_RESOURCE_BUNDLE = "RESOURCE_BUNDLE"; //$NON-NLS-1$

  private ParameterInterface[] internalParameters;
  private String objectType;
  private String name;
  private String description;
  private String testResponsible;
  private String version;
  private String testClass;
  private String resourceBundle;
  private ParameterInterface[] inputParameters;
  private ITestMessageDescription[] messages;

  public TestDefinition(
    String name,
    String description,
    String testResponsible,
    String resourceBundle,
    String version,
    ParameterInterface[] internalParameters,
    String objectType,
    ParameterInterface[] inputParameters,
    ITestMessageDescription[] messages)
    throws BadTreeException {

    this.name = name;
    this.description = description;
    this.testResponsible = testResponsible;
    this.resourceBundle = resourceBundle;
    this.version = version;
    this.objectType = objectType;
    this.messages = messages;
    if (inputParameters == null)
      this.inputParameters = new ParameterInterface[0];
    else {
      this.inputParameters = inputParameters;
    }

    try {
      this.internalParameters = internalParameters;
      for (int i = 0; i < internalParameters.length; i++) {
        if (internalParameters[i].getName().equals(PARAMETER_TEST_CLASS)) {
          this.testClass = (String) internalParameters[i].getValue();
        }
      }
    } catch (Exception e) {
      throw new BadTreeException(e);
    }

  }

  public TestDefinition(StructureTree tree) throws BadTreeException {
    name = tree.getParameter(TEST_LABEL);
    resourceBundle = tree.getParameter(PARAMETER_RESOURCE_BUNDLE);
    testResponsible = tree.getParameter(TEST_RESPONSIBLE);
    description = tree.getOnlyChild(TEST_DESCRIPTION).getText();
    version = tree.getParameter(TEST_VERSION);
    if (tree.getChildren(TEST_INTERNAL_PARAMETERS).length > 0) {
      StructureTree parTree = tree.getOnlyChild(TEST_INTERNAL_PARAMETERS);
      StructureTree[] internalParams = parTree.getChildren();
      internalParameters = new ParameterInterface[internalParams.length];
      for (int i = 0; i < internalParameters.length; i++) {
        internalParameters[i] =
          ParameterTool.createParameter(
            internalParams[i].getTag(),
            ParameterTool.PAR_TYPE_STRING,
            internalParams[i].getText());
        if (internalParameters[i].getName().equals(PARAMETER_TEST_CLASS)) {
          this.testClass = (String) internalParameters[i].getValue();
        }
      }
    }
    objectType = tree.getOnlyChild(TEST_TEST_OBJECT_TYPE).getText();
    try {
      StructureTree[] parameterTrees =
        tree.getOnlyChild(TEST_SIGNATURE).getChildren(
          ParameterInterface.INPUT_PARAMETER_TAG);
      ParameterInterface[] out = new ParameterInterface[parameterTrees.length];
      for (int i = 0; i < out.length; i++)
        out[i] = ParameterTool.createParameter(parameterTrees[i]);
      inputParameters = out;
    } catch (Exception e) {
      inputParameters = new ParameterInterface[0];
    }
    StructureTree msgs =
      tree.getOnlyChild(ITestMessageDescription.TAG_MESSAGES);
    if (msgs != null) {
      StructureTree[] msgTrees =
        msgs.getChildren(ITestMessageDescription.TAG_MESSAGE);
      List msgList = new ArrayList(msgTrees.length);
      for (int i = 0; i < msgTrees.length; i++) {
        msgList.add(TestMessageDescription.fromStructureTree(msgTrees[i]));
      }
      this.messages =
        (ITestMessageDescription[]) msgList.toArray(
          new ITestMessageDescription[0]);
    } else {
      this.messages = new ITestMessageDescription[0];
    }
  }

  public StructureTree toStructureTree() {
    Header header = new Header(PARAMETER_TAG_TEST);
    header.setParameter(TEST_LABEL, getName());
    if (resourceBundle != null) {
      header.setParameter(PARAMETER_RESOURCE_BUNDLE, resourceBundle);
    }
    header.setParameter(TEST_RESPONSIBLE, getTestResponsible());
    header.setParameter(TEST_VERSION, getVersion());
    StructureTree tree = new StructureTree(header);
    StructureTree descriptionTree =
      new StructureTree(new Header(TEST_DESCRIPTION));
    descriptionTree.setText(getDescription());
    tree.addChild(descriptionTree);
    //internal parameters
    StructureTree testObjectTree =
      new StructureTree(new Header(TEST_TEST_OBJECT_TYPE));
    testObjectTree.setText(objectType);
    tree.addChild(testObjectTree);

    tree.addChild(getInternalParameterTree());
    StructureTree signatureTree = new StructureTree(new Header(TEST_SIGNATURE));

    ParameterInterface[] inputParameters = getInputParameters();
    for (int i = 0; i < inputParameters.length; i++) {
      signatureTree.addChild(inputParameters[i].toStructureTree());
    }
    tree.addChild(signatureTree);
    StructureTree msgsTree =
      new StructureTree(new Header(ITestMessageDescription.TAG_MESSAGES));
    for (int i = 0; i < messages.length; i++) {
      msgsTree.addChild(messages[i].toStructureTree());
    }
    tree.addChild(msgsTree);
    return tree;
  }

  public String getVersion() {
    return version;
  }

  public String getTestResponsible() {
    return testResponsible;
  }

  public String getDescription() {
    return description;
  }

  public String getName() {
    return name;
  }

  public String getObjectType() {
    return objectType;
  }

  public String getTestClass() {
    return testClass;
  }

  public String getResourceBundle() {
    return resourceBundle;
  }

  public ParameterInterface[] getInternalParameters() {
    return internalParameters;
  }

  private StructureTree getInternalParameterTree() {
    StructureTree out = new StructureTree(new Header(TEST_INTERNAL_PARAMETERS));
    for (int i = 0; i < internalParameters.length; i++) {
      if (internalParameters[i]
        .getType()
        .equals(ParameterTool.PAR_TYPE_STRING)) {
        StructureTree child =
          new StructureTree(new Header(internalParameters[i].getName()));
        child.setText((String) internalParameters[i].getValue());
        out.addChild(child);
      } else if (
        internalParameters[i].getType().equals(
          ParameterTool.PAR_TYPE_STRING_ARRAY)) {

        String[] value = (String[]) internalParameters[i].getValue();
        Header paramHeader = new Header(TEST_INTERNAL_PARAMETERS);
        paramHeader.setParameter(ParameterInterface.PARAMETER_NAME, name);
        paramHeader.setParameter(
          ParameterInterface.PARAMETER_TYPE,
          internalParameters[i].getType());
        StructureTree paramTree = new StructureTree(paramHeader);

        if (null != value && 0 != value.length) {
          Header contentHeader = new Header(ParameterInterface.PARAMETER_VALUE);
          StructureTree contentTree = new StructureTree(contentHeader);
          paramTree.addNewChild(contentTree);

          for (int ii = 0; ii < value.length; ii++) {
            if (null != value[ii]) {
              StructureTree itemTree = new StructureTree(new Header("ITEM"));
              itemTree.setText(value[i]);
              contentTree.addChild(itemTree);
            }
          }
        }
        return paramTree;
      } else if (
        internalParameters[i].getType().equals(
          ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {

        StructureTree value = (StructureTree) internalParameters[i].getValue();
        Header paramHeader = new Header(TEST_INTERNAL_PARAMETERS);
        paramHeader.setParameter(ParameterInterface.PARAMETER_NAME, name);
        paramHeader.setParameter(
          ParameterInterface.PARAMETER_TYPE,
          ParameterTool.PAR_TYPE_DEEP_STRUCTURE);
        StructureTree paramTree = new StructureTree(paramHeader);
        if (null != value) {
          Header contentHeader = new Header(ParameterInterface.PARAMETER_VALUE);
          StructureTree contentTree = new StructureTree(contentHeader);
          paramTree.addNewChild(contentTree);
          contentTree.addNewChild(value);
        }
        return paramTree;
      }

    }
    return out;
  }

  public ParameterInterface getInternalParameter(String parameterName) {
    for (int i = 0; i < internalParameters.length; i++) {
      if (internalParameters[i].getName().equals(parameterName)) {
        return internalParameters[i];
      }
    }
    return null;
  }

  public ParameterInterface getInputParameter(String parameterName) {
    for (int i = 0; i < inputParameters.length; i++) {
      if (inputParameters[i].getName().equals(parameterName)) {
        return inputParameters[i];
      }
    }
    return null;
  }

  public ParameterInterface[] getInputParameters() {
    return inputParameters;
  }

  public int getNodeType() {
    return TEST;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface#getMessages()
   */
  public ITestMessageDescription[] getMessages() {
    return messages;
  }
}