/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_core/src/com/sap/tc/jtools/jlint/CollectResults.java#3 $
 */

package com.sap.tc.jtools.jlint;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import com.sap.tc.jtools.jlint.extension.TestProvider;
import com.sap.tc.jtools.jlint.extension.TestProviderInterface;
import com.sap.tc.jtools.jtci.Dispatcher;
import com.sap.tc.jtools.jtci.PerformResponse;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.exceptions.NestableRuntimeException;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * This class can be used to process "raw" JLin results. The meaning of "process"
 * is test-dependent.
 *  
 * Copyright (c) 2003, SAP AG
 */
public class CollectResults {

  private Dispatcher dispatcher = null;
  private List rawResults = new ArrayList();
  private Properties properties = new Properties();
  private PerformResponse collectedResponse;

  public CollectResults(String jlinHomeDir)
    throws BadTreeException, FileNotFoundException, IOException, XMLParseException {
    TestProviderInterface provider = null;
    if (TestProvider.isInitialized()) {
      provider = TestProvider.getTestProvider();
    } else {
      provider =
        TestProvider.initialize(new File(jlinHomeDir, "/config/config.xml"));
    }
    dispatcher = new Dispatcher(provider);
  }

  private FilenameFilter indexFilter = new FilenameFilter() {
    public boolean accept(File dir, String name) {
      return "index.xml".equals(name);
    }
  };
  private FileFilter resultFilter = new FileFilter() {
    public boolean accept(File file) {
      return file.isDirectory()
        || (file.getName().startsWith("JLResults")
          && file.getName().endsWith(".xml"));
    }
  };

  public void setProperties(Properties properties) {
    this.properties = properties;
  }

  /**
   * processes the raw results 
   * 
   * @param resDir result directory to traverse for raw results
   * @param cleanUp if <code>true</code>, the files containing raw results are deleted
   * after processing
   * @return maximum severity level encountered, or -1 if an error occurred
   */
  public int execute(String resDir, boolean cleanUp) {
    try {
      return execute(resDir, new File(resDir, "index.xml"), cleanUp);
    } catch (IOException e) {
      e.printStackTrace();
      return -1;
    } catch (IllegalArgumentException iae) {
      iae.printStackTrace();
      return -1;
    }
  }
  
  /**
   * processes the raw results 
   * 
   * @param resDir result directory to traverse for raw results
   * @param resultFile file the collected results will be written into
   * @param cleanUp if <code>true</code>, the files containing raw results are deleted
   * after processing
   * @return maximum severity level encountered, or -1 if an error occurred
   */
  public int execute(String resDir, File resultFile, boolean cleanUp) throws IOException {
    int severityLevel = 0;
    rawResults.clear();
    File currDir = new File(resDir);
    if (!currDir.isDirectory()) {
      throw new IllegalArgumentException(resDir + " is not a directory!");
    }
    collectedResponse = new PerformResponse(currDir.getAbsolutePath());
    collectedResponse.setProperties(properties);
    addResultsFromDirectory(currDir);
    ResultInterface[] processedResults =
      dispatcher.processResults(
        (ResultInterface[]) rawResults.toArray(new ResultInterface[0]));
    for (int i = 0; i < processedResults.length; i++) {
      severityLevel = combine(severityLevel, processedResults[i].getPriority());
      collectedResponse.addResult(processedResults[i]);
    }

    Writer fileWriter = null;
    try {
      fileWriter = new BufferedWriter(new FileWriter(resultFile));
      XMLTool.writeDocument(collectedResponse.toStructureTree(), fileWriter);
    } finally {
      if (fileWriter != null) {
        fileWriter.close();
      }
    }

    if (cleanUp) {
      cleanUpDirectory(currDir);
    }
    properties.clear();
    return severityLevel;
  }
  
  public PerformResponse getCollectedResponse() {
    return collectedResponse;
  }

  private void addResultsFromDirectory(File dir) {
    File[] indexArray = dir.listFiles(indexFilter);
    if (indexArray.length == 1) {
      // get results from index.xml and return 
      addResultsFromFile(indexArray[0]);
      return;
    } else {
      // add results from all files and subdirectories
      File[] children = dir.listFiles(resultFilter);
      for (int i = 0; i < children.length; i++) {
        if (children[i].isDirectory())
          addResultsFromDirectory(children[i]);
        else
          addResultsFromFile(children[i]);
      }
    }
  }

  private void addResultsFromFile(File file) {
    String fileName = null;
    Reader fileReader = null;
    try {
      fileName = file.getCanonicalPath();
      fileReader = new BufferedReader(new FileReader(file));
      PerformResponse pr =
        new PerformResponse(XMLTool.parseReader(fileReader));
      rawResults.addAll(Arrays.asList(pr.getResults()));
    } catch (BadTreeException e) {
      throw new NestableRuntimeException("JLin error: badly formed file: " + fileName, e);
    } catch (IOException e) {
      throw new NestableRuntimeException("JLin error: IO Exception: " + fileName, e);
    } catch (XMLParseException e) {
      throw new NestableRuntimeException("JLin error: XML Exception: " + fileName, e);
    } finally {
      if (fileReader != null) {
        try {
          fileReader.close();
        } catch (IOException e1) {
          throw new NestableRuntimeException("JLin error: IOException: " + fileName, e1);
        }
      }
    }
  }

  private PerformResponse getResponseFromFile(File file) throws IOException {
    Reader fileReader = null;
    PerformResponse pr = null;
    try {
      fileReader = new BufferedReader(new FileReader(file));
      pr = new PerformResponse(XMLTool.parseReader(fileReader));
    } catch (BadTreeException e) {
      throw new NestableRuntimeException("JLin error: badly formed file: " + file.getAbsolutePath(), e);
    } catch (XMLParseException e) {
      throw new NestableRuntimeException("JLin error: XML Exception: " + file.getAbsolutePath(), e);
    } finally {
      if (fileReader != null) {
        fileReader.close();
      }
    }
    return pr;
  }

  private void cleanUpDirectory(File dir) {
    File[] indexArray = dir.listFiles(indexFilter);
    if (indexArray.length == 1) {
      indexArray[0].delete();
      return;
    } else {
      File[] children = dir.listFiles(resultFilter);
      for (int i = 0; i < children.length; i++) {
        if (children[i].isDirectory())
          cleanUpDirectory(children[i]);
        else
          children[i].delete();
      }
    }
  }

  private static int combine(int levelSoFar, int newEntry) {
    if (levelSoFar == ResultInterface.SEVERITY_ERROR)
      return ResultInterface.SEVERITY_ERROR;
    if (newEntry == ResultInterface.SEVERITY_ERROR)
      return ResultInterface.SEVERITY_ERROR;
    if (levelSoFar == ResultInterface.SEVERITY_WARNING)
      return ResultInterface.SEVERITY_WARNING;
    if (newEntry == ResultInterface.SEVERITY_WARNING)
      return ResultInterface.SEVERITY_WARNING;
    if (levelSoFar == ResultInterface.SEVERITY_INFO)
      return ResultInterface.SEVERITY_INFO;
    if (newEntry == ResultInterface.SEVERITY_INFO)
      return ResultInterface.SEVERITY_INFO;
    return levelSoFar;
  }
  
  public Dispatcher getDispatcher() {
    return dispatcher;
  }
  
  public void collect(File[] rawResultFiles, File collectedResultFile) throws IOException {
    List allResultList = new ArrayList();
    for (int i = 0; i < rawResultFiles.length; i++) {
      allResultList.addAll(Arrays.asList(getResponseFromFile(rawResultFiles[i]).getResults()));
    }
    collectedResponse = new PerformResponse("Collected Response");
    collectedResponse.setProperties(properties);
    ResultInterface[] processedResults =
      dispatcher.processResults(
        (ResultInterface[]) allResultList.toArray(new ResultInterface[0]));
    collectedResponse.addResults(processedResults);
    Writer fileWriter = null;
    try {
      fileWriter = new BufferedWriter(new FileWriter(collectedResultFile));
      XMLTool.writeDocument(collectedResponse.toStructureTree(), fileWriter);
    } finally {
      if (fileWriter != null) {
        fileWriter.close();
      }
    }
    properties.clear();
  }
  
}
