/*
 * Copyright (c) 2003 SAP AG - All Rights Reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_model_dynamicrfc/java/com/sap/tc/webdynpro/modelimpl/dynamicrfc/IWDDynamicRFCModel.java#1 $
 */
package com.sap.tc.webdynpro.modelimpl.dynamicrfc;

import com.sap.mw.jco.JCO;
import com.sap.tc.webdynpro.progmodel.model.api.IWDModel;
import com.sap.tc.webdynpro.progmodel.model.api.IWDModelRegistry;

/**
 * Provides access to all generic functionality that an Adaptive RFC Model
 * implements. </p>
 * 
 * Beside the functionality defined through the extended interfaces (originating
 * from the Common Model Interface (CMI) and the Web Dynpro Model Interface),
 * this includes mainly connection management. </p>
 * 
 * Application frameworks can cast from generated Adaptive RFC Models to this 
 * interface if they need to deal with generated Models in a generic way (e.g.
 * when developing application frameworks). </p>
 *
 * <h3>Connection Management</h3>
 * The connection management works as follows:<br>
 * <ul>
 * <li>When the execute() method of an executable ModelClass instance is called, 
 *     a connection (a <code>JCO.Client</code>) is needed. This connection is
 *     retrieved from the Model instance to which the ModelClass instance belongs
 *     (this is defined as the Model instance with the same <code>scope</code> 
 *     and <code>instanceId</code> as the ModelClass instance).
 * <li>A Model instance is initialized with the default logical system name 
 *     that has been declared at design time. This name can be used to retrieve a 
 *     <code>JCO.Client</code> from the WDSystemLandscape service which in turn
 *     queries the System Landscape Directory (SLD). 
 * <li>The <code>JCO.Client</code> to be used by a Model instance can be set at
 *     any time using one of the following options (with increasing priority):
 *     <ol>
 *     <li>Use the Model's default as declared at design time in the IDE using
 *         the setting "defaultLogicalSystemName"
 *     <li>Set the JCO.Client implicitly by setting a logical system name using
 *         {@link #setSystemName(String)}
 *     <li>Set the JCO.Client explicitly using {@link #setJcoClient(JCO.Client)}
 *     <li>Set the JCO.Client implicitly using {@link #setConnectionProvider(IWDDynamicRFCModel)}.
 *         This Model will share the connection with the given model, which is 
 *         the only responsible for that connection. See "Sharing Connections"
 *         below for details.
 *     </ol>
 *     See method {@link #getCurrentlyUsedJcoClient()} for details.
 * </ul>
 * 
 * When multiple of the methods above are called, the one with the highest 
 * priority will determine the active <code>JCO.Client</code>. This means, that
 * setting another Model instance as connection provider will override setting
 * the <code>JCO.Client</code> explicitly, which in turn will override any  
 * logicalSystemName, and setting a logicalSystemName will override the 
 * defaultLogicalSystemName.</p>
 * 
 * Calling one of the methods above with an argument of <code>null</code> 
 * will reset (undo) any previous call of the same method. E.g. setJcoClient(null)
 * will cause the logicalSystemName to be used to attain a <code>JCO.Client</code> 
 * (if given, otherwise the default is used). setLogicalSystemName(null) will 
 * cause the defaultLogicalSystemName to be used (unless a <code>JCO.Client</code>
 * was set explicitly), setLogicalSystemName(null) and setJcoClient(null) will 
 * cause the defaultLogicalSystemName to be used. </p>
 *
 * <h3>Sharing connections between multiple Models</h3>
 * Setting the <code>JCO.Client</code> implicitly by setting a connection provider
 * using {@link #setConnectionProvider(IWDDynamicRFCModel)} allows sharing a 
 * connection between multiple Model instances. But be aware that only the scope 
 * of the connection providing Model ("master" Model instance) is responsible for 
 * cleaning up the connection. If this connection is disconnected and disposed 
 * before a slave instance is finished, the above methods for <code>JCO.Client</code>
 * retrieval are again used for the master instance. This could finally result 
 * in using the defaultLogicalSystemName of the master instance. </p>
 * 
 * If any but the last method above is used, the connection will be disconnected 
 * automatically at the end of the Model's scope, if the scope is other than 
 * <code>NO_SCOPE</code>. This means that the connection will be disconnected, 
 * when the lifetime of the Model ends. If automatic disconnection of the 
 * <code>JCO.Client</code> connection is not desired, use either the scope
 * <code>NO_SCOPE</code> or use setJcoClient(null) before the scope ends. 
 * In both cases you are responsible for proper clean-up of the used connection.<br>
 * <strong>Note that connections are a very expensive resource and can cause a 
 * server to crash if not maintained properly.</strong> </p>
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_model_dynamicrfc/java/com/sap/tc/webdynpro/modelimpl/dynamicrfc/IWDDynamicRFCModel.java#1 $
 */
public interface IWDDynamicRFCModel extends IWDModel, IWDModelRegistry
{
  /**
   * Assigns the logical system name used to retrieve connection information
   * and credentials from the System Landscape Directory (SLD). Using this
   * method allows to determine the <code>JCO.Client</code> for this Model
   * by naming the logical system.
   *
   * @return name of logical system in System Landscape Directory.
   */
  void setSystemName(String systemName);
  
  /**
   * Returns the logical system name as set by {@link #setSystemName(String)}. 
   * If no name has been set so far, the defaultLogicalSystemName as specified
   * at design time is returned. </p>
   * If none of the higher prioritized methods has been used, the return value 
   * of this method is used to retrieve connection information and credentials
   * from the System Landscape Directory (SLD).
   *
   * @return name of logical system in System Landscape Directory.
   */
  String getSystemName();

  /**
   * Assigns this model instance's <code>JCO.Client</code>.
   * The Client will be used as long as this model lives. The (maximum) lifetime
   * of a Model instance is determined by the lifetime of its scope (e.g. 
   * <code>TASK_SCOPE</code> or <code>APPLICATION_SCOPE</code>).
   *
   * @param client <code>JCO.Client</code> to use to connect to backend.
   */
  void setJcoClient(JCO.Client client);
  
  /**
   * Returns the <code>JCO.Client</code> assigned explicitly to this Model 
   * instance or <code>null</code> if none has been assigned yet.
   * This does not necessarily represent the <code>JCO.Client</code> to use
   * for this model, e.g. if this model has been assigned a ConnectionProvider.
   * See interface documentation for details.
   *
   * @return Jco Client.
   */
  JCO.Client getJcoClient();

  /**
   * Assigns this Model instance's connection provider.
   * The connection provider will be used for retrieval of the JCO.Client from 
   * now on. If a <code>JCO.Client</code> was previously assigned, it will no 
   * longer be used, but will still be disconnected when the scope of this Model
   * ends. 
   *
   * @param connProvider DynamicRFCModel instance to use to retrive JCO.Client for connecting to backend.
   */
  void setConnectionProvider(IWDDynamicRFCModel connProvider);
  
  /**
   * Returns this Model instance's connection provider (or null if none was 
   * assigned). The connection provider is the Model instance to be used for
   * retrieval of the <code>JCO.Client</code>. 
   * 
   * Note: The connection provider is not referenced directly but retrieved 
   * by looking it up in the WDModelFactory with the class, scope and instanceId
   * as defined by the Model instance that was previously set as the connection 
   * provider.
   * 
   * @param connProvider IWDDynamicRFCModel instance to use to retrieve the 
   *                     <code>JCO.Client</code> for connecting to backend.
   */
  IWDDynamicRFCModel getConnectionProvider();
  
  /**
   * Returns the <code>JCO.Client</code> currently to be used for this Model instance.
   * 
   * The returned <code>JCO.Client</code> is determined by the following algorithm:
   * <ol>
   * <li>If a connection provider has been defined for this Model using  
   *     {@link #setConnectionProvider(IWDDynamicRFCModel)} and if that connection
   *     provider still has a connection (its scope did not end and the associated
   *     connection was not cleared), then the connection of the connection provider
   *     is returned.
   * <li>If a <code>JCO.Client</code> has been set explicitly by calling 
   *     {@link #setJcoClient(JCO.Client)}, then that client is returned.
   * <li>If a logical system name has been defined using {@link #setSystemName(String)}
   *     then the System Landscape Directory (SLD) is queried for the connection 
   *     information and credentials stored under the given name. A JCO connection
   *     with the retrieved properties is created and returned. If accessing the 
   *     SLD or querying the given name or creating the connection fails for any
   *     reason, a WDDynamicRFCException is thrown. 
   * <li>If none of the above applies, that is if none of the connection defining
   *     methods has been called, then the default logical system name - as declared
   *     at design time - is used to lookup connection information and credentials 
   *     in the SLD and to create a connection from the retrieved data. As it was
   *     stated in the previous case, any error during SLD access or connection 
   *     creation results in an WDDynamicRFCException being thrown.
   * </ol>
   * </p>
   * 
   *<strong> Note that none of the options above implies the automatic connection of the
   * returned <code>JCO.Client</code> </strong>.
   * 
   * @return JCO.Client currently used to connect to backend.
   */
  JCO.Client getCurrentlyUsedJcoClient() throws WDDynamicRFCException;

	/**
	 * checks if the connection is alive (connected) and disconnects it
	 * 
	 * This will always guarantee, that the connection is no longer in use
	 * and all locks in the backend as well as the rollarea is released.
	 * 
	 * NOTE(1):
	 * This method only works for automatically attained connections as well as for
	 * non pooled connections, set via the API method setJcoClient(). If you have assigned
	 * a pooled JcoClient via setJcoClient(), then you must release the client yourself
	 * and call setJcoClient(null)
	 * 
	 * NOTE(2):
	 * If this model is being provided a connection through a connectionProvider,
	 * then this connection will NOT be disconnected. It must then be disconnected
	 * through the connectionProvider directly 
	 */
	void disconnectIfAlive();
}
