/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/operands/ResourceOperandType.java#2 $
 */

package com.sapportals.wcm.service.relation.operands;

// #SP5LOG# */ import com.sapportals.wcm.util.log.Category;
/*
 * #SP6LOG#
 */
import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IProperty;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.IResourceFactory;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.Property;
import com.sapportals.wcm.repository.PropertyName;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.service.IServiceTypesConst;
import com.sapportals.wcm.service.relation.IRelationOperand;
import com.sapportals.wcm.service.relation.IRelationOperandType;
import com.sapportals.wcm.service.urimapper.IUriMapperService;
/*
 * #SP6LOG#
 */
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;

import java.util.Iterator;

/**
 * <code>IRelationOperandType</code> implementation for <code>IResource</code>
 * s. <br>
 *
 */
public class ResourceOperandType
   implements IRelationOperandType {

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  // #SP5LOG# */ private static Category log = Category.getInstance(ResourceOperandType.class.getName());
  /*
   * #SP6LOG#
   */
  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.service.relation.operands.ResourceOperandType.class);
  protected static IResourceFactory factory = null;

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  protected IProperty resourceType;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create a resource operand type for any type of resource.
   */
  public ResourceOperandType() {

    this.resourceType = null;

  }


  // ------------------------------------------------------------------------
  /**
   * Create a resource operand type for a specific resource type.
   *
   * @param type the <code>String</code> which defines the resource type. if
   *      <code>null</code> , this type will match any resource type.
   * @throws ResourceException if the type could not be build.
   */
  public ResourceOperandType(String type)
    throws ResourceException {

    if (type == null) {
      this.resourceType = null;
    }
    else {
      this.resourceType = new Property(PropertyName.createResourceType(), type);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Create a resource operand type for a given resource's type.
   *
   * @param resourceType TBD: Description of the incoming method parameter
   * @throws InvalidArgumentException if the given type property
   *      is not a resource type property.
   */
  public ResourceOperandType(IProperty resourceType)
    throws InvalidArgumentException {

    if ((resourceType != null)
       && (!PropertyName.createResourceType().equals(resourceType.getPropertyName()))
      ) {
      throw new InvalidArgumentException("only resource type property allowed");
    }

    this.resourceType = resourceType;

  }


  // ------------------------------------------------------------------------
  /**
   * Create a resource operand type for a given resource's type.
   *
   * @param resource the <code>IResource</code> which specifies the resource
   *      type to use. if <code>null</code> , this type will match any resource
   *      type.
   * @throws ResourceException if the resource's resource type
   *      could not be retrieved.
   */
  public ResourceOperandType(IResource resource)
    throws ResourceException {

    if (resource == null) {
      this.resourceType = null;
    }
    else {
      try {
        this.resourceType = resource.getProperty(PropertyName.createResourceType());
      }
      catch (ResourceException e) {
        this.resourceType = null;
      }
    }

  }


  // ------
  // Object -------------------------------------------------------------------
  // ------

  // ------------------------------------------------------------------------
  public boolean equals(Object object) {

    if (object instanceof ResourceOperandType) {
      ResourceOperandType type = (ResourceOperandType)object;
      if (this.resourceType == null) {
        return (type.resourceType == null);
      }
      else {
        return (this.resourceType.equals(type.resourceType));
      }
    }
    return false;
  }


  // --------------------
  // IRelationOperandType -----------------------------------------------------
  // --------------------

  // ------------------------------------------------------------------------
  public boolean isDeleteTrackingSupported() {

    return true;// id can be checked for existence

  }


  // ------------------------------------------------------------------------
  public boolean isChangeTrackingSupported() {

    return true;// [only with eventing] change for id's will be detected on event

  }


  // ------------------------------------------------------------------------
  public boolean isCopyTrackingSupported() {

    return true;// [only with eventing] copy for id's will be detected on event

  }


  // ------------------------------------------------------------------------
  public boolean check(IRelationOperand operand)
    throws ResourceException {

    if (operand == null) {
      return false;// can't check <null>
    }

    IRelationOperandType otherType = operand.getType();
    if (!(otherType instanceof ResourceOperandType)) {
      return false;// can't handle other implementations
    }

    if (this.resourceType == null) {
      return true;// this type doesn't care about resource types
    }

    ResourceOperandType other = (ResourceOperandType)otherType;
    if (other.resourceType == null) {
      return false;// this type is for a specific resource type, the other type is not
    }

    try {
      if (this.resourceType.isMultivalued()) {
        if (other.resourceType.isMultivalued()) {
          Iterator iterator = this.resourceType.getValues().iterator();
          while (iterator.hasNext()) {
            if (!other.resourceType.getValues().contains((String)iterator.next())) {
              return false;// both are multivalued but one of this values is not contained in the other value list
            }
          }
          return true;// both are multivalued and all of this values are contained in the other list
        }
        else {
          return false;// this is multivalued but the other is not
        }
      }
      else {
        if (other.resourceType.isMultivalued()) {
          return other.resourceType.getValues().contains(this.resourceType.getValueAsString());// other is multivalued but this is not
        }
        else {
          return other.resourceType.getValueAsString().equals(this.resourceType.getValueAsString());// both are single valued
        }
      }
    }
    catch (ResourceException e) {
      // #SP5LOG# */ if( ResourceOperandType.log.isWarnEnabled() ) {
      // #SP5LOG# */   ResourceOperandType.log.warn("unable to get type property: " + e.getMessage());
      // #SP5LOG# */ }
      /*
       * #SP6LOG#
       */
      if (ResourceOperandType.log.beWarning()) {
        /*
         * #SP6LOG#
         */
        ResourceOperandType.log.warningT("check(266)", "unable to get type property: " + e.getMessage());
        /*
         * #SP6LOG#
         */
      }
      throw e;
    }

  }


  // ------------------------------------------------------------------------
  public IRelationOperand getOperand(String id,
    IResourceContext context)
    throws InvalidArgumentException,
    ResourceException {

    if (id == null) {
      throw new InvalidArgumentException("<null> not allowed as id");
    }

    if (ResourceOperandType.factory == null) {
      // get the resource factory
      synchronized (ResourceOperandType.class) {
        if (ResourceOperandType.factory == null) {
          ResourceOperandType.factory = ResourceFactory.getInstance();
        }
      }
    }

    // #SP5LOG# */ RID rid = new RID(id);
    /*
     * #SP6LOG#
     */
    RID rid = RID.getRID(id);
    IResource r = ResourceOperandType.factory.getResource(rid, context);
    if (r == null) {
      // Resource not found. This can happen because delete events of relation-source
      // repositories are not handled up to now.
      return new ResourceOperand(rid);
    }
    else {
      return new ResourceOperand(r);
    }
  }


  // ------------------------------------------------------------------------
  public IRelationOperand getOperand(Object object)
    throws InvalidArgumentException {

    if (object == null) {
      throw new InvalidArgumentException("<null> not allowed as operand value");
    }

    if (object instanceof ResourceOperand) {
      return (ResourceOperand)object;
    }

    if (!(object instanceof IResource)) {
      throw new InvalidArgumentException("class " + object.getClass().getName() + " not supported as ResourceOperand");
    }

    return new ResourceOperand((IResource)object);
  }

}
