/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/operands/ResourceOperand.java#2 $
 */

package com.sapportals.wcm.service.relation.operands;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.service.relation.IRelationOperand;
import com.sapportals.wcm.service.relation.IRelationOperandType;
import com.sapportals.wcm.util.uri.RID;

/**
 * <code>IRelationOperand</code> implementation for <code>IResource</code> s.
 * <br>
 *
 */
public class ResourceOperand
   extends AbstractRelationOperand {

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  protected IResource resource;
  protected RID rid;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create a <code>RelationOperand</code> out of a <code>IResource</code> .
   *
   * @param resource the <code>IResource</code> to wrap.
   * @throws InvalidArgumentException if the given resource is
   *      <code>null</code> .
   */
  public ResourceOperand(IResource resource)
    throws InvalidArgumentException {

    if (resource == null) {
      throw new InvalidArgumentException("<null> resource not allowed");
    }

    this.resource = resource;
    try {
      this.rid = resource.getRID();
    }
    catch (ResourceException e) {
      throw new InvalidArgumentException("unable to get resource RID");
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Create a <code>RelationOperand</code> out of a <code>IResource</code> .
   *
   * @param rid TBD: Description of the incoming method parameter
   * @throws InvalidArgumentException if the given resource is
   *      <code>null</code> .
   */
  public ResourceOperand(RID rid)
    throws InvalidArgumentException {

    if (rid == null) {
      throw new InvalidArgumentException("<null> rid not allowed");
    }

    this.resource = null;
    this.rid = rid;

  }


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  // ------------------------------------------------------------------------
  /**
   * Get the wrapped resource.
   *
   * @return the <code>IResource</code> for this operand.
   */
  public IResource getValue() {

    return this.resource;
  }


  // ------------------------------------------------------------------------
  /**
   * If the wrapped resource is an internal link, return the target of the link
   * (following several links). if it's not a internal link just return the
   * resource.
   *
   * @return the <code>IResource</code> for this operand if it's not an internal
   *      link, otherwise the target of this operand.
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getLinkValue()
    throws ResourceException {

    IResource target = this.resource;
    while ((target != null)
       && (target.getLinkType() == LinkType.INTERNAL)
      ) {
      target = target.getTargetResource();
    }

    return target;
  }


  // ------
  // Object -------------------------------------------------------------------
  // ------

  // ------------------------------------------------------------------------
  public boolean equals(Object object) {

    if (object instanceof ResourceOperand) {
      ResourceOperand operand = (ResourceOperand)object;
      return (this.rid.toString().equals(operand.rid.toString()));
    }

    return false;
  }


  // ----------------
  // IRelationOperand ---------------------------------------------------------
  // ----------------

  // ------------------------------------------------------------------------
  public IRelationOperandType getType()
    throws ResourceException {

    return new ResourceOperandType(this.resource);
  }


  // ------------------------------------------------------------------------
  public String getID()
    throws ResourceException {

    if (rid == null) {
      return null;
    }

    return this.rid.toString();
  }


  // ------------------------------------------------------------------------
  public Object getObject() {

    return this.resource;
  }


  // ------------------------------------------------------------------------
  public IResource asResource()
    throws ResourceException {

    return this.resource;
  }

}
