/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationWatcher.java#1 $
 */

package com.sapportals.wcm.service.relation;

import com.sapportals.wcm.repository.IProperty;
import com.sapportals.wcm.repository.IPropertyName;

/**
 * A watcher for relations. <br>
 * A <code>IRelationWatcher</code> s must be registered to a relation type from
 * a application, it will not be restarted automatically.
 */
public interface IRelationWatcher {

  /**
   * Get a unique ID for this watcher.
   *
   * @return a <code>String</code> with the unique id of this watcher.
   */
  public String getID();


  /**
   * returns true if this watcher is interested in create events.
   *
   * @return a <code>boolean</code> true if <code>onCreate</code> should be
   *      called.
   */
  public boolean isCreateWatcher();


  /**
   * callback when a new relation was created.
   *
   * @param relation the <code>IRelation</code> which was created.
   */
  public void onCreate(IRelation relation);


  /**
   * returns true if this watcher is interested in delete events.
   *
   * @return a <code>boolean</code> true if <code>onDelete</code> should be
   *      called.
   */
  public boolean isDeleteWatcher();


  /**
   * callback when a relation is about to be deleted.
   *
   * @param relation the <code>IRelation</code> which is about to be deleted.
   */
  public void onDelete(IRelation relation);


  /**
   * returns true if this watcher is interested in set property events.
   *
   * @return a <code>boolean</code> true if <code>onPropertySet</code> should be
   *      called.
   */
  public boolean isPropertySetWatcher();


  /**
   * callback when a property was set on a relation.
   *
   * @param relation the <code>IRelation</code> where the property was set.
   * @param setProperty TBD: Description of the incoming method parameter
   */
  public void onPropertySet(IRelation relation,
    IProperty setProperty);


  /**
   * returns true if this watcher is interested in delete property events.
   *
   * @return a <code>boolean</code> true if <code>onPropertyDelete</code> should
   *      be called.
   */
  public boolean isPropertyDeleteWatcher();


  /**
   * callback when a property was deleted from a relation.
   *
   * @param relation the <code>IRelation</code> where the property was deleted.
   * @param deletedProperty the <code>IPropertyName</code> of the property that
   *      was deleted from the relation.
   */
  public void onPropertyDelete(IRelation relation,
    IPropertyName deletedProperty);


  /**
   * returns true if this watcher is interested in source change events.
   *
   * @return a <code>boolean</code> true if <code>onSourceChange</code> should
   *      be called.
   */
  public boolean isSourceChangeWatcher();


  /**
   * callback when a relation's source operand rename was detected. (the
   * relation is only changed if automatic change tracking is enabled)
   *
   * @param relation the <code>IRelation</code> which was affected.
   * @param oldSourceId the <code>String</code> with the old source id.
   * @param newSourceId TBD: Description of the incoming method parameter
   */
  public void onSourceChange(IRelation relation,
    String oldSourceId,
    String newSourceId);


  /**
   * returns true if this watcher is interested in target change events.
   *
   * @return a <code>boolean</code> true if <code>onTargetChange</code> should
   *      be called.
   */
  public boolean isTargetChangeWatcher();


  /**
   * callback when a relation's target operand rename was detected. (the
   * relation is only changed if automatic change tracking is enabled)
   *
   * @param relation the <code>IRelation</code> which was affected.
   * @param oldTargetId the <code>String</code> with the old target id.
   * @param newSourceId TBD: Description of the incoming method parameter
   */
  public void onTargetChange(IRelation relation,
    String oldTargetId,
    String newSourceId);


  /**
   * returns true if this watcher is interested in source delete events.
   *
   * @return a <code>boolean</code> true if <code>onSourceDelete</code> should
   *      be called.
   */
  public boolean isSourceDeleteWatcher();


  /**
   * callback when a relation source operand was deleted. (the relation will
   * only be deleted if automatic delete tracking is enabled)
   *
   * @param relation the <code>IRelation</code> which is about to be deleted.
   */
  public void onSourceDelete(IRelation relation);


  /**
   * returns true if this watcher is interested in target delete events.
   *
   * @return a <code>boolean</code> true if <code>onTargetDelete</code> should
   *      be called.
   */
  public boolean isTargetDeleteWatcher();


  /**
   * callback when a relation target operand was deleted. (the relation will
   * only be deleted if automatic delete tracking is enabled)
   *
   * @param relation the <code>IRelation</code> which is about to be deleted.
   */
  public void onTargetDelete(IRelation relation);


  /**
   * returns true if this watcher is interested in source copy events.
   *
   * @return a <code>boolean</code> true if <code>onSourceCopy</code> should be
   *      called.
   */
  public boolean isSourceCopyWatcher();


  /**
   * callback when an exisiting relation was copied because the source operand
   * was copied (will happen only, when <code>isSourceCopyTrackingEnabled()
   * </code> is set for the relation type).
   *
   * @param relation the <code>IRelation</code> which was created.
   * @param originalRelation the <code>IRelation</code> from which the relation
   *      was copied.
   */
  public void onSourceCopy(IRelation relation,
    IRelation originalRelation);


  /**
   * returns true if this watcher is interested in target copy events.
   *
   * @return a <code>boolean</code> true if <code>onTargetCopy</code> should be
   *      called.
   */
  public boolean isTargetCopyWatcher();


  /**
   * callback when an exisiting relation was copied because the target operand
   * was copied (will happen only, when <code>isTargetCopyTrackingEnabled()
   * </code> is set for the relation type).
   *
   * @param relation the <code>IRelation</code> which was created.
   * @param originalRelation the <code>IRelation</code> from which the relation
   *      was copied.
   */
  public void onTargetCopy(IRelation relation,
    IRelation originalRelation);

}
