/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationTypeList.java#1 $
 */

package com.sapportals.wcm.service.relation;

import java.util.Collection;
import java.util.Comparator;

/**
 * A typed list for <code>IRelationType</code> s. <br>
 *
 */
public interface IRelationTypeList {

  /**
   * Return this typed list as an standard java collection.
   *
   * @return a <code>Collection</code> representing this typed list.
   */
  public Collection getCollection();


  /**
   * Return the number of elements in the list.
   *
   * @return an <code>int</code> with the size of the list.
   */
  public int size();


  /**
   * Check if the specified relation type is in the list.
   *
   * @param relationType TBD: Description of the incoming method parameter
   * @return a <code>boolean</code> <code>true</code> if the specified relation
   *      type is in the list.
   */
  public boolean contains(IRelationType relationType);


  /**
   * Add a relation type to the list.
   *
   * @param relationType the <code>IRelationType</code> to add to the list.
   * @return a <code>boolean</code> <code>true</code> if the specified relation
   *      type was added, <code>false</code> if it already exists.
   */
  public boolean add(IRelationType relationType);


  /**
   * Add all the elements in the specified list to this list.
   *
   * @param list the <code>IRelationTypeList</code> whose elements are to be
   *      added to this list.
   * @return TBD: Description of the outgoing return value
   */
  public boolean addAll(IRelationTypeList list);


  /**
   * Remove the first occurence of the specified relation type from the list.
   *
   * @param relationType TBD: Description of the incoming method parameter
   * @return a <code>boolean</code> <code>true</code> if the specified relation
   *      type was contained in the list and removed, <code>false</code> if not.
   */
  public boolean remove(IRelationType relationType);


  /**
   * Return the relation type at the specified position.
   *
   * @param index an <code>int</code> with the position of the desired relation
   *      type.
   * @return an <code>IRelationType</code> with the relation type at the
   *      specified position.
   */
  public IRelationType get(int index);


  /**
   * Replace the element at the specified position in this list with the
   * specified relation type.
   *
   * @param index an <code>int</code> with the position of the relation type to
   *      replace.
   * @param relationType the <code>IRelationType</code> to be stored at the
   *      specified position.
   * @return a <code>IRelationType</code> with the element previously at the
   *      specified position.
   */
  public IRelationType set(int index,
    IRelationType relationType);


  /**
   * Insert the specified relation type at the specified position in this list.
   * The relation types after this position will be shifted to the right (one
   * will be added to their indices).
   *
   * @param index an <code>int</code> with the position of the relation type to
   *      be inserted.
   * @param relationType the <code>IRelationType</code> to be inserted at the
   *      specified position.
   */
  public void add(int index,
    IRelationType relationType);


  /**
   * Remove the relation type at the specified position in this list. The
   * relation types after this position will be shifted to the left (one will be
   * subtracted from their indices).
   *
   * @param index an <code>int</code> with the position of the relation type to
   *      remove.
   * @return a <code>IRelationType</code> with the removed element.
   */
  public IRelationType remove(int index);


  /**
   * Return the index of the first occurrence of the specified relation type in
   * this list, or <code>-1</code> if the list does not contain the relation
   * type.
   *
   * @param relationType the <code>IRelationType</code> to search for.
   * @return an <code>int</code> with the index of the first occurrence of the
   *      specified relation type, or <code>-1</code> if the list does not
   *      contain the given relation type.
   */
  public int indexOf(IRelationType relationType);


  /**
   * Return the index of the last occurrence of the specified relation type in
   * this list, or <code>-1</code> if the list does not contain the relation
   * type.
   *
   * @param relationType the <code>IRelationType</code> to search for.
   * @return an <code>int</code> with the index of the last occurrence of the
   *      specified relation type, or <code>-1</code> if the list does not
   *      contain the given relation type.
   */
  public int lastIndexOf(IRelationType relationType);


  /**
   * Return an iterator for the elements in this list.
   *
   * @return a <code>IRelationTypeListIterator</code> for the elements in this
   *      list.
   */
  public IRelationTypeListIterator listIterator();


  /**
   * Return an iterator for the elements in this list which starts at the
   * specified position in the list.
   *
   * @param index an <code>int</code> with the index to start the iterator at.
   * @return a <code>IRelationTypeListIterator</code> for the elements in this
   *      list.
   */
  public IRelationTypeListIterator listIterator(int index);


  /**
   * Return a list of elements from this list which starts at the first
   * specified position in the list and end at the second specified position.
   *
   * @param startIndex an <code>int</code> with the index to start the sub list
   *      at.
   * @param stopIndex an <code>int</code> with the index to stop the sub list
   *      at.
   * @return a <code>IRelationTypeList</code> with the elements of the sub-list.
   */
  public IRelationTypeList subList(int startIndex,
    int stopIndex);


  /**
   * Sort the elements in the list using the specified comparator.
   *
   * @param comparator the <code>Comparator</code> to be used for sorting.
   */
  public void sort(Comparator comparator);


  /**
   * Get the union of this list and the given list. <br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>a</i> , <i>b</i>
   * , <i>c</i> and <i>d</i> . Unlike as with <code>addAll()</code> duplicates
   * will be removed.
   *
   * @param list the <code>IRelationTypeList</code> to union with this list.
   * @return the <code>IRelationTypeList</code> with the union list.
   */
  public IRelationTypeList union(IRelationTypeList list);


  /**
   * Get the intersection of this list and the given list. <br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>b</i> and <i>c
   * </i>. Duplicates will be removed.
   *
   * @param list the <code>IRelationTypeList</code> to intersect with this list.
   * @return the <code>IRelationTypeList</code> with the intersection list.
   */
  public IRelationTypeList intersection(IRelationTypeList list);


  /**
   * Get the difference of this list to the given list.<br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain only <i>a</i> .
   * Duplicates will be removed.
   *
   * @param list the <code>IRelationTypeList</code> to subtract from this list.
   * @return the <code>IRelationTypeList</code> with the difference list.
   */
  public IRelationTypeList difference(IRelationTypeList list);

}
