/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationType.java#4 $
 */

package com.sapportals.wcm.service.relation;
import com.sapportals.wcm.repository.AccessDeniedException;
import com.sapportals.wcm.repository.IPropertyMap;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.NameAlreadyExistsException;
import com.sapportals.wcm.repository.NotSupportedException;
import com.sapportals.wcm.repository.ResourceException;

import com.sapportals.wcm.util.name.IName;

import java.util.Locale;

/**
 * The type of a relation. <br>
 * The <code>IRelationType</code> is an abstraction of the 'relation-operation'
 * (e.g. 'member-of') and. It implies also the types of the source- and
 * target-operands (<code>IRelationOperandType</code> ). A <code>RelationType
 * </code> is always registered to a unique namespace.
 */
public interface IRelationType
   extends IName {

  /**
   * Get the namespace for this relation type as used for registration.
   *
   * @return a <code>String</code> with the namespace of the relation type.
   */
  public String getNamespace();


  /**
   * Get the name of this relation type as used for registration.
   *
   * @return a <code>String</code> with the name of the relation type.
   */
  public String getName();


  /**
   * Get a description for this relation type.
   *
   * @return a <code>String</code> with the description of the relation type for
   *      the context of the relation type.
   */
  public String getDescription();


  /**
   * Get a description for this relation type for a given locale.
   *
   * @param locale TBD: Description of the incoming method parameter
   * @return a <code>String</code> with the description of the relation type for
   *      the given locale.
   */
  public String getDescription(Locale locale);


  /**
   * Get the source operand's type.
   *
   * @return the <code>IRelationOperandType</code> of this relation type's
   *      source operand type.
   */
  public IRelationOperandType getSourceType();


  /**
   * Get the target operand's type.
   *
   * @return the <code>IRelationOperand</code> of this relation type's target
   *      operand type.
   */
  public IRelationOperandType getTargetType();


  /**
   * Get the context of this relation type.
   *
   * @return a <code>IResourceContext</code> with the context for this relation
   *      type (which was used when retrieving this relation type).
   */
  public IResourceContext getContext();


  /**
   * Get the effective attributes of this relation type.
   *
   * @return a <code>IRelationTypeAttributes</code> with the attributes for this
   *      relation type.
   */
  public IRelationTypeAttributes getAttributes();


  /**
   * Check if {@link #setAttributes(IRelationTypeAttributes)} can be called or would throw an <code>
   * AccessDeniedException</code> .
   *
   * @return TBD: Description of the outgoing return value
   */
  public boolean areAttributesMutable();


  /**
   * Change the effective attributes of this relation type.
   *
   * @param attributes the <code>IRelationTypeAttributes</code> to use.
   * @return a <code>IRelationTypeAttributes</code> with the effective
   *      attributes for this relation type. <b>Note:</b> the returned
   *      attributes might differ from the passed attributes if not all
   *      attributes could be changed.
   * @throws InvalidArgumentException if the given attributes were
   *      <code>null</code> .
   * @throws AccessDeniedException if the relation type was not
   *      accessible from the types given context or if the relation type's
   *      attributes are marked as not changeable.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationTypeAttributes setAttributes(IRelationTypeAttributes attributes)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Register a watcher to this relation type.
   *
   * @param watcher a <code>IRelationWatcher</code> which will be able to handle
   *      callbacks from the triggers of a relation.
   * @throws InvalidArgumentException if the watcher was <code>
   *      null</code> .
   * @throws AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void startWatching(IRelationWatcher watcher)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Deregister a watcher from this relation type.
   *
   * @param watcher a <code>IRelationWatcher</code> which has to be
   *      deregistered.
   * @throws InvalidArgumentException if the watcher was <code>
   *      null</code> .
   * @throws AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void stopWatching(IRelationWatcher watcher)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Deregister this relation type and delete all of it's relations.
   *
   * @throws AccessDeniedException if the relation type could not
   *      be deleted within the given types context.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void delete()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Create a new relation from wrapped operands.
   *
   * @param source the <code>IRelationOperand</code> which is the wrapped source
   *      operand of the relation.
   * @param target the <code>IRelationOperand</code> which is the wrapped target
   *      operand of the relation.
   * @param properties a <code>IPropertyMap</code> with the properties for this
   *      relation.
   * @return an <code>IRelation</code> with the created relation for the given
   *      type, source and target.
   * @throws InvalidArgumentException if the source or target is
   *      <code>null</code> or the given source or target is not valid with
   *      respect to the source-/target type.
   * @throws AccessDeniedException if the given context (as passed
   *      when retrieving the relation type) has no permissions to create the
   *      relation.
   * @throws NameAlreadyExistsException if the relation already
   *      exists for the given type, source and target.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation createRelation(IRelationOperand source,
    IRelationOperand target,
    IPropertyMap properties)
    throws InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * a shortcut for {@link #createRelation(IRelationOperand, IRelationOperand,
   * IPropertyMap) createRelation(source, target, null)}.
   *
   * @param source TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation createRelation(IRelationOperand source,
    IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * Create a relation from operand values.
   *
   * @param source the <code>Object</code> which is the source operand value of
   *      the relation.
   * @param target the <code>Object</code> which is the target operand value of
   *      the relation.
   * @param properties a <code>IPropertyMap</code> with the properties for this
   *      relation.
   * @return an <code>IRelation</code> with the created relation for the given
   *      type, source and target.
   * @throws InvalidArgumentException if the source or target is
   *      <code>null</code> or the given source or target is not valid with
   *      respect to the source-/target type.
   * @throws AccessDeniedException if the given context (as passed
   *      when retrieving the relation type) has no permissions to create the
   *      relation.
   * @throws NameAlreadyExistsException if the relation already
   *      exists for the given type, source and target.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation createRelation(Object source,
    Object target,
    IPropertyMap properties)
    throws InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * a shortcut for {@link #createRelation(Object, Object, IPropertyMap)
   * createRelation(source, target, null)}.
   *
   * @param source TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation createRelation(Object source,
    Object target)
    throws InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * Get all availabe relations for this type.
   *
   * @return a <code>IRelationList</code> with all the relations.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelations()
    throws AccessDeniedException,
    ResourceException;


  /**
   * @param source a <code>IRelationOperand</code> which defines the source
   *      operand to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @deprecated as of NW04. Please use getRelationsBySource instead Get all the relations
   *      of this type for the given source operand.
   * @throws  InvalidArgumentException if the source was <code>null
   *      </code>.
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelations(IRelationOperand source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get all the relations of this type for the given source operand.
   *
   * @param source a <code>IRelationOperand</code> which defines the source
   *      operand to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the source was <code>null
   *      </code>.
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelationsBySource(IRelationOperand source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * @param source a <code>Object</code> which defines the source operands value
   *      to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source value.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @deprecated as of NW04. Please use gteRelationsBySource instead Get all the relations
   *      of this type for the given source operand value.
   * @throws  InvalidArgumentException if the source value was
   *      <code>null</code> .
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelations(Object source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get all the relations of this type for the given source operand value.
   *
   * @param source a <code>Object</code> which defines the source operands value
   *      to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source value.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the source value was
   *      <code>null</code> .
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelationsBySource(Object source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get all the relations of this type for the given target operand.
   *
   * @param target a <code>IRelationOperand</code> which defines the target
   *      operand to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      target.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the target was <code>null
   *      </code>.
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelationsByTarget(IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get all the relations of this type for the given target operand value.
   *
   * @param target a <code>Object</code> which defines the target operands value
   *      to get the relations for.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      target value.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the target value was
   *      <code>null</code> .
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelationsByTarget(Object target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get the relations for this type, which match a given query expression.
   *
   * @param query the <code>IRelationQuery</code> the searched relations should
   *      match or <code>null</code> if all relations should be returned.
   * @return a <code>IRelationList</code> with all the relations of the given
   *      type. the list might be empty if no relations are found for the given
   *      type which match the given query.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception RelationQueryException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  RelationQueryException if the relation query was not
   *      understood.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelations(IRelationQuery query)
    throws AccessDeniedException,
    RelationQueryException,
    ResourceException;


  /**
   * Get the relations for this type, which match a given query expression. <br>
   * Not yet supported!
   *
   * @param query the <code>IRelationQuery</code> the searched relations should
   *      match or <code>null</code> if all relations should be returned.
   * @param order the <code>IRelationQueryOrder</code> the searched relation
   *      list should be sorted by. if <code>null</code> the list is not
   *      ordered.
   * @return a <code>IRelationList</code> with all the relations of the given
   *      type. the list might be empty if no relations are found for the given
   *      type which match the given query.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception RelationQueryException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  RelationQueryException if the relation query was not
   *      understood.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationList getRelations(IRelationQuery query,
    IRelationQueryOrder order)
    throws AccessDeniedException,
    RelationQueryException,
    ResourceException;


  /**
   * Get a specific relation for a given type and id.
   *
   * @param id the <code>id</code> to get the relation for.
   * @return the <code>IRelation</code> for the given type and id or <code>null
   *      </code> if no relation exists for this id.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the type or the id was
   *      <code>null</code> .
   * @throws  AccessDeniedException if the relation was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation getRelation(String id)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get the relation for a given wrapped source and target.
   *
   * @param source the <code>IRelationOperand</code> which is the wrapped source
   *      operand of the relation.
   * @param target the <code>IRelationOperand</code> which is the wrapped target
   *      operand of the relation.
   * @return an <code>IRelation</code> with the relation for this type and the
   *      given source and target or <code>null</code> if no such relation
   *      exist.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if one of the source or the
   *      target parameters was <code>null</code> .
   * @throws  AccessDeniedException if the relation was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation getRelation(IRelationOperand source,
    IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get the relation for a given source and target value.
   *
   * @param source the <code>Object</code> which is the source operand value of
   *      the relation.
   * @param target the <code>Object</code> which is the target operand value of
   *      the relation.
   * @return an <code>IRelation</code> with the relation for this type and the
   *      given source and target or <code>null</code> if no such relation
   *      exist.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if one of the source or the
   *      target parameters was <code>null</code> .
   * @throws  AccessDeniedException if the relation was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation getRelation(Object source,
    Object target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  // --------------------
  // IOrderedRelationType -----------------------------------------------------
  // --------------------

  /**
   * Create a new ordered relation from wrapped operands. <br>
   * Not yet supported!
   *
   * @param position the <code>IOrderedRelationPosition</code> which defines
   *      where to put the newly created relation to. if <code>null</code> is
   *      given, a <code>OrderedRelationPosition.LAST</code> is used.
   * @param source the <code>IRelationOperand</code> which is the wrapped source
   *      operand of the relation.
   * @param target the <code>IRelationOperand</code> which is the wrapped target
   *      operand of the relation.
   * @param properties a <code>IPropertyMap</code> with the properties for this
   *      relation.
   * @return an <code>IRelation</code> with the created relation for the given
   *      type, source and target.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  NotSupportedException if this method is not
   *      supported, because this type is not a ordered relation type.
   * @throws  InvalidArgumentException if the source or target is
   *      <code>null</code> or the given source or target is not valid with
   *      respect to the source-/target type.
   * @throws  AccessDeniedException if the given context (as passed
   *      when retrieving the relation type) has no permissions to create the
   *      relation.
   * @throws  NameAlreadyExistsException if the relation already
   *      exists for the given type, source and target.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation createRelation(IOrderedRelationPosition position,
    IRelationOperand source,
    IRelationOperand target,
    IPropertyMap properties)
    throws NotSupportedException,
    InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * a shortcut for {@link #createRelation(IOrderedRelationPosition,
   * IRelationOperand, IRelationOperand, IPropertyMap) createRelation(source,
   * target, null)}. <br>
   * Not yet supported!
   *
   * @param position TBD: Description of the incoming method parameter
   * @param source TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation createRelation(IOrderedRelationPosition position,
    IRelationOperand source,
    IRelationOperand target)
    throws NotSupportedException,
    InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * Create a ordered relation from operand values. <br>
   * Not yet supported!
   *
   * @param position the <code>IOrderedRelationPosition</code> which defines
   *      where to put the newly created relation to. if <code>null</code> is
   *      given, a <code>OrderedRelationPosition.LAST</code> is used.
   * @param source the <code>Object</code> which is the source operand value of
   *      the relation.
   * @param target the <code>Object</code> which is the target operand value of
   *      the relation.
   * @param properties a <code>IPropertyMap</code> with the properties for this
   *      relation.
   * @return an <code>IRelation</code> with the created relation for the given
   *      type, source and target.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  NotSupportedException if this method is not
   *      supported, because this type is not a ordered relation type.
   * @throws  InvalidArgumentException if the source or target is
   *      <code>null</code> or the given source or target is not valid with
   *      respect to the source-/target type.
   * @throws  AccessDeniedException if the given context (as passed
   *      when retrieving the relation type) has no permissions to create the
   *      relation.
   * @throws  NameAlreadyExistsException if the relation already
   *      exists for the given type, source and target.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelation createRelation(IOrderedRelationPosition position,
    Object source,
    Object target,
    IPropertyMap properties)
    throws NotSupportedException,
    InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * a shortcut for {@link #createRelation(IOrderedRelationPosition, Object,
   * Object, IPropertyMap) createRelation(source, target, null)}. <br>
   * Not yet supported!
   *
   * @param position TBD: Description of the incoming method parameter
   * @param source TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation createRelation(IOrderedRelationPosition position,
    Object source,
    Object target)
    throws NotSupportedException,
    InvalidArgumentException,
    AccessDeniedException,
    NameAlreadyExistsException,
    ResourceException;


  /**
   * Reorder the relations for this type. <br>
   * Not yet supported!
   *
   * @param ordering TBD: Description of the incoming method parameter
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  NotSupportedException if this method is not
   *      supported, because this type is not a ordered relation type.
   * @throws  InvalidArgumentException if there is a pair of
   *      relation/relative IDs which are not compatible with respect to their
   *      type and source operand.
   * @throws  AccessDeniedException if the relation type was not
   *      accessible from the types given context.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void reorder(IOrderedRelationPositioningList ordering)
    throws NotSupportedException,
    InvalidArgumentException,
    AccessDeniedException,
    ResourceException;

}
