/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationService.java#2 $
 */

package com.sapportals.wcm.service.relation;

import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.AccessDeniedException;
import com.sapportals.wcm.repository.IPropertyMap;
import com.sapportals.wcm.repository.IPropertyNameList;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.NameAlreadyExistsException;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.util.resource.Descriptions;

import java.util.Properties;

/**
 * A service to store <code>IRelation</code> s between two objects. <br>
 * A <code>IRelation</code> is relationship between two <code>IRelationOperand
 * </code>s, called <i>source</i> and <i>target</i> operand. <br>
 * <code>IRelation</code> s belong to their <code>IRelationType</code> , which
 * in turn defines the type of the <code>IRelationOperandType</code> s for the
 * source and target. <br>
 * For example: the <code>RelationType</code> object with name <i>attached-to
 * </i> would store the relationships between documents, thus the <code>
 * RelationOperandTypes</code> for source and target operands are <code>
 * IRelation</code> s. A specific relationship like '<i>
 * /documents/public/specification.doc</i> is <i>attached-to</i> <i>
 * /documents/public/infos/project.doc</i> ' is then a <code>IRelation</code>
 * object.
 */
public interface IRelationService
   extends IService {

  public final String NAMESPACE = "com.sapportals.wcm.service.relation";


  /**
   * Get a query builder to build search query expressions and order expressions
   * for retrieving relations.
   *
   * @return a <code>IQueryBuilder</code> which builds query and order
   *      expressions which can be used with the {@link #getRelations(IResourceContext, IRelationQuery)} and the 
   * {@link #getRelations(IResourceContext, IRelationQuery, IRelationQueryOrder)} methods.
   * @throws ResourceException if there was an internal error
   *      while retrieving the query builder.
   */
  public IRelationQueryBuilder getQueryBuilder()
    throws ResourceException;


  /**
   * Get a type handler for a relation type (which defines 'workarea' for a
   * relation type, like caches and tablespaces).
   *
   * @param id a <code>String</code> with the id of the type handler to retrieve
   *      or <code>null</code> to get the default type handler.
   * @return the <code>IRelationTypeHandler</code> for the given id, or <code>
   *      null</code> if no such handler exists.
   * @exception ResourceException Exception raised in failure situation
   * @see #getRelationTypeAttributes(IRelationTypeHandler, IPropertyNameList, boolean, boolean, boolean, boolean, boolean, boolean)
   * @throws ResourceException if there was an internal error
   *      while retrieving the handler.
   */
  public IRelationTypeHandler getRelationTypeHandler(String id)
    throws ResourceException;


  /**
   * Same as {@link #getRelationTypeAttributes(IRelationTypeHandler,
   * IPropertyNameList, boolean, boolean, boolean, boolean, boolean, boolean,
   * boolean, boolean)} with <code>sourceCopyTracking</code> and <code>
   * targetCopyTracking</code> set to <code>false</code>
   *
   * @param typeHandler TBD: Description of the incoming method parameter
   * @param requiredProperties TBD: Description of the incoming method parameter
   * @param oneToOne TBD: Description of the incoming method parameter
   * @param ordered TBD: Description of the incoming method parameter
   * @param sourceDeleteTracking TBD: Description of the incoming method
   *      parameter
   * @param targetDeleteTracking TBD: Description of the incoming method
   *      parameter
   * @param sourceChangeTracking TBD: Description of the incoming method
   *      parameter
   * @param targetChangeTracking TBD: Description of the incoming method
   *      parameter
   * @return relationTypeAttributes
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationTypeAttributes getRelationTypeAttributes(IRelationTypeHandler typeHandler,
    IPropertyNameList requiredProperties,
    boolean oneToOne,
    boolean ordered,
    boolean sourceDeleteTracking,
    boolean targetDeleteTracking,
    boolean sourceChangeTracking,
    boolean targetChangeTracking)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Get a set of unchecked attributes which can be use to created a <code>
   * IRelationType</code> . <br>
   * These attributes may not be compatible with the specific <code>
   * IRelationOperandType</code> s of a <code>IRelationType</code> , see 
   * {@link #checkRelationTypeAttributes(IRelationTypeAttributes, IRelationOperandType, IRelationOperandType)}.
   *
   * @param typeHandler a <code>IRelationTypeHandler</code> which defines the
   *      workarea for this relation type (see {@link #getRelationTypeHandler(String)})
   * @param requiredProperties a <code>IPropertyNameList</code> with the names
   *      of the required properties.
   * @param oneToOne a <code>boolean</code> <code>true</code> if only one
   *      relation per source operand id is allowed (1:1 relations).
   * @param ordered a <code>boolean</code> <code>true</code> if the relations
   *      have an intrinsic order per type and source operand. Note that it
   *      doesn't make much sense together with the <code>oneToOne</code> flag!
   * @param sourceDeleteTracking a <code>boolean</code> <code>true</code> if
   *      relations should be deleted when their source operand is deleted.
   * @param targetDeleteTracking a <code>boolean</code> <code>true</code> if
   *      relations should be deleted when their target operand is deleted.
   * @param sourceChangeTracking a <code>boolean</code> <code>true</code> if the
   *      relations source operand should be changed when the object it
   *      references is moved.
   * @param targetChangeTracking a <code>boolean</code> <code>true</code> if the
   *      relations target operand should be changed when the object it
   *      references is moved.
   * @param sourceCopyTracking a <code>boolean</code> <code>true</code> if the
   *      another relation should be created between source-copy and target,
   *      when an exisiting relations source is copied.
   * @param targetCopyTracking TBD: Description of the incoming method parameter
   * @return the unchecked <code>IRelationTypeAttributes</code> for the given
   *      parameters.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @see #createRelationType(IResourceContext, String, String, Descriptions,
   *      IRelationOperandType, IRelationOperandType, IRelationTypeAttributes)
   * @throws InvalidArgumentException if the given typeHandler is
   *      not usable with this service.
   * @throws ResourceException if there was an internal error
   *      getting the relation type attributes.
   */
  public IRelationTypeAttributes getRelationTypeAttributes(IRelationTypeHandler typeHandler,
    IPropertyNameList requiredProperties,
    boolean oneToOne,
    boolean ordered,
    boolean sourceDeleteTracking,
    boolean targetDeleteTracking,
    boolean sourceChangeTracking,
    boolean targetChangeTracking,
    boolean sourceCopyTracking,
    boolean targetCopyTracking)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Get a checked set of attributes which are compatible with the given source
   * and target operand types and can be use to created a <code>IRelationType
   * </code>. <br>
   * Features that are not supported are set to false, e.g. if the given <code>
   * sourceDeleteTracking</code> -parameter is <code>true</code> , but the
   * specified <code>sourceType</code> does not support delete tracking, this
   * method will reset the <code>sourceDeleteTracking</code> -parameter to
   * <code>false</code> in the attributes it returns.
   *
   * @param attributes the <code>IRelationTypeAttributes</code> to check against
   *      the given operand types.
   * @param sourceType the <code>IRelationOperandType</code> of the source
   *      operands.
   * @param targetType the <code>IRelationOperandType</code> of the target
   *      operands.
   * @return the checked <code>IRelationTypeAttributes</code> .
   * @see #createRelationType(IResourceContext, String, String, Descriptions,
   *      IRelationOperandType, IRelationOperandType, IRelationTypeAttributes)
   * @throws InvalidArgumentException if the given source- or
   *      targetType is <code>null</code> , or if the given attribute's
   *      typeHandler is not usable with this service.
   * @throws ResourceException if there was an internal error
   *      checking the relation type attributes.
   */
  public IRelationTypeAttributes checkRelationTypeAttributes(IRelationTypeAttributes attributes,
    IRelationOperandType sourceType,
    IRelationOperandType targetType)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Convenience method to create a relation type for resources as source and
   * target. <br>
   * Attributes are checked within this method, the rest is like with {@link
   * #createRelationType(IResourceContext, String, String, Descriptions,
   * IRelationOperandType, IRelationOperandType, IRelationTypeAttributes)} with
   * both <code>IRelationOperandType</code> s set to <code>
   * {@link com.sapportals.wcm.service.relation.operands.ResourceOperandType} </code> .
   *
   * @param context the <code>IResourceContext</code> to use for creating the
   *      relation type.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      type.
   * @param name a <code>String</code> with the name of the relation type.
   * @param descriptions the set of <code>Descriptions</code> for this relation
   *      type. if <code>null</code> , a default description is build from
   *      namespace and name.
   * @param attributes a set of <code>IRelationTypeAttributes</code> to be used
   *      for this relation type, may be unchecked. if <code>null</code> , the
   *      default attributes are used.
   * @return a <code>IRelationType</code> with the given parameters, for
   *      resource-type sources and resource-type targets.
   * @throws NameAlreadyExistsException if the given namespace and
   *      name are already assigned to a relation type.
   * @throws InvalidArgumentException if the given namespace or
   *      name are not valid (exceed the maximum length of 255 chars when
   *      encoded) or the given context is <code>null</code> .
   * @throws AccessDeniedException if the relation type could not
   *      be created within the given context.
   * @throws ResourceException if there was an internal error
   *      creating the relation type.
   */
  public IRelationType createResourceRelationType(IResourceContext context,
    String namespace,
    String name,
    Descriptions descriptions,
    IRelationTypeAttributes attributes)
    throws NameAlreadyExistsException,
    InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Convenience method to create a relation type for resources as source and
   * strings as target. <br>
   * Attributes are checked within this method, the rest is like with {@link
   * #createRelationType(IResourceContext, String, String, Descriptions,
   * IRelationOperandType, IRelationOperandType, IRelationTypeAttributes)} with
   * <code>sourceType</code> set to <code> 
   * {@link com.sapportals.wcm.service.relation.operands.ResourceOperandType}</code> and <code>
   * targetType</code> set to <code>{@link
   * com.sapportals.wcm.service.relation.operands.StringOperandType}</code> .
   *
   * @param context the <code>IResourceContext</code> to use for creating the
   *      relation type.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      type.
   * @param name a <code>String</code> with the name of the relation type.
   * @param descriptions the set of <code>Descriptions</code> for this relation
   *      type. if <code>null</code> , a default description is build from
   *      namespace and name.
   * @param attributes a set of <code>IRelationTypeAttributes</code> to be used
   *      for this relation type, may be unchecked. if <code>null</code> , the
   *      default attributes are used.
   * @return a <code>IRelationType</code> with the given parameters, for
   *      resource-type sources and string-type targets.
   * @throws NameAlreadyExistsException if the given namespace and
   *      name are already assigned to a relation type.
   * @throws InvalidArgumentException if the given namespace or
   *      name are not valid (exceed the maximum length of 255 chars when
   *      encoded) or the given context is <code>null</code> .
   * @throws AccessDeniedException if the relation type could not
   *      be created within the given context.
   * @throws ResourceException if there was an internal error
   *      creating the relation type.
   */
  public IRelationType createStringRelationType(IResourceContext context,
    String namespace,
    String name,
    Descriptions descriptions,
    IRelationTypeAttributes attributes)
    throws NameAlreadyExistsException,
    InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Create a new relation type for a given namespace and name if it does not
   * already exist. Same as {@link #createRelationType(IResourceContext, String,
   * String, Descriptions, IRelationOperandType, IRelationOperandType,
   * IRelationTypeAttributes)} with <code>attributes</code> set to <code>null
   * </code>.
   *
   * @param context the <code>IResourceContext</code> to use for creating the
   *      relation type.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      type.
   * @param name a <code>String</code> with the name of the relation type.
   * @param descriptions the set of <code>Descriptions</code> for this relation
   *      type. if <code>null</code> , a default description is build from
   *      namespace and name.
   * @param sourceType the <code>IRelationOperandType</code> for the source
   *      operands, e.g. {@link com.sapportals.wcm.service.relation.operands.ResourceOperand}
   * @param targetType the <code>IRelationOperandType</code> for the target
   *      operands, e.g. {@link com.sapportals.wcm.service.relation.operands.StringOperand}
   * @return a <code>IRelationType</code> for the given namespace and name.
   * @exception NameAlreadyExistsException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws InvalidArgumentException if the given namespace or
   *      name are not valid (exceed the maximum length of 255 chars when
   *      encoded) or the given context is <code>null</code> .
   * @throws AccessDeniedException if the relation type could not
   *      be created within the given context.
   * @throws ResourceException if there was an internal error
   *      creating the relation type.
   */
  public IRelationType createRelationType(IResourceContext context,
    String namespace,
    String name,
    Descriptions descriptions,
    IRelationOperandType sourceType,
    IRelationOperandType targetType)
    throws NameAlreadyExistsException,
    InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Create a new relation type for a given namespace, name and set of
   * attributes if it does not already exist.
   *
   * @param context the <code>IResourceContext</code> to use for creating the
   *      relation type.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      type, e.g. <code>http://sap.com/xmlns/cm/service/relations</code> .
   * @param name a <code>String</code> with the name of the relation type, e.g.
   *      <code>attached-to</code> .
   * @param descriptions the set of <code>Descriptions</code> for this relation
   *      type in several languages. if <code>null</code> , a default
   *      description is build from namespace and name).
   * @param sourceType the <code>IRelationOperandType</code> for the source
   *      operands.
   * @param targetType the <code>IRelationOperandType</code> for the target
   *      operands.
   * @param attributes TBD: Description of the incoming method parameter
   * @return a <code>IRelationType</code> with the relation type for the given
   *      namespace and name.
   * @throws NameAlreadyExistsException if the given namespace and
   *      name are already assigned to a relation type.
   * @throws InvalidArgumentException if the given namespace or
   *      name are not valid (exceed the maximum length of 255 chars when
   *      encoded) or the given context is <code>null</code> .
   * @throws AccessDeniedException if the relation type could not
   *      be created within the given context.
   * @throws ResourceException if there was an internal error
   *      creating the relation type.
   */
  public IRelationType createRelationType(IResourceContext context,
    String namespace,
    String name,
    Descriptions descriptions,
    IRelationOperandType sourceType,
    IRelationOperandType targetType,
    IRelationTypeAttributes attributes)
    throws NameAlreadyExistsException,
    InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get an empty list for relation types.
   *
   * @return an empty <code>IRelationTypeList</code> .
   * @throws ResourceException if there was an internal error
   *      retrieving the list.
   */
  public IRelationTypeList getRelationTypeList()
    throws ResourceException;


  /**
   * Get a list of all relation types which are available within the given
   * context.
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relation types.
   * @return a <code>IRelationTypeList</code> with the list of relation types.
   * @throws InvalidArgumentException if the given context is
   *      <code>null</code> .
   * @throws ResourceException if there was an internal error
   *      retrieving the relations.
   */
  public IRelationTypeList getRelationTypes(IResourceContext context)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Get a list of all relation types of the given namespace which are available
   * within the given context.
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relation types.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      types to search for.
   * @return a <code>IRelationTypeList</code> with the list of relation types.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws InvalidArgumentException if the given namespace is
   *      not valid (exceed the maximum length of 255 chars when encoded) or the
   *      given context is <code>null</code> .
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationTypeList getRelationTypes(IResourceContext context,
    String namespace)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Get a specific relation type for a given namespace and name.
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relation type.
   * @param namespace a <code>String</code> with the namespace of the relation
   *      type to look for.
   * @param name a <code>String</code> with the name of the relation type to
   *      look for.
   * @return a <code>IRelationType</code> with the relation type for the given
   *      namespace and name, or <code>null</code> if no such relation type
   *      exists.
   * @throws InvalidArgumentException if the given namespace or
   *      name are not valid (exceed the maximum length of 255 chars when
   *      encoded) or if the given context is <code>null</code> .
   * @throws AccessDeniedException if the relation type could not
   *      be accessed within the given context.
   * @throws ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IRelationType getRelationType(IResourceContext context,
    String namespace,
    String name)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get an empty list for relations.
   *
   * @return an empty <code>IRelationList</code> .
   * @throws ResourceException if there was an internal error
   *      retrieving the list.
   */
  public IRelationList getRelationList()
    throws ResourceException;


  /**
   * Get the relations which match a given query expression. <br>
   * The query expression can be build using the <code>IRelationQueryBuilder
   * </code> returned from {@link #getQueryBuilder()}.
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relations.
   * @param query the <code>IRelationQuery</code> the searched relations must
   *      match. if <code>null</code> , all relations will be returned.
   * @return a <code>IRelationList</code> with all the relations that match the
   *      given query. the list might be empty if no matching relations are
   *      found. the list might become very large, if the query is not properly
   *      specified.
   * @throws InvalidArgumentException if the given query is not
   *      understood or the given context is <code>null</code> .
   * @throws RelationQueryException if the relation query was not
   *      understood.
   * @throws ResourceException if there was an internal error
   *      executing the query or retrieving the relations.
   */
  public IRelationList getRelations(IResourceContext context,
    IRelationQuery query)
    throws InvalidArgumentException,
    RelationQueryException,
    ResourceException;


  /**
   * Get the relations which match a given query expression, ordered by a given
   * order expression. <br>
   * Not yet supported! <br>
   * The query and the order expressions can be build using the <code>
   * IRelationQueryBuilder</code> returned from {@link #getQueryBuilder()}.
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relations.
   * @param query the <code>IRelationQuery</code> the searched relations must
   *      match. if <code>null</code> , all relations will be returned.
   * @param order the <code>IRelationQueryOrder</code> the searched relation
   *      list will be sorted with. if <code>null</code> , the list is not
   *      ordered (same as <code>getRelations(IResourceContext, IRelationQuery
   *      </code>)
   * @return a <code>IRelationList</code> with all the relations that match the
   *      given query. the list might be empty if no matching relations are
   *      found. the list might become very large, if the query is not properly
   *      specified.
   * @exception AccessDeniedException Exception raised in failure situation
   * @throws InvalidArgumentException if the given query is not
   *      understood or the given context is <code>null</code> .
   * @throws RelationQueryException if the relation query was not
   *      understood.
   * @throws ResourceException if there was an internal error
   *      executing the query or retrieving the relations.
   */
  public IRelationList getRelations(IResourceContext context,
    IRelationQuery query,
    IRelationQueryOrder order)
    throws InvalidArgumentException,
    AccessDeniedException,
    RelationQueryException,
    ResourceException;


  /**
   * Get all relations of any type for the given source. <br>
   *
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relations.
   * @param source the <code>IRelationOperand</code> which belongs to the
   *      searched relations as source operand.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source operand. the list might be empty if no matching relations are
   *      found.
   * @throws InvalidArgumentException if the given source operand
   *      is <code>null</code> .
   * @throws ResourceException if there was an internal error
   *      while retrieving the relations.
   */
  public IRelationList getRelationsBySource(IResourceContext context,
    IRelationOperand source)
    throws InvalidArgumentException,
    ResourceException;

  /**
   * Get all relations of any type for the given target. <br>
   *
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relations.
   * @param target the <code>IRelationOperand</code> which belongs to the
   *      searched relations as target operand.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      target operand. the list might be empty if no matching relations are
   *      found.
   * @throws InvalidArgumentException if the given target operand
   *      is <code>null</code> .
   * @throws ResourceException if there was an internal error
   *      while retrieving the relations.
   */
  public IRelationList getRelationsByTarget(IResourceContext context,
    IRelationOperand target)
    throws InvalidArgumentException,
    ResourceException;

  /**
   * Get all relations of any type for the given source and target. <br>
   *
   *
   * @param context the <code>IResourceContext</code> to use for retrieving the
   *      relations.
   * @param source the <code>IRelationOperand</code> which belongs to the
   *      searched relations as source operand.
   * @param target the <code>IRelationOperand</code> which belongs to the
   *      searched relations as target operand.
   * @return a <code>IRelationList</code> with all the relations for the given
   *      source and target operand. the list might be empty if no matching
   *      relations are found.
   * @throws InvalidArgumentException if the given source or
   *      target operand is <code>null</code> .
   * @throws ResourceException if there was an internal error
   *      while retrieving the relations.
   */
  public IRelationList getRelations(IResourceContext context,
    IRelationOperand source,
    IRelationOperand target)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Get an empty list for reordering. <br>
   * Not yet supported!
   *
   * @return a <code>IOrderedRelationPositioningList</code> .
   * @throws ResourceException if the list could not be created.
   */
  public IOrderedRelationPositioningList getRelationPositioningList()
    throws ResourceException;

}
