/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationQueryBuilder.java#2 $
 */

package com.sapportals.wcm.service.relation;

import com.sapportals.wcm.repository.IPropertyName;

/**
 * A query builder wich creates <code>IQueryExpressions</code> to be used with
 * the relation service. <br>
 * A <code>IRelationQueryBuilder</code> constructs <code>IRelationQuery</code>
 * s, which can be executed by <code>{@link
 * IRelationService#getRelations(com.sapportals.wcm.repository.IResourceContext, IRelationQuery)}/<code>
 * Constructing a query is done like <pre>
 * query = service.getQueryBuilder();
 * memberOfType = service.getRelationType(context, "collaboration", "member-of");
 * attributeName = new PropertyName("office", "location");
 * queryExpression = query.relation(memberOfType).and(query.source(attributeName).equals("Walldorf"));
 * </pre>
 */
public interface IRelationQueryBuilder {

  /**
   * Return query expression to select all relations.
   *
   * @return a <code>IRelationQuery</code> to select any relations.
   */
  public IRelationQuery all();


  /**
   * Return query expression to selects all the relations for those relation
   * types which match the given namespace.
   *
   * @param namespace a <code>String</code> with the namespace of the relation
   *      types. if <code>null</code> , all relation namespaces are selected.
   * @return a <code>IRelationQuery</code> to select the relations for the
   *      relation types within the given namespace.
   */
  public IRelationQuery all(String namespace);


  /**
   * Return query expression to select the relations of the given relation type.
   *
   * @param type the <code>IRelationType</code> to use as selection criteria. if
   *      <code>null</code> , all types are selected.
   * @return a <code>IRelationQuery</code> to select the relations for the given
   *      relation type.
   */
  public IRelationQuery relationType(IRelationType type);


  /**
   * Return an expression for the source operand's id (select only those
   * relations with source id matching the given expression).
   *
   * @return a <code>IRelationOperandQuery</code> with the 'source' expression.
   */
  public IRelationOperandQuery source();


  /**
   * Return an expression for the target operand's id (select only those
   * relations with target id matching the given expression).
   *
   * @return a <code>IRelationOperandQuery</code> with the 'target' expression.
   */
  public IRelationOperandQuery target();


  /**
   * Return an expression for the source operand's or target operand's id
   * (select only those relations with source id or target id matching the given
   * expression). This is just a shortcut, allowing to replace <code>
   * query.source().equal(<i>value</i> ).or(query.target().equal(<i>value</i> ))
   * </code> with <code>query.sourceOrTarget().equal(<i>value</i> )</code> .
   *
   * @return a <code>IRelationOperandQuery</code> with the 'target' expression.
   */
  public IRelationOperandQuery sourceOrTarget();


  /**
   * Return an expression for the given relation property's value (select only
   * those relations with the given relation property matching the given
   * expression).
   *
   * @param propertyName the <code>IPropertyName<(/code> of the property to use
   *      for selection.
   * @return a <code>IRelationPropertyQuery</code> with the 'relation-property'
   *      expression.
   */
  public IRelationPropertyQuery property(IPropertyName propertyName);


  /**
   * Get an order expression which sorts by source id. <br>
   *
   *
   * @return a <code>IRelationQueryOrder</code> for the source id.
   */
  public IRelationQueryOrder orderBySource();


  /**
   * Get an order expression which sorts by target id. <br>
   *
   *
   * @return a <code>IRelationQueryOrder</code> for the target id.
   */
  public IRelationQueryOrder orderByTarget();


  /**
   * Get an order expression which sorts by the value of a given property. <br>
   *
   *
   * @param propertyName the <code>IPropertyName<(/code> of the property to use
   *      for ordering.
   * @return a <code>IRelationQueryOrder</code> for the given property.
   */
  public IRelationQueryOrder orderByProperty(IPropertyName propertyName);

}
