/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationOperandType.java#2 $
 */

package com.sapportals.wcm.service.relation;

import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.ResourceException;

/**
 * The type for a <code>IRelationOperand</code> . <br>
 * <code>IRelationOperandType</code> s define the type of a relation operand,
 * enabling type checking for relations.
 */
public interface IRelationOperandType {

  /**
   * Check, if this operand type supports tracking of object deletion.
   *
   * @return a <code>boolean</code> <code>true</code> , if this operand type
   *      supports the tracking of deletes for objects of this type.
   */
  public boolean isDeleteTrackingSupported();


  /**
   * Check, if this operand type supports tracking of object renaming.
   *
   * @return a <code>boolean</code> <code>true</code> , if this operand type
   *      supports the tracking of rename or move for objects of this type.
   */
  public boolean isChangeTrackingSupported();


  /**
   * Check, if this operand type supports tracking of object copy.
   *
   * @return a <code>boolean</code> <code>true</code> , if this operand type
   *      supports the tracking of copy for objects of this type.
   */
  public boolean isCopyTrackingSupported();


  /**
   * Check, if a given operand belongs to this type.
   *
   * @param operand the <code>IRelationOperand</code> which should be checked
   *      for compatibility with this type.
   * @return a <code>boolean</code> <code>true</code> , if the operand is
   *      compatible with this type.
   * @exception ResourceException Exception raised in failure situation
   * @throws ResourceException if there was an error while
   *      retrieving the operand's type.
   */
  public boolean check(IRelationOperand operand)
    throws ResourceException;


  /**
   * Get the operand for a given id and context.
   *
   * @param id the <code>String</code> with the unique id of the operand to get.
   * @param context the <code>IResouceContext</code> to use for retrieving the
   *      operand.
   * @return the <code>IRelationOperand</code> for the given id.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws InvalidArgumentException if the id is <code>null
   *      </code>.
   * @throws ResourceException if there was an error while
   *      retrieving the operand.
   */
  public IRelationOperand getOperand(String id,
    IResourceContext context)
    throws InvalidArgumentException,
    ResourceException;


  /**
   * Create the proper operand for a given object, if it belongs to this type.
   *
   * @param object TBD: Description of the incoming method parameter
   * @return the <code>IRelationOperand</code> for the given object.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws InvalidArgumentException if the given object is not
   *      compatible with this type or if it is <code>null</code> .
   * @throws ResourceException if there was an error while
   *      retrieving the operand.
   */
  public IRelationOperand getOperand(Object object)
    throws InvalidArgumentException,
    ResourceException;

}
