/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationOperandQuery.java#2 $
 */

package com.sapportals.wcm.service.relation;

import com.sapportals.wcm.repository.IPropertyName;

/**
 * A query term used for a <code>IRelationQueryOperand</code> . <br>
 *
 */
public interface IRelationOperandQuery {

  /**
   * Return an expression which selects only those operands matching the given
   * string.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery equal(String value);


  /**
   * case insensitive version of {@link #equal(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery equalIgnoreCase(String value);


  /**
   * Return an expression which selects only those operands being alphabetically
   * before the given string.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery less(String value);


  /**
   * case insensitive version of {@link #less(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery lessIgnoreCase(String value);


  /**
   * Return an expression which selects only those operands being alphabetically
   * before the given string or matching it.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery lessEqual(String value);


  /**
   * case insensitive version of {@link #lessEqual(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery lessEqualIgnoreCase(String value);


  /**
   * Return an expression which selects only those operands being alphabetically
   * after the given string.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery greater(String value);


  /**
   * case insensitive version of {@link #greater(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery greaterIgnoreCase(String value);


  /**
   * Return an expression which selects only those operands being alphabetically
   * after the given string or matching it.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery greaterEqual(String value);


  /**
   * case insensitive version of {@link #greaterEqual(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery greaterEqualIgnoreCase(String value);


  /**
   * Return an expression which selects only those operands being 'like' the
   * given string. In this string the regular SQL-wildcards (like '%' for any
   * string) are used, while '\' is used as the escape character.
   *
   * @param value the <code>String</code> to compare this term with.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery like(String value);


  /**
   * case insensitive version of {@link #like(String)}.
   *
   * @param value TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery likeIgnoreCase(String value);


  /**
   * Return an expression which selects only those between the given values,
   * such as lowerValue <= x <= upperValue.
   *
   * @param lowerValue a <code>String</code> the values have to be greater or
   *      equal to.
   * @param upperValue a <code>String</code> the values have to be less or equal
   *      to.
   * @return the <code>IRelationQuery</code> .
   */
  public IRelationQuery between(String lowerValue,
    String upperValue);


  /**
   * case insensitive version of {@link #between(String, String)}.
   *
   * @param lowerValue TBD: Description of the incoming method parameter
   * @param upperValue TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IRelationQuery betweenIgnoreCase(String lowerValue,
    String upperValue);


  /**
   * Return an expression which selects only those which match one of the given
   * queries source operand's id (subselect on source id).
   *
   * @param query the <code>IRelationQuery</code> to use for the subselect.
   * @return the <code>IRelationQuery</code> .
   * @exception RelationQueryException Exception raised in failure situation
   * @throws RelationQueryException if the given subquery is not
   *      understood by this query's builder.
   */
  public IRelationQuery inSourceSelection(IRelationQuery query)
    throws RelationQueryException;


  /**
   * Return an expression which selects only those which match one of the given
   * queries target operand's id (subselect on target id).
   *
   * @param query the <code>IRelationQuery</code> to use for the subselect.
   * @return the <code>IRelationQuery</code> .
   * @exception RelationQueryException Exception raised in failure situation
   * @throws RelationQueryException if the given subquery is not
   *      understood by this query's builder.
   */
  public IRelationQuery inTargetSelection(IRelationQuery query)
    throws RelationQueryException;


  /**
   * Return an expression which selects only those which match one of the given
   * queries property values (subselect on the given property).
   *
   * @param name the <code>IPropertyName</code> of the property to use.
   * @param query the <code>IRelationQuery</code> to use for the subselect.
   * @return the <code>IRelationQuery</code> .
   * @exception RelationQueryException Exception raised in failure situation
   * @throws RelationQueryException if the given subquery is not
   *      understood by this query's builder.
   */
  public IRelationQuery inRelationPropertySelection(IPropertyName name,
    IRelationQuery query)
    throws RelationQueryException;

}
