/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationOperand.java#2 $
 */

package com.sapportals.wcm.service.relation;
import com.sapportals.wcm.repository.AccessDeniedException;

import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.ResourceException;

/**
 * A <code>IRelation</code> s operand - the so called <i>source</i> or <i>target
 * </i> of a relation. <br>
 * <code>IRelationOperand</code> s belong to a <code>{@link
 * IRelationOperandType}</code> . The wrap the objects they reference to enable
 * type checking within the relation service. <br>
 * Each <code>IRelationOperand</code> implementation should also implement the
 * method <code>getValue()</code> to retrieve the operands wrapped object
 * without a type cast.
 */
public interface IRelationOperand {

  /**
   * Get the relation operand's type.
   *
   * @return a <code>IRelationOperandType</code> with the type of this operand.
   * @throws ResourceException an error occured while retrieving
   *      the operand's type.
   */
  public IRelationOperandType getType()
    throws ResourceException;


  /**
   * Get the relation operand's unique id.
   *
   * @return a <code>String</code> with the unique id of this operand. the id is
   *      only unique within the operand's type.
   * @throws ResourceException an error occured while retrieving
   *      the operand's id.
   */
  public String getID()
    throws ResourceException;


  /**
   * Get the relation operand's unwrapped object.
   *
   * @return an <code>Object</code> with the unwrapped operand, referenced by
   *      this <code>IRelationOperand</code> .
   */
  public Object getObject();


  /**
   * Get the relation operand as resource, if this operand is managed by a
   * repository manager.
   *
   * @return a <code>IResource</code> with the resource represented by this
   *      operand. returns <code>null</code> , if this operand can not be
   *      converted to a resource.
   * @throws ResourceException an error occured while retrieving
   *      the operand's resource.
   */
  public IResource asResource()
    throws ResourceException;


  /**
   * same as <code>IRelationService.getRelationsBySource(context, this);</code>
   *
   * @param context TBD: Description of the incoming method parameter
   * @return relationsBySource
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsBySource(IResourceContext context)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>type.getRelationBySource(this);</code>
   *
   * @param type TBD: Description of the incoming method parameter
   * @return relationsBySource
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsBySource(IRelationType type)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>IRelationService.getRelations(context, this, target);</code>
   *
   * @param context TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return relationsBySource
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsBySource(IResourceContext context,
    IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>type.getRelation(this, target);</code>
   *
   * @param type TBD: Description of the incoming method parameter
   * @param target TBD: Description of the incoming method parameter
   * @return relationBySource
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation getRelationBySource(IRelationType type,
    IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>IRelationService.getRelationsByTarget(context, this);</code>
   *
   * @param context TBD: Description of the incoming method parameter
   * @return relationsByTarget
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsByTarget(IResourceContext context)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>type.getRelationByTarget(this);</code>
   *
   * @param type TBD: Description of the incoming method parameter
   * @return relationsByTarget
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsByTarget(IRelationType type)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>IRelationService.getRelations(source, this);</code>
   *
   * @param context TBD: Description of the incoming method parameter
   * @param source TBD: Description of the incoming method parameter
   * @return relationsByTarget
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelationList getRelationsByTarget(IResourceContext context,
    IRelationOperand source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * same as <code>type.getRelation(source, this);</code>
   *
   * @param type TBD: Description of the incoming method parameter
   * @param source TBD: Description of the incoming method parameter
   * @return relationByTarget
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public IRelation getRelationByTarget(IRelationType type,
    IRelationOperand source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;

}
