/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelationList.java#1 $
 */

package com.sapportals.wcm.service.relation;

import java.util.Collection;
import java.util.Comparator;

/**
 * A typed list for <code>IRelation</code> s. <br>
 *
 */
public interface IRelationList {

  /**
   * Return this typed list as an standard java collection.
   *
   * @return a <code>Collection</code> representing this typed list.
   */
  public Collection getCollection();


  /**
   * Return the number of entries in the list.
   *
   * @return an <code>int</code> with the size of the list.
   */
  public int size();


  /**
   * Check if the specified relation is in the list.
   *
   * @param relation TBD: Description of the incoming method parameter
   * @return a <code>boolean</code> <code>true</code> m if the specified
   *      relation is in the list.
   */
  public boolean contains(IRelation relation);


  /**
   * Add a relation to the list.
   *
   * @param relation the <code>IRelation</code> to add to the list.
   * @return a <code>boolean</code> <code>true</code> , if the specified
   *      relation was added, <code>false</code> if it's already in the list.
   */
  public boolean add(IRelation relation);


  /**
   * Add all the elements in the specified list to this list.
   *
   * @param list the <code>IRelationList</code> with the elements to be added to
   *      this list.
   * @return a <code>boolean</code> <code>true</code> , if the list was changed,
   *      <code>false</code> if no element was added to the list at all.
   */
  public boolean addAll(IRelationList list);


  /**
   * Remove the first occurence of the specified relation from the list.
   *
   * @param relation TBD: Description of the incoming method parameter
   * @return a <code>boolean</code> <code>true</code> , if the specified
   *      relation was contained in the list and removed, <code>false</code> if
   *      the list was not changed.
   */
  public boolean remove(IRelation relation);


  /**
   * Return the relation at the specified position.
   *
   * @param index an <code>int</code> with the position of the desired relation.
   * @return an <code>IRelation</code> with the relation at the specified
   *      position.
   */
  public IRelation get(int index);


  /**
   * Replace the element at the specified position in this list with the
   * specified relation.
   *
   * @param index an <code>int</code> with the position of the relation to
   *      replace.
   * @param relation the <code>IRelation</code> to be stored at the specified
   *      position.
   * @return a <code>IRelation</code> with the old element that was at the
   *      specified position before replacement.
   */
  public IRelation set(int index,
    IRelation relation);


  /**
   * Insert the specified relation at the specified position in this list. The
   * relations after this position will be shifted to the right (one will be
   * added to their indices).
   *
   * @param index an <code>int</code> with the position of the relation to be
   *      inserted.
   * @param relation the <code>IRelation</code> to be inserted at the specified
   *      position.
   */
  public void add(int index,
    IRelation relation);


  /**
   * Remove the relation at the specified position in this list. The relations
   * after this position will be shifted to the left (one will be subtracted
   * from their indices).
   *
   * @param index an <code>int</code> with the position of the relation to
   *      remove.
   * @return a <code>IRelation</code> with the removed element.
   */
  public IRelation remove(int index);


  /**
   * Return the index of the first occurrence of the specified relation in this
   * list, or <code>-1</code> if the list does not contain the relation.
   *
   * @param relation the <code>IRelation</code> to search for.
   * @return an <code>int</code> with the index of the first occurrence of the
   *      specified relation, or <code>-1</code> if the list does not contain
   *      the given relation.
   */
  public int indexOf(IRelation relation);


  /**
   * Return the index of the last occurrence of the specified relation in this
   * list, or <code>-1</code> if the list does not contain the relation.
   *
   * @param relation the <code>IRelation</code> to search for.
   * @return an <code>int</code> with the index of the last occurrence of the
   *      specified relation, or <code>-1</code> if the list does not contain
   *      the given relation.
   */
  public int lastIndexOf(IRelation relation);


  /**
   * Return an iterator for the elements in this list.
   *
   * @return a <code>IRelationListIterator</code> for the elements in this list.
   */
  public IRelationListIterator listIterator();


  /**
   * Return an iterator for the elements in this list which starts at the
   * specified position in the list.
   *
   * @param index an <code>int</code> with the index to start the iterator at.
   * @return a <code>IRelationListIterator</code> for the elements in this list.
   */
  public IRelationListIterator listIterator(int index);


  /**
   * Return a list of elements from this list which starts at the first
   * specified position in the list and end at the second specified position.
   *
   * @param startIndex an <code>int</code> with the index to start the sub list
   *      at.
   * @param stopIndex an <code>int</code> with the index to stop the sub list
   *      at.
   * @return a <code>IRelationList</code> with the elements of the sub-list.
   */
  public IRelationList subList(int startIndex,
    int stopIndex);


  /**
   * Sort the elements in the list using the specified comparator.
   *
   * @param comparator the <code>Comparator</code> to be used for sorting.
   */
  public void sort(Comparator comparator);


  /**
   * Get the union of this list and the given list. <br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>a</i> , <i>b</i>
   * , <i>c</i> and <i>d</i> . Unlike as with <code>addAll()</code> duplicates
   * will be removed.
   *
   * @param list the <code>IRelationList</code> to union with this list.
   * @return the <code>IRelationList</code> with the union list.
   */
  public IRelationList union(IRelationList list);


  /**
   * Get the intersection of this list and the given list. <br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>b</i> and <i>c
   * </i>. Duplicates will be removed.
   *
   * @param list the <code>IRelationList</code> to intersect with this list.
   * @return the <code>IRelationList</code> with the intersection list.
   */
  public IRelationList intersection(IRelationList list);


  /**
   * Get the difference of this list to the given list.<br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain only <i>a</i> .
   * Duplicates will be removed.
   *
   * @param list the <code>IRelationList</code> to subtract from this list.
   * @return the <code>IRelationList</code> with the difference list.
   */
  public IRelationList difference(IRelationList list);

}
