/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_relation/java/api/com/sapportals/wcm/service/relation/IRelation.java#2 $
 */

package com.sapportals.wcm.service.relation;
import com.sapportals.wcm.repository.AccessDeniedException;
import com.sapportals.wcm.repository.IProperty;
import com.sapportals.wcm.repository.IPropertyMap;
import com.sapportals.wcm.repository.IPropertyName;
import com.sapportals.wcm.repository.IPropertyNameList;

import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.InvalidArgumentException;
import com.sapportals.wcm.repository.NotSupportedException;
import com.sapportals.wcm.repository.ResourceException;

/**
 * A relation. <br>
 *
 */
public interface IRelation {

  /**
   * Get the relation's unique id. the id is only unique with respect to the
   * relation's type.
   *
   * @return a <code>String</code> with the unique id of the relation.
   */
  public String getID();


  /**
   * Get the relation's type.
   *
   * @return the <code>IRelationType</code> of the relation.
   */
  public IRelationType getType();


  /**
   * Get the relation's source operand.
   *
   * @return the <code>IRelationOperand</code> with the wrapped source operand
   *      of the relation or <code>null</code> if the source isn't retrievable
   *      any more.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws AccessDeniedException if the operand is not gettable
   *      within the context of the relation type.
   * @throws ResourceException if there was an error while
   *      retrieving the source operand.
   */
  public IRelationOperand getSource()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Get the relation's source operand's value.
   *
   * @return the <code>Object</code> with the unwrapped source operand of the
   *      relation.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the operand is not gettable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while
   *      retrieving the source operand.
   */
  public Object getSourceOperand()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Set the relation's source operand.
   *
   * @param source the <code>IRelationOperand</code> with the wrapped operand to
   *      set as source operand for the relation.
   * @return the changed <code>IRelation</code> with the new target operand.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the source is <code>null
   *      </code>.
   * @throws  AccessDeniedException if the source is not settable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while setting
   *      the source operand.
   */
  public IRelation setSource(IRelationOperand source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Set the relation's source operand by value.
   *
   * @param source an <code>Object</code> with the operand value to set as
   *      source operand for the relation.
   * @return the changed <code>IRelation</code> with the new target operand.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the source value is
   *      <code>null</code> .
   * @throws  AccessDeniedException if the source is not settable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while setting
   *      the source operand.
   */
  public IRelation setSourceOperand(Object source)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get the relation's target operand.
   *
   * @return the <code>IRelationOperand</code> with the wrapped target operand
   *      of the relation or <code>null</code> if the target isn't retrievable
   *      any more.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the operand is not gettable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while
   *      retrieving the target operand.
   */
  public IRelationOperand getTarget()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Get the relation's target operand's value.
   *
   * @return the <code>Object</code> with the unwrapped target operand value of
   *      the relation.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the operand is not gettable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while
   *      retrieving the target operand.
   */
  public Object getTargetOperand()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Set the relation's target operand.
   *
   * @param target the <code>IRelationOperand</code> with the wrapped operand to
   *      set as target operand for the relation.
   * @return the changed <code>IRelation</code> with the new target operand.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the target is <code>null
   *      </code>.
   * @throws  AccessDeniedException if the target is not settable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while setting
   *      the target operand.
   */
  public IRelation setTarget(IRelationOperand target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Set the relation's target operand value.
   *
   * @param target the <code>Object</code> with the operand value to set as
   *      target operand for the relation.
   * @return the changed <code>IRelation</code> with the new target operand.
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  InvalidArgumentException if the target is <code>null
   *      </code>.
   * @throws  AccessDeniedException if the target is not settable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error while setting
   *      the target operand.
   */
  public IRelation setTargetOperand(Object target)
    throws InvalidArgumentException,
    AccessDeniedException,
    ResourceException;


  /**
   * Get all properties of the relation.
   *
   * @return a <code>IPropertyMap</code> with the references to all properties
   *      of the relation.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the properties are not
   *      gettable within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public IPropertyMap getProperties()
    throws AccessDeniedException,
    ResourceException;


  /**
   * Get several specified properties.
   *
   * @param propertyNameList a <code>IPropertyNameList</code> as a list with the
   *      names of the requested properties.
   * @return a <code>IPropertyMap</code> with the references to the requested
   *      and exisiting properties. The list might be empty if no properties
   *      with the given names where found. if the given list of names was
   *      <code>null</code> all properties are selected.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the properties are not
   *      gettable within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public IPropertyMap getProperties(IPropertyNameList propertyNameList)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Set several specified properties. Properties which already exist will
   * change their values, those which do not will be added. Nothing happens, if
   * the given properties are <code>null</code> .
   *
   * @param properties a <code>IPropertyMap</code> as a list of properties to
   *      add or change.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the properties are not
   *      gettable within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public void setProperties(IPropertyMap properties)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Delete several specified properties (if they exist). Nothing happens if the
   * given list is <code>null</code> .
   *
   * @param propertyNameList a <code>IPropertyNameList</code> as a list with the
   *      names of the properties to delete.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the properties are not
   *      deleteable within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public void deleteProperties(IPropertyNameList propertyNameList)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Get a specific property.
   *
   * @param propertyName a <code>IPropertyName</code> with the name of the
   *      property to read.
   * @return a <code>IProperty</code> with a reference to the property or <code>
   *      null</code> if the property was not found or the given property name
   *      was <code>null</code> .
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the property is not gettable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public IProperty getProperty(IPropertyName propertyName)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Set a specific property. If the property does not exist it will be added,
   * otherwise it's value is changed. If the given property is <code>null</code>
   * , nothing happens.
   *
   * @param property the <code>IProperty</code> to set.
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the property is not settable
   *      within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public void setProperty(IProperty property)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Delete a specific property. If the property does not exist or is <code>null
   * </code>, the method will return without doing anything.
   *
   * @param propertyName TBD: Description of the incoming method parameter
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the property is not
   *      deleteable within the context of the relation type.
   * @throws  ResourceException if there was an error in the
   *      relation manager.
   */
  public void deleteProperty(IPropertyName propertyName)
    throws AccessDeniedException,
    ResourceException;


  /**
   * Delete a given relation.
   *
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  AccessDeniedException if the given context (as passed
   *      when retrieving the relation type) has no permissions to delete the
   *      relation.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void delete()
    throws AccessDeniedException,
    ResourceException;


  // ----------------
  // IOrderedRelation ---------------------------------------------------------
  // ----------------

  /**
   * Get the relation's positioning. (only supported for <code>IOrderedRelation
   * </code>s).
   *
   * @return the <code>IOrderedRelationPosition</code> of the relation. is
   *      <code>OrderedRelationType.LAST</code> , if it's the last relation with
   *      respect to the relations type and source, otherwise the type is <code>
   *      OrderType.AFTER</code> and the relative is the predecessors id.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  NotSupportedException if this relation does not
   *      belong to a ordered relation type.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public IOrderedRelationPosition getPosition()
    throws NotSupportedException,
    ResourceException;


  /**
   * Change the relation's position (only supported for <code>IOrderedRelation
   * </code>s).
   *
   * @param position the new <code>IOrderedRelationPosition</code> for this
   *      relation. if <code>null<code>, the relation's position remains
   *      unchanged.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception InvalidArgumentException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @throws  NotSupportedException if this relation does not
   *      belong to a ordered relation type.
   * @throws  InvalidArgumentException if the given position can
   *      not be used for this relation.
   * @throws  ResourceException if there was an internal error in
   *      the relation manager.
   */
  public void setPosition(IOrderedRelationPosition position)
    throws NotSupportedException,
    InvalidArgumentException,
    ResourceException;

}
