header {
package com.togethersoft.sca.internal.jparser.parser;

import com.togethersoft.sca.ast.AstCaseLabel;
import com.togethersoft.sca.internal.jparser.ast.*;
import com.togethersoft.sca.ast.AstModifiers;
import com.togethersoft.sca.ast.AstTypeReference;

import java.util.ArrayList;
}

/** Java 1.2 Recognizer
 *
 * Contributing authors:
 *		John Mitchell		johnm@non.net
 *		Terence Parr		parrt@magelang.com
 *		John Lilley			jlilley@empathy.com
 *		Scott Stanchfield	thetick@magelang.com
 *		Markus Mohnen       mohnen@informatik.rwth-aachen.de
 *      Peter Williams      pete.williams@sun.com
 *      Allan Jacobs        Allan.Jacobs@eng.sun.com
 *      Steve Messick       messick@redhills.com
 *
 * Version 1.00 December 9, 1997 -- initial release
 * Version 1.01 December 10, 1997
 *      fixed bug in octal def (0..7 not 0..8)
 * Version 1.10 August 1998 (parrt)
 *      added tree construction
 *      fixed definition of WS,comments for mac,pc,unix newlines
 *      added unary plus
 * Version 1.11 (Nov 20, 1998)
 *      Added "shutup" option to turn off last ambig warning.
 *      Fixed inner class def to allow named class defs as statements
 *      synchronized requires compound not simple statement
 *      add [] after builtInType DOT class in primaryExpression
 *      "const" is reserved but not valid..removed from modifiers
 * Version 1.12 (Feb 2, 1999)
 *      Changed LITERAL_xxx to xxx in tree grammar.
 *      Updated java.g to use tokens {...} now for 2.6.0 (new feature).
 *
 * Version 1.13 (Apr 23, 1999)
 *      Didn't have (stat)? for else clause in tree parser.
 *      Didn't gen ASTs for interface extends.  Updated tree parser too.
 *      Updated to 2.6.0.
 * Version 1.14 (Jun 20, 1999)
 *      Allowed final/abstract on local classes.
 *      Removed local interfaces from methods
 *      Put instanceof precedence where it belongs...in relationalExpr
 *          It also had expr not type as arg; fixed it.
 *      Missing ! on SEMI in classBlock
 *      fixed: (expr) + "string" was parsed incorrectly (+ as unary plus).
 *      fixed: didn't like Object[].class in parser or tree parser
 * Version 1.15 (Jun 26, 1999)
 *      Screwed up rule with instanceof in it. :(  Fixed.
 *      Tree parser didn't like (expr).something; fixed.
 *      Allowed multiple inheritance in tree grammar. oops.
 * Version 1.16 (August 22, 1999)
 *      Extending an interface built a wacky tree: had extra EXTENDS.
 *      Tree grammar didn't allow multiple superinterfaces.
 *      Tree grammar didn't allow empty var initializer: {}
 * Version 1.17 (October 12, 1999)
 *      ESC lexer rule allowed 399 max not 377 max.
 *      java.tree.g didn't handle the expression of synchronized
 *          statements.
 * Version 1.18 (August 12, 2001)
 *      Terence updated to Java 2 Version 1.3 by observing/combining work of
 *      Allan Jacobs and Steve Messick.  Handles 1.3 src.
 *		Summary:
 *		o  primary didn't include boolean.class kind of thing
 *      o  constructor calls parsed explicitly now:
 * 		   see explicitConstructorInvocation
 *		o  add strictfp modifier
 *      o  missing objBlock after new expression in tree grammar
 *		o  merged local class definition alternatives, moved after declaration
 *		o  fixed problem with ClassName.super.field
 *      o  reordered some alternatives to make things more efficient
 *		o  long and double constants were not differentiated from int/float
 *		o  whitespace rule was inefficient: matched only one char
 *		o  add an examples directory with some nasty 1.3 cases
 *		o  made Main.java use buffered IO and a Reader for Unicode support
 *		o  supports UNICODE?
 *		   Using Unicode charVocabulay makes code file big, but only
 *		   in the bitsets at the end. I need to make ANTLR generate
 *		   unicode bitsets more efficiently.
 *
 * This grammar is in the PUBLIC DOMAIN
 */

class JavaRecognizer extends Parser;
options {
    k = 2;                           // two token lookahead
    exportVocab=Java;                // Call its vocabulary "Java"
    codeGenMakeSwitchThreshold = 2;  // Some optimizations
    codeGenBitsetTestThreshold = 3;
    defaultErrorHandler = false;     // Don't generate parser error handlers
}

tokens {
    BLOCK; MODIFIERS; OBJBLOCK; SLIST; CTOR_DEF; METHOD_DEF; VARIABLE_DEF; 
    INSTANCE_INIT; STATIC_INIT; TYPE; CLASS_DEF; INTERFACE_DEF; 
    PACKAGE_DEF; ARRAY_DECLARATOR; EXTENDS_CLAUSE; IMPLEMENTS_CLAUSE;
    PARAMETERS; PARAMETER_DEF; LABELED_STAT; TYPECAST; INDEX_OP; 
    POST_INC; POST_DEC; METHOD_CALL; EXPR; ARRAY_INIT; 
    IMPORT; UNARY_MINUS; UNARY_PLUS; CASE_GROUP; ELIST; FOR_INIT; FOR_CONDITION; 
    FOR_ITERATOR; EMPTY_STAT; FINAL="final"; ABSTRACT="abstract";
	SUPER_CTOR_CALL; CTOR_CALL; COMMENT; AUTOFIX_TOKEN;
}

{
    JavaLexer lexer;

    public void setLexer(JavaLexer lexer) {
	this.lexer = lexer;
    }

    boolean deep = false;
    
    public void setDeep(boolean deep) {
	this.deep = deep;
    }
}
    
// Compilation Unit: In Java, this is a single file.  This is the start
//   rule for this parser
compilationUnit returns [CompilationUnit cu = null]
    {
	ImportDeclaration i;
	TypeDeclaration t;
	Name pkg;
	cu = new CompilationUnit(!deep);
	boolean filecomment = false;
    }
    :   // A compilation unit starts with an optional package definition
        (   packageDefinition[cu]
        |   /* nothing */
        )

        // Next we have a series of zero or more import statements
        (   i = importDefinition 
	    	{  cu.addImportDeclaration(i); } 
		)*

        // Wrapping things up with any number of class or interface
        //    definitions
        (   t = typeDefinition 
	    	{ cu.addTypeDeclaration(t); } 
		)*

        eof:EOF
        { cu.eofToken = (JavaToken)eof; }
    ;


// Package statement: "package" followed by an identifier.
packageDefinition[CompilationUnit cu]
    options {defaultErrorHandler = true;} // let ANTLR handle errors
	{ Name pkg = null; }
    :   tkn:"package" pkg = identifier semi:SEMI
	    { 
	    	cu.setPackage(pkg); 
	    	cu.packageToken = (JavaToken)tkn; 
	    	cu.semicolon = (JavaToken)semi;
	    }
    ;

// Import statement: import followed by a package or class name
importDefinition returns [ImportDeclaration imp = null]
    options {defaultErrorHandler = true;}
    :   impToken:"import" imp = identifierStar 
	{ imp.importToken = (JavaToken)impToken; }
	SEMI
    ;

// A type definition in a file is either a class or interface definition.
typeDefinition returns [TypeDeclaration type = null]
    options {defaultErrorHandler = true;}
    { Modifiers mods; }
    :   mods = modifiers
        ( type = classDefinition[mods]
        | type = interfaceDefinition[mods]
        )
    |   SEMI
    ;

/** A declaration is the creation of a reference or primitive-type variable
 *  Create a separate Type/Var tree for each var in the var list.
 */
declaration[] returns [VariableDeclaration v]
    { 
	Modifiers mods; 
	TypeExpr  type;
    }
    :   mods = modifiers type = typeSpec v = variableDefinitions[mods, type]
    ;

// A list of zero or more modifiers.  We could have used (modifier)* in
//   place of a call to modifiers, but I thought it was a good idea to keep
//   this rule separate so they can easily be collected in a Vector if
//   someone so desires
modifiers returns [Modifiers mods = null]
    { Modifier m; }
    :   (   m = modifier 
	    { if (mods == null) mods = new Modifiers(); mods.addModifier(m); }
	)*
    ;


// A type specification is a type name with possible brackets afterwards
//   (which would make it an array type).
typeSpec returns [TypeExpr t = null]
    : t = classTypeSpec
    | t = builtInTypeSpec
    ;

// A class type specification is a class type with possible brackets afterwards
//   (which would make it an array type).
classTypeSpec returns [TypeExpr type = null]
    { Name t; }
    :   t = identifier 
	{ type = new TypeExpr(t); }
	(
	    lb:LBRACK rb:RBRACK
	    { type.addBrackets(new Brackets((JavaToken)lb, (JavaToken)rb)); }
	)*
    ;

// A builtin type specification is a builtin type with possible brackets
// afterwards (which would make it an array type).
builtInTypeSpec returns [TypeExpr type = null]
    { Name t; }
    :   t = builtInType
	{ type = new TypeExpr(t); }
	(
	    lb:LBRACK rb:RBRACK
	    { type.addBrackets(new Brackets((JavaToken)lb, (JavaToken)rb)); }
	)*
    ;

// A type name. which is either a (possibly qualified) class name or
//   a primitive (builtin) type
type returns [Name t = null]
    :   t = identifier
    |   t = builtInType
    ;

// The primitive types.
builtInType returns [BuiltInType t = null]
    :   id1:"void"
	{ t = new BuiltInType(AstTypeReference.TID_VOID, (JavaToken)id1); }
    |   id2:"boolean"
	{ t = new BuiltInType(AstTypeReference.TID_BOOLEAN, (JavaToken)id2); }
    |   id3:"byte"
	{ t = new BuiltInType(AstTypeReference.TID_BYTE, (JavaToken)id3); }
    |   id4:"char"
	{ t = new BuiltInType(AstTypeReference.TID_CHAR, (JavaToken)id4); }
    |   id5:"short"
	{ t = new BuiltInType(AstTypeReference.TID_SHORT, (JavaToken)id5); }
    |   id6:"int"
	{ t = new BuiltInType(AstTypeReference.TID_INT, (JavaToken)id6); }
    |   id7:"float"
	{ t = new BuiltInType(AstTypeReference.TID_FLOAT, (JavaToken)id7); }
    |   id8:"long"
	{ t = new BuiltInType(AstTypeReference.TID_LONG, (JavaToken)id8); }
    |   id9:"double"
	{ t = new BuiltInType(AstTypeReference.TID_DOUBLE, (JavaToken)id9); }
    ;

// A (possibly-qualified) java identifier.  We start with the first IDENT
//   and expand its name by adding dots and following IDENTS
identifier returns [Name id = null]
    :   id1:IDENT 
	{ id = new SimpleName((JavaToken)id1 ); }
	(   dot:DOT id2:IDENT
	    { id = new FieldAccess(id, FieldAccess.SIMPLE, (JavaToken)dot, (JavaToken)id2); }
	)*
    ;

identifierStar returns [ImportDeclaration imp = null]
    { Name id = null;
      boolean star = false; }
    :   id1:IDENT 
	{ id = new SimpleName((JavaToken)id1); }
        ( dot:DOT id2:IDENT
	    { id = new FieldAccess(id, FieldAccess.SIMPLE, (JavaToken)dot, (JavaToken)id2); } 
	)*
        ( DOT STAR { star = true; } )?
	{ imp = new ImportDeclaration(id, star); }
    ;


// modifiers for Java classes, interfaces, class/instance vars and methods
modifier returns [Modifier m = null]
    :   m1:"private"       
		{ m = new Modifier(AstModifiers.ACC_PRIVATE, (JavaToken)m1); }
    |   m2:"public"
		{ m = new Modifier(AstModifiers.ACC_PUBLIC, (JavaToken)m2); }
    |   m3:"protected"
		{ m = new Modifier(AstModifiers.ACC_PROTECTED, (JavaToken)m3); }
    |   m4:"static"
		{ m = new Modifier(AstModifiers.ACC_STATIC, (JavaToken)m4); }
    |   m5:"transient"
		{ m = new Modifier(AstModifiers.ACC_TRANSIENT, (JavaToken)m5); }
    |   m6:"final"
		{ m = new Modifier(AstModifiers.ACC_FINAL, (JavaToken)m6); }
    |   m7:"abstract"
		{ m = new Modifier(AstModifiers.ACC_ABSTRACT, (JavaToken)m7); }
    |   m8:"native"
		{ m = new Modifier(AstModifiers.ACC_NATIVE, (JavaToken)m8); }
    |   m9:"synchronized"
		{ m = new Modifier(AstModifiers.ACC_SYNCHRONIZED, (JavaToken)m9); }
    |   m10:"volatile"
		{ m = new Modifier(AstModifiers.ACC_VOLATILE, (JavaToken)m10); }
    |   m11:"strictfp"
		{ m = new Modifier(AstModifiers.ACC_STRICTFP, (JavaToken)m11); }
//  |   "const"        // reserved word, but not valid
//  |   "threadsafe"
    ;


// Definition of a Java class
classDefinition[Modifiers mods] returns [TypeDeclaration cl = null]
    :   classToken:"class" id:IDENT
    	{ 
		    cl = new TypeDeclaration(TypeDeclaration.CLASS, (JavaToken)classToken, 
					     (JavaToken)id, mods);
    	}
        // it _might_ have a superclass...
        superClassClause[cl]
        // it might implement some interfaces...
        implementsClause[cl]
        // now parse the body of the class
        classBlock[cl]
    ;

superClassClause[TypeDeclaration type]
    { Name id; }
    :   ( extTkn:"extends" id = identifier 
    		{   type.extendsToken = (JavaToken)extTkn; 
    			type.setSuperType(new Inheritance(id, false)); 
    		} 
    	)?
    ;

// Definition of a Java Interface
interfaceDefinition[Modifiers mods] returns [TypeDeclaration ifc = null]
    :   classToken:"interface" id:IDENT
    	{ 
		    ifc = new TypeDeclaration(TypeDeclaration.INTERFACE, (JavaToken)classToken, 
						 (JavaToken)id, mods);
    	}
        // it might extend some other interfaces
        interfaceExtends[ifc]
        // now parse the body of the interface (looks like a class...)
        classBlock[ifc]
    ;


// This is the body of a class.  You can have fields and extra semicolons,
// That's about it (until you see what a field is...)
classBlock[TypeDeclaration cl]
    :   lb:LCURLY
            ( field[cl] | SEMI )*
        rb:RCURLY
	{ 
		cl.leftBrace = (JavaToken)lb; 
		cl.rightBrace = (JavaToken)rb; 
	}
    ;

// An interface can extend several other interfaces...
interfaceExtends[TypeDeclaration type]
    { Name id; AstVector im = null; }
    :   ( extTkn:"extends" id = identifier 
			{ 
				type.extendsToken = (JavaToken)extTkn;
				im = new AstVector(); im.add(new Inheritance(id, false)); 
			}
			( COMMA id = identifier { im.add(new Inheritance(id, false)); } )*
        )?
        {
        	type.setSuperInterfaces(im);
        }
    ;

// A class can implement several interfaces...
implementsClause[TypeDeclaration type]
    { Name id; AstVector im = null; }
    :   ( implTkn:"implements" id = identifier 
	    	{ 
	    		type.implementsToken = (JavaToken)implTkn;
	    		im = new AstVector(); im.add(new Inheritance(id, true)); 
	    	}
	    	( COMMA id = identifier { im.add(new Inheritance(id, true)); } )*
        )?
        {
        	type.setSuperInterfaces(im);
        }
    ;

// Now the various things that can be defined inside a class or interface...
// Note that not all of these are really valid in an interface (constructors,
//   for example), and if this grammar were used for a compiler there would
//   need to be some semantic checks to make sure we're doing the right thing...
field[TypeDeclaration cb]
	{
		VariableDeclaration v;
		TypeDeclaration     t;
	    Modifiers           mods;
		TypeExpr            type;
		Block               bl;
		MethodDeclaration   method;
		Stmt                methodBody = null;
		AstVector           params;	
		AstVector           brackets;
		ClassInitializer    clinit;
		AstVector           throwSpec = null;
	}
    :   // method, constructor, or variable declaration
        mods = modifiers
        (   method = ctorDeclaration[mods] // constructor
	    	{ cb.addConstructor(method); }

        |   t = classDefinition[mods]     // inner class
	    	{ cb.addNestedType(t); }
            
        |   t = interfaceDefinition[mods] // inner interface
	    	{ cb.addNestedType(t); }

        |   type = typeSpec   // method or variable declaration(s)
            (   id:IDENT  // the name of the method

                // parse the formal parameter declarations.
                lp:LPAREN params = parameterDeclarationList rp:RPAREN

                brackets = declaratorBrackets

                // get the list of exceptions that this method is declared to throw
                (thrToken:"throws" throwSpec = throwsClause)?

                ( methodBody = compoundStatement[] | methodBody = emptyStmt[] )
				{
				    method = new MethodDeclaration(mods, type, id.getText(),
								   (JavaToken)lp, params, (JavaToken)rp, brackets, 
								   (JavaToken)thrToken, throwSpec, methodBody,
								   (JavaToken)id, false, true);
				    cb.addMethod(method);
				}
            |   v = variableDefinitions[mods, type] SEMI
				{ cb.addField(v); }
            )
        )

    // "static { ... }" class initializer
    |   id2:"static" bl = compoundStatement[]
	{ 
		mods = new Modifiers();
		mods.addModifier(new Modifier(AstModifiers.ACC_STATIC, (JavaToken)id2));
	    clinit = new ClassInitializer(mods, bl, true);
	    cb.addClassInitializer(clinit); 
	}

    // "{ ... }" instance initializer
    |   bl = compoundStatement[]
	{ 
	    clinit = new ClassInitializer(null, bl, true);
	    cb.addClassInitializer(clinit); 
	}
    ;

variableDefinitions[Modifiers mods, TypeExpr type] returns [VariableDeclaration v]
    {
		VariableDeclarator d; 
		AstVector declList = new AstVector();
		v = new VariableDeclaration(mods, type); 
    }
    :   d = variableDeclarator[v] { declList.add(d); }
        (   COMMA
            d = variableDeclarator[v] { declList.add(d); }
        )*
        { v.setDeclarators(declList); }
    ;

/** Declaration of a variable.  This can be a class/instance variable,
 *   or a local variable in a method
 * It can also include possible initialization.
 */
variableDeclarator[VariableDeclaration decl] returns [VariableDeclarator v = null]
    { 
	Expr init = null;
	AstVector brackets;
    }
    :   id:IDENT brackets = declaratorBrackets 
    	( as:ASSIGN init = initializer )?
	{ v = new VariableDeclarator(decl, (JavaToken)id, (JavaToken)as, init, brackets); }
    ;

declaratorBrackets returns [AstVector brackets = null]
    :   (
	    lb:LBRACK rb:RBRACK
	    {
	    	if (brackets == null) {
		    	brackets = new AstVector();
	    	}
	    	brackets.add(new Brackets((JavaToken)lb, (JavaToken)rb));
	    }
	)*
    ;

// This is an initializer used to set up an array.
arrayInitializer returns[ArrayInitializer arrayInit]
    { 
	Expr init;
	arrayInit = new ArrayInitializer(); 
    }
    :   lc:LCURLY
            (   init = initializer
		{ arrayInit.addInitializer(init); }
                (
                    // CONFLICT: does a COMMA after an initializer start a new
                    //           initializer or start the option ',' at end?
                    //           ANTLR generates proper code by matching
                    //           the comma as soon as possible.
                    options {
                        warnWhenFollowAmbig = false;
                    }
                :
                    COMMA init = initializer
		    { arrayInit.addInitializer(init); }
                )*
                (COMMA)?
            )?
        rc:RCURLY
	{
	    arrayInit.leftBrace  = (JavaToken)lc;
	    arrayInit.rightBrace = (JavaToken)rc;
	}
    ;


// The two "things" that can initialize an array element are an expression
//   and another (nested) array initializer.
initializer returns[Expr init = null]
    :   init = expression
    |   init = arrayInitializer
    ;

// This is the header of a method.  It includes the name and parameters
//   for the method.
//   This also watches for a list of exception classes in a "throws" clause.
ctorDeclaration[Modifiers mods] returns [MethodDeclaration ctor = null]
    {
	Block  body;
	AstVector params, throwSpec = null;	
    }
    :   id:IDENT  // the name of the method

        // parse the formal parameter declarations.
        lp:LPAREN params = parameterDeclarationList rp:RPAREN

        // get the list of exceptions that this method is declared to throw
        (thrToken:"throws" throwSpec = throwsClause)?

	body = constructorBody[]
	{
	    ctor = new MethodDeclaration(mods, null, id.getText(), (JavaToken)lp, 
	    	params, (JavaToken)rp, null, (JavaToken)thrToken, throwSpec, body, 
	    	(JavaToken)id, true, true);
	}
    ;

constructorBody[] returns [Block body = null]
    { 
	Stmt s; 
	ArrayList block = null;
      	if (deep) { block = new ArrayList(); }
    }
    :   lb:LCURLY
			{ if (!deep) { lexer.skipBody(); return null; } }
		// Predicate might be slow but only checked once per constructor def
		// not for general methods.
		( (explicitConstructorInvocation[]) => s = explicitConstructorInvocation[]
			{ block.add(s); } 
		|
		)
        ( s = statement[] { block.add(s); } )*
        rb:RCURLY
	{ 
        body = new Block(
		    (JavaToken)lb, 
		    (Stmt[])block.toArray(new Stmt[block.size()]), 
	    	(JavaToken)rb); 
	}
    ;

explicitConstructorInvocation[] returns [Stmt s = null]
	{ Expr e, base; AstVector args; }
    :   (	options {
				// this/super can begin a primaryExpression too; with finite
				// lookahead ANTLR will think the 3rd alternative conflicts
				// with 1, 2.  I am shutting off warning since ANTLR resolves
				// the nondeterminism by correctly matching alts 1 or 2 when
				// it sees this( or super(
				generateAmbigWarnings=false;
			}
		:	thisTkn:"this" lp1:LPAREN args = argList rp1:RPAREN sm1:SEMI
			{
				e = new ThisCall(new ThisExpr((JavaToken)thisTkn), 
							     (JavaToken)lp1, args, (JavaToken)rp1);
				s = new ExprStmt(e, (JavaToken)sm1);
			}

	    |   superTkn1:"super" lp2:LPAREN args = argList rp2:RPAREN sm2:SEMI
			{
				e = new SuperCall(null, null, new SuperExpr((JavaToken)superTkn1), 
								  (JavaToken)lp2, args, (JavaToken)rp2);
				s = new ExprStmt(e, (JavaToken)sm2);
			}

			// (new Outer()).super()  (create enclosing instance)
		|	base = primaryExpression dotTkn:DOT superTkn2:"super" 
				lp3:LPAREN args = argList rp3:RPAREN sm3:SEMI
			{
				e = new SuperCall(base, (JavaToken)dotTkn, new SuperExpr((JavaToken)superTkn2), 
								  (JavaToken)lp3, args, (JavaToken)rp3);
				s = new ExprStmt(e, (JavaToken)sm3);
			}
		)
    ;

// This is a list of exception classes that the method is declared to throw
throwsClause returns [AstVector v = null]
    { Name id; v = new AstVector(); }
    :   id = identifier { v.add(new ThrowSpecifier(id)); } 
	( COMMA id = identifier { v.add(new ThrowSpecifier(id)); } )*
    ;

// A list of formal parameters
parameterDeclarationList returns [AstVector v]
    {
	FormalParameter p;
	v = new AstVector(); 
    }
    :   ( 
	    p = parameterDeclaration { v.add(p); }
	    ( COMMA p = parameterDeclaration { v.add(p); } )* 
	)?
    ;

// A formal parameter.
parameterDeclaration returns [FormalParameter p = null]
    {
	TypeExpr           type;
	Modifiers          mods;
	VariableDeclarator v;
	AstVector          brackets;
    }
    :   mods = modifiers type = typeSpec id:IDENT
        brackets = declaratorBrackets
	{ 
	    p = new FormalParameter(mods, type);
	    v = new VariableDeclarator(p, (JavaToken)id, null, null, brackets);
	    p.declarator = v;
	}
    ;

// Compound statement.  This is used in many contexts:
//   Inside a class definition prefixed with "static":
//      it is a class initializer
//   Inside a class definition without "static":
//      it is an instance initializer
//   As the body of a method
//   As a completely indepdent braced block of code inside a method
//      it starts a new scope for variable definitions

compoundStatement[] returns [Block b = null]
    { 
		Stmt s; 
		ArrayList block = null;
      	if (deep) { block = new ArrayList(); }
    }
    :   lb:LCURLY
		{ if (!deep) { lexer.skipBody(); return null; } }
        ( 
			// include the (possibly-empty) list of statements
	    	s = statement[] 
	    	{ block.add(s); } 
		)*
        rb:RCURLY
	{ 
	    b = new Block(
			(JavaToken)lb, 
			(Stmt[])block.toArray(new Stmt[block.size()]), 
			(JavaToken)rb); 
	}
    ;

statement[] returns [Stmt s = null]
    { 
		Expr e;
		Modifiers mods = null; 
		Declaration decl;
    }
    :   
    (
    // A list of statements in curly braces -- start a new scope!
	s = compoundStatement[]
	
    // declarations are ambiguous with "ID DOT" relative to expression
    // statements.  Must backtrack to be sure.  Could use a semantic
    // predicate to test symbol table to see what the type was coming
    // up, but that's pretty hard without a symbol table ;)
    |   (declaration[])=> decl = declaration[] semi1:SEMI
	{ s = new DeclarationStmt(decl, (JavaToken) semi1); }

    // An expression statement.  This could be a method call,
    // assignment statement, or any other expression evaluated for
    // side-effects.
    |   e = expression id3:SEMI { s = new ExprStmt(e,(JavaToken)id3); }

     // class definition
    |   mods = modifiers decl = classDefinition[mods]
	{ s = new DeclarationStmt(decl, null); }

    // Attach a label to the front of a statement
    |   label:IDENT id4:COLON s = statement[] 
	{ s.addLabel(new SimpleName((JavaToken)label)); }

    // If-else statement
    |   s = ifStmt[]

    // For statement
    |	s = forStmt[]

    // While statement
    |   s = whileStmt[]

    // do-while statement
    |   s = doStmt[]

    // get out of a loop (or switch)
    |   s = breakStmt[]

    // do next iteration of a loop
    |   s = continueStmt[]

    // Return an expression
    |	s = returnStmt[]

    // switch/case statement
    |	s = switchStmt[]

    // exception try-catch block
    |   s = tryBlock[]

    // throw an exception
    |	s = throwStmt[]

    // synchronize a statement
    |	s = syncStmt[]

    // empty statement
    |   s = emptyStmt[]
    
    // assertion statement
    |   s = assertStmt[]
    )
    ;

// If-else statement
ifStmt[] returns [IfStmt s = null]
    {
	Expr cond;
	Stmt thenStmt, elseStmt = null;
    }
    :   id:"if" lp:LPAREN cond = expression rp:RPAREN thenStmt = statement[]
        (
            // CONFLICT: the old "dangling-else" problem...
            //           ANTLR generates proper code matching
            //           as soon as possible.  Hush warning.
            options {
                warnWhenFollowAmbig = false;
            }
        :
            elseTkn:"else" elseStmt = statement[]
        )?
	{ s = new IfStmt((JavaToken)id, new ParExpr((JavaToken) lp, cond, (JavaToken) rp), 
		thenStmt, (JavaToken)elseTkn, elseStmt); }
    ;

// While statement
whileStmt[] returns [WhileStmt s = null]
    { Expr cond; Stmt body; }
    :	id:"while" lp:LPAREN cond = expression rp:RPAREN body = statement[]
	{ s = new WhileStmt((JavaToken)id, new ParExpr((JavaToken) lp, cond, (JavaToken) rp), body);	}
    ;

// do-while statement
doStmt[] returns [DoStmt s = null]
    { Expr cond; Stmt body; }
    :   id:"do" body = statement[] whileTkn:"while" 
	lp:LPAREN cond = expression rp:RPAREN semi:SEMI
	{ s = new DoStmt((JavaToken)id, body, new ParExpr((JavaToken) lp, cond, (JavaToken) rp), 
		(JavaToken)whileTkn, (JavaToken)semi); }
    ;

// Break statement
breakStmt[] returns [BreakStmt s = null]
    { SimpleName label = null; }
    :   id:"break" ( l:IDENT { label = new SimpleName((JavaToken)l); } )? semi:SEMI
	{ s = new BreakStmt((JavaToken)id, label, (JavaToken)semi); }
    ;

// Continue statement
continueStmt[] returns [ContinueStmt s = null]
    { SimpleName label = null; }
    :   id:"continue" ( l:IDENT { label = new SimpleName((JavaToken)l); } )? semi:SEMI
	{ s = new ContinueStmt((JavaToken)id, label, (JavaToken)semi); }
    ;

// Return statement
returnStmt[] returns [ReturnStmt s = null]
    { Expr e = null; }
    :   id:"return" ( e = expression )? semi:SEMI
	{	s = new ReturnStmt((JavaToken)id, e, (JavaToken)semi); }
    ;

// Throw statement
throwStmt[] returns [ThrowStmt s = null]
    { Expr e; }
    :   id:"throw" e = expression semi:SEMI
	{ s = new ThrowStmt((JavaToken)id, e, (JavaToken)semi); }
    ;

// synchronized statement
syncStmt[] returns [SynchronizedStmt s = null]
    { Expr cond; Block body; }
    :   id:"synchronized" 
	lp:LPAREN cond = expression rp:RPAREN 
	body = compoundStatement[]
	{ s = new SynchronizedStmt((JavaToken)id, new ParExpr(
		(JavaToken)lp, cond, (JavaToken)rp), body); }
    ;

switchStmt[] returns [SwitchStmt s = null]
    { Expr cond; AstVector v = new AstVector(); SwitchGroup gr = null, def = null; }
    :   id:"switch" lp:LPAREN cond = expression rp:RPAREN 
	lb:LCURLY
            (	gr = casesGroup[] { v.add(gr); if (gr.getDefaultLabel() != null) def = gr; }
	    )*
        rb:RCURLY
	{ s = new SwitchStmt((JavaToken)id, new ParExpr((JavaToken) lp, cond, (JavaToken) rp), 
		(JavaToken)lb, v, (JavaToken)rb, def); }
    ;

casesGroup[] returns [SwitchGroup ret = null]
    { 
		AstCaseLabel l, def = null;
		AstVector labels = new AstVector();
		Stmt s;
		AstVector stmts = new AstVector();
    }
    :   (   // CONFLICT: to which case group do the statements bind?
            //           ANTLR generates proper code: it groups the
            //           many "case"/"default" labels together then
            //           follows them with the statements
            options {
                warnWhenFollowAmbig = false;
            }
            :
            l = aCase
	    { labels.add(l); if (l.isDefault()) def = l; }
        )+
        ( 
	    s = statement[]
	    { stmts.add(s); } 
	)*
	{ 	ret = new SwitchGroup(labels, stmts, def); }
    ;

aCase returns[AstCaseLabel l = null]
    { Expr e; }
    :   (
	    id1:"case" e = expression id2:COLON 
	    { l = new CaseLabel((JavaToken)id1, e, (JavaToken)id2); }
	| 
	    id3:"default" id4:COLON
	    { l = new DefaultLabel((JavaToken)id3, (JavaToken)id4); }
	) 
    ;

// For statement
forStmt[] returns [ForStmt s = null]
    { 
	AstVector inits, updates;
	Expr cond;
	Stmt body;
    }
    :   id:"for"
            lp:LPAREN
                inits   = forInit[] SEMI   // initializer
                cond    = forCond[] SEMI   // condition test
                updates = forIter[]        // updater
            rp:RPAREN
            body = statement[]      // statement to loop over
	{ s = new ForStmt((JavaToken)id, (JavaToken)lp, inits, cond, updates, (JavaToken)rp, body); }
    ;

// The initializer for a for loop
forInit[] returns [AstVector v = null]
    { 
	DeclarationStmt s = null; 
	Declaration decl;
    }
        // if it looks like a declaration, it is
    :   (   (declaration[])=> decl = declaration[] 
	    { s = new DeclarationStmt(decl, null); 
	      v = new AstVector(); v.add(s); }
        // otherwise it could be an expression list...
        |   v = exprStmtList
        )?
    ;

forCond[] returns [Expr e = null]
    :   ( e = expression )?
    ;

forIter[] returns [AstVector v = null]
    :   ( v = exprStmtList )?
    ;

// an exception handler try/catch block
tryBlock[] returns [TryStmt s = null]
    { 
	Block b, fb;
	CatchClause c;
	AstVector vc = new AstVector();
	FinallyClause f = null;
    }
    :   id1:"try" b = compoundStatement[]
        ( c = handler[] { vc.add(c); } )*
        (   id2:"finally" fb = compoundStatement[]
	    { f = new FinallyClause((JavaToken)id2, fb);  } 
	)?
	{ s = new TryStmt((JavaToken)id1, b, vc, f); }
    ;


// an exception handler
handler[] returns [CatchClause s = null]
    { 
	Block b; 
	FormalParameter p;
    }
    :   id:"catch" lp:LPAREN p = parameterDeclaration rp:RPAREN 
	b = compoundStatement[]
	{ s = new CatchClause((JavaToken)id, (JavaToken)lp, p, (JavaToken)rp, b); }
    ;

// an empty statement
emptyStmt[] returns [EmptyStmt s = null]
    :	semi:SEMI
    	{ s = new EmptyStmt((JavaToken)semi); }
    ;

// an assetion statement
assertStmt[] returns [AssertStmt s = null]
	{ Expr e1, e2 = null; }
    :	tkn:"assert" e1 = expression
    	(
    		colon:COLON e2 = expression
    	)?
    	semi:SEMI
    	{ s = new AssertStmt((JavaToken)tkn, e1, (JavaToken)colon, e2, (JavaToken)semi); }
    ;


// expressions
// Note that most of these expressions follow the pattern
//   thisLevelExpression :
//       nextHigherPrecedenceExpression
//           (OPERATOR nextHigherPrecedenceExpression)*
// which is a standard recursive definition for a parsing an expression.
// The operators in java have the following precedences:
//    lowest  (13)  = *= /= %= += -= <<= >>= >>>= &= ^= |=
//            (12)  ?:
//            (11)  ||
//            (10)  &&
//            ( 9)  |
//            ( 8)  ^
//            ( 7)  &
//            ( 6)  == !=
//            ( 5)  < <= > >=
//            ( 4)  << >>
//            ( 3)  +(binary) -(binary)
//            ( 2)  * / %
//            ( 1)  ++ -- +(unary) -(unary)  ~  !  (type)
//                  []   () (method call)  . (dot -- identifier qualification)
//                  new   ()  (explicit parenthesis)
//
// the last two are not usually on a precedence chart; I put them in
// to point out that new has a higher precedence than '.', so you
// can validy use
//     new Frame().show()
// 
// Note that the above precedence levels map to the rules below...
// Once you have a precedence chart, writing the appropriate rules as below
//   is usually very straightfoward



// the mother of all expressions
expression returns[Expr e = null]
    :   e = assignmentExpression
    ;


// This is a list of expressions.
expressionList returns[AstVector v = null]
    { 
	Expr e; 
	v = new AstVector(1);
    }
    :   e = expression { v.add(e); } 
	(COMMA e = expression { v.add(e); } )*
    ;

// This is a list of expression statements.
exprStmtList returns[AstVector v = null]
    { 
	Expr e; 
	v = new AstVector();
    }
    :   e = expression { v.add(new ExprStmt(e, null)); } 
	(COMMA e = expression { v.add(new ExprStmt(e, null)); } )*
    ;

// assignment expression (level 13)
assignmentExpression returns[Expr e = null]
    { 
	int       kind = 0;
	JavaToken opToken = null;
	Expr   expr;
    }
    :   e = conditionalExpression
        (   (   tkn1:ASSIGN
		{ kind = AssignExpr.ASSIGN;
		  opToken = (JavaToken)tkn1; }
            |   tkn2:PLUS_ASSIGN
		{ kind = AssignExpr.PLUS_ASSIGN;
		  opToken = (JavaToken)tkn2; }
            |   tkn3:MINUS_ASSIGN
		{ kind = AssignExpr.MINUS_ASSIGN;
		  opToken = (JavaToken)tkn3; }
            |   tkn4:STAR_ASSIGN
		{ kind = AssignExpr.STAR_ASSIGN;
		  opToken = (JavaToken)tkn4; }
            |   tkn5:DIV_ASSIGN
		{ kind = AssignExpr.DIV_ASSIGN;
		  opToken = (JavaToken)tkn5; }
            |   tkn6:MOD_ASSIGN
		{ kind = AssignExpr.MOD_ASSIGN;
		  opToken = (JavaToken)tkn6; }
            |   tkn7:SR_ASSIGN
		{ kind = AssignExpr.SR_ASSIGN;
		  opToken = (JavaToken)tkn7; }
            |   tkn8:BSR_ASSIGN
		{ kind = AssignExpr.BSR_ASSIGN;
		  opToken = (JavaToken)tkn8; }
            |   tkn9:SL_ASSIGN
		{ kind = AssignExpr.SL_ASSIGN;
		  opToken = (JavaToken)tkn9; }
            |   tkn10:BAND_ASSIGN
		{ kind = AssignExpr.BAND_ASSIGN;
		  opToken = (JavaToken)tkn10; }
            |   tkn11:BXOR_ASSIGN
		{ kind = AssignExpr.BXOR_ASSIGN;
		  opToken = (JavaToken)tkn11; }
            |   tkn12:BOR_ASSIGN
		{ kind = AssignExpr.BOR_ASSIGN;
		  opToken = (JavaToken)tkn12; }
            )
            expr = assignmentExpression
	    { e = new AssignExpr(kind, e, opToken, expr); }
        )?
    ;


// conditional test (level 12)
conditionalExpression returns[Expr e = null]
    { Expr trueExpr, falseExpr; }
    :   e = logicalOrExpression
        ( 
	    quest:QUESTION trueExpr = assignmentExpression 
	    colon:COLON falseExpr = conditionalExpression 
	    { e = new ConditionalExpr(e, (JavaToken)quest, trueExpr, (JavaToken)colon, falseExpr); }
	)?
    ;


// logical or (||)  (level 11)
logicalOrExpression returns[Expr e = null]
    { Expr expr; }
    :   e = logicalAndExpression 
	(
	    tkn:LOR expr = logicalAndExpression
	    { e = new BinaryExpr(BinaryExpr.LOR, e, (JavaToken)tkn, expr); }
	)*
    ;


// logical and (&&)  (level 10)
logicalAndExpression returns[Expr e = null]
    { Expr expr; }
    :   e = inclusiveOrExpression 
	(
	    tkn:LAND expr = inclusiveOrExpression
	    { e = new BinaryExpr(BinaryExpr.LAND, e, (JavaToken)tkn, expr); }
	)*
    ;


// bitwise or non-short-circuiting or (|)  (level 9)
inclusiveOrExpression returns[Expr e = null]
    { Expr expr; }
    :   e = exclusiveOrExpression 
	(
	    tkn:BOR expr = exclusiveOrExpression
	    { e = new BinaryExpr(BinaryExpr.BOR, e, (JavaToken)tkn, expr); }
	)*
    ;


// exclusive or (^)  (level 8)
exclusiveOrExpression returns[Expr e = null]
    { Expr expr; }
    :   e = andExpression 
	(
	    tkn:BXOR expr = andExpression
	    { e = new BinaryExpr(BinaryExpr.BXOR, e, (JavaToken)tkn, expr); }
	)*
    ;


// bitwise or non-short-circuiting and (&)  (level 7)
andExpression returns[Expr e = null]
    { Expr expr; }
    :   e = equalityExpression 
	(
	    tkn:BAND expr = equalityExpression
	    { e = new BinaryExpr(BinaryExpr.BAND, e, (JavaToken)tkn, expr); }
	)*
    ;


// equality/inequality (==/!=) (level 6)
equalityExpression returns[Expr e = null]
    { 
	Expr   expr; 
	JavaToken opToken = null; 
	int       kind = 0;
    }
    :   e = relationalExpression 
	(
	    ( tkn1:NOT_EQUAL { kind = BinaryExpr.NE; opToken = (JavaToken)tkn1; } 
	    | tkn2:EQUAL     { kind = BinaryExpr.EQ; opToken = (JavaToken)tkn2; } ) 
	    expr = relationalExpression
	    { e = new BinaryExpr(kind, e, opToken, expr); }
	)*
    ;


// boolean relational expressions (level 5)
relationalExpression returns[Expr e = null]
    { 
	TypeExpr type;
	Expr     expr; 
        JavaToken   opToken = null; 
	int         kind = 0;
    }
    :   e = shiftExpression
        (   (   (   tkn1:LT { kind = BinaryExpr.LT; opToken = (JavaToken)tkn1; }
                |   tkn2:GT { kind = BinaryExpr.GT; opToken = (JavaToken)tkn2; }
                |   tkn3:LE { kind = BinaryExpr.LE; opToken = (JavaToken)tkn3; }
                |   tkn4:GE { kind = BinaryExpr.GE; opToken = (JavaToken)tkn4; }
                )
                expr = shiftExpression
		{ e = new BinaryExpr(kind, e, opToken, expr); }
            )*
        |   tkn5:"instanceof" type = typeSpec
	    { e = new BinaryExpr(BinaryExpr.INSTANCEOF, e, (JavaToken)tkn5, type); }
        )
    ;


// bit shift expressions (level 4)
shiftExpression returns[Expr e = null]
    { 
	Expr   expr; 
        JavaToken opToken = null; 
	int       kind = 0;
    }
    :   e = additiveExpression 
	(
	    (	tkn1:SL  { kind = BinaryExpr.SL; opToken = (JavaToken)tkn1; }
	    | 	tkn2:SR  { kind = BinaryExpr.SR; opToken = (JavaToken)tkn2; }
	    | 	tkn3:BSR { kind = BinaryExpr.BSR; opToken = (JavaToken)tkn3; }
	    ) 
	    expr = additiveExpression
	    { e = new BinaryExpr(kind, e, opToken, expr); }
	)*
    ;


// binary addition/subtraction (level 3)
additiveExpression returns[Expr e = null]
    { 
	Expr   expr; 
        JavaToken opToken = null; 
	int       kind = 0;
    }
    :   e = multiplicativeExpression 
	(
	    (	tkn1:PLUS  { kind = BinaryExpr.PLUS; opToken = (JavaToken)tkn1; }
	    | 	tkn2:MINUS { kind = BinaryExpr.MINUS; opToken = (JavaToken)tkn2; }
	    ) 
	    expr = multiplicativeExpression
	    { e = new BinaryExpr(kind, e, opToken, expr); }
	)*
    ;


// multiplication/division/modulo (level 2)
multiplicativeExpression returns[Expr e = null]
    { 
	Expr   expr; 
        JavaToken opToken = null; 
	int       kind = 0;
    }
    :   e = unaryExpression 
	(
	    (	tkn1:STAR { kind = BinaryExpr.MUL; opToken = (JavaToken)tkn1; }
	    | 	tkn2:DIV  { kind = BinaryExpr.DIV; opToken = (JavaToken)tkn2; }
	    | 	tkn3:MOD  { kind = BinaryExpr.MOD; opToken = (JavaToken)tkn3; }
	    ) 
	    expr = unaryExpression
	    { e = new BinaryExpr(kind, e, opToken, expr); }
	)*
    ;

unaryExpression returns[Expr e = null]
    { Expr expr; }
    :   id1:INC expr = unaryExpression 
	{ e = new IncDecExpr(expr, IncDecExpr.PRE_INC, (JavaToken)id1); }
    |   id2:DEC expr = unaryExpression
	{ e = new IncDecExpr(expr, IncDecExpr.PRE_DEC, (JavaToken)id2); }
    |   id3:MINUS expr = unaryExpression
	{ e = new UnaryPlusMinusExpr(expr, UnaryPlusMinusExpr.MINUS, (JavaToken)id3); }
    |   id4:PLUS expr = unaryExpression
	{ e = new UnaryPlusMinusExpr(expr, UnaryPlusMinusExpr.PLUS, (JavaToken)id4); }
    |   e = unaryExpressionNotPlusMinus
    ;

unaryExpressionNotPlusMinus returns[Expr e = null]
    { 
	TypeExpr type; 
	Expr     expr, e2;
    }
    :   id1:BNOT expr = unaryExpression
	{ e = new BinaryNotExpr(expr, (JavaToken)id1); }
    |   id2:LNOT expr = unaryExpression
	{ e = new LogicalNotExpr(expr, (JavaToken)id2); }

    |   (   // subrule allows option to shut off warnings
            options {
                // "(int" ambig with postfixExpr due to lack of sequence
                // info in linear approximate LL(k).  It's ok.  Shut up.
                generateAmbigWarnings=false;
            }
        :   // If typecast is built in type, must be numeric operand
            // Also, no reason to backtrack if type keyword like int, float...
            lp1:LPAREN type = builtInTypeSpec rp1:RPAREN
            e2 = unaryExpression
	    { e = new CastExpr((JavaToken)lp1, type, (JavaToken)rp1, e2); }

            // Have to backtrack to see if operator follows.  If no operator
            // follows, it's a typecast.  No semantic checking needed to parse.
            // if it _looks_ like a cast, it _is_ a cast; else it's a "(expr)"
        |   (LPAREN classTypeSpec RPAREN unaryExpressionNotPlusMinus)=>
            lp2:LPAREN type = classTypeSpec rp2:RPAREN
            e2 = unaryExpressionNotPlusMinus
	    { e = new CastExpr((JavaToken)lp2, type, (JavaToken)rp2, e2); }

        |   e = postfixExpression
        )
    ;

// qualified names, array expressions, method invocation, post inc/dec
postfixExpression returns[Expr e = null]
    { 
	AstVector   args = null; 
	Expr        index;
    }
    :   e = primaryExpression // start with a primary

        (   // qualified id (id.id.id.id...) -- build the name
            dot1:DOT ( id1:IDENT
		    		{ e = new FieldAccess(e, FieldAccess.SIMPLE, (JavaToken)dot1, (JavaToken)id1); }
                | id2:"this"
		    		{ e = new FieldAccess(e, FieldAccess.THIS, (JavaToken)dot1, (JavaToken)id2); }
                | id3:"class"
		    		{ e = new FieldAccess(e, FieldAccess.CLASS, (JavaToken)dot1, (JavaToken)id3); }
                | e = newExpression[e]
                | stkn:"super" 
					{ e = new FieldAccess(e, FieldAccess.SUPER, (JavaToken)dot1, (JavaToken)stkn); }
		    	)
            // the above line needs a semantic check to make sure "class"
            //   is the _last_ qualifier.

            // allow ClassName[].class
        |   ( LBRACK RBRACK )+
            dot2:DOT id4:"class"
	        { e = new FieldAccess(e, FieldAccess.CLASS, (JavaToken)dot2, (JavaToken)id4); }

            // an array indexing operation
        |   lb:LBRACK index = expression rb:RBRACK
	    { e = new ArrayAccess(e, index, (JavaToken)lb, (JavaToken)rb); }

			// method invocation
			// The next line is not strictly proper; it allows x(3)(4) or
			//  x[2](4) which are not valid in Java.  If this grammar were used
			//  to validate a Java program a semantic check would be needed, or
			//   this rule would get really ugly...
			// It also allows ctor invocation like super(3) which is now
			// handled by the explicit constructor rule, but it would
			// be hard to syntactically prevent ctor calls here
        | lp2:LPAREN { args = null; } args = argList rp2:RPAREN
	    	{ e = new MethodCall(e, (JavaToken)lp2, args, (JavaToken)rp2); }
        )*

        // possibly add on a post-increment or post-decrement.
        // allows INC/DEC on too much, but semantics can check
        (   id5:INC
	    { e = new IncDecExpr(e, IncDecExpr.POST_INC, (JavaToken)id5); }
        |   id6:DEC
	    { e = new IncDecExpr(e, IncDecExpr.POST_DEC, (JavaToken)id6); }
        |   // nothing
        )
    ;

// the basic element of an expression
primaryExpression returns[Expr e = null]
	{ TypeExpr primType = null; }
    :   id:IDENT    { e = new SimpleName((JavaToken)id); }
    |   e = constant
    |   id2:"true"  { e = new BooleanLiteral((JavaToken)id2, true); }
    |   id3:"false" { e = new BooleanLiteral((JavaToken)id3, false); }
    |   id4:"this"  { e = new ThisExpr((JavaToken)id4); }
    |   id5:"null"  { e = new NullLiteral((JavaToken)id5); }
	|	id8:DBC_RESULT { e = new DbcResultExpr((JavaToken)id8); }
    |   e = newExpression[null]
    |   lp:LPAREN e = assignmentExpression rp:RPAREN
			{ e = new ParExpr((JavaToken)lp, e, (JavaToken)rp); }
    |   id1:"super" { e = new SuperExpr((JavaToken)id1); }
        // look for int.class and int[].class
    |   primType = builtInTypeSpec
   	 	dot:DOT id7:"class"
			{ e = new FieldAccess(primType, FieldAccess.CLASS, (JavaToken)dot, 
				(JavaToken)id7); }
    ;

// object instantiation.
newExpression[Expr base] returns[Expr e = null]
    { 
	Name t; 
	AstVector dims, args;
	ArrayInitializer init = null;
	TypeDeclaration acls = null;
    }
    :   tkn:"new" t = type
        (   lp:LPAREN args = argList rp:RPAREN (classBlock[acls = new TypeDeclaration()])?
	    { e = new NewExpr(base, t, (JavaToken)tkn, (JavaToken)lp, args, (JavaToken)rp, acls); }

            //java 1.1
            // Note: This will allow bad constructs like
            //    new int[4][][3] {exp,exp}.
            //    There needs to be a semantic check here...
            // to make sure:
            //   a) [ expr ] and [ ] are not mixed
            //   b) [ expr ] and an init are not used together

        |   dims = newArrayDeclarator (init = arrayInitializer)?
	    { e = new ArrayCreationExpr((JavaToken)tkn, t, dims, init); } 
        )
    ;

argList returns[AstVector v = null]
    :   (   v = expressionList
        |   /*nothing*/
        )
    ;

newArrayDeclarator returns [AstVector dims]
    {
	Expr expr = null;
	Brackets brackets;
	dims = new AstVector();
    }
    :   (
            // CONFLICT:
            // newExpression is a primaryExpression which can be
            // followed by an array index reference.  This is ok,
            // as the generated code will stay in this loop as
            // long as it sees an LBRACK (proper behavior)
            options {
                warnWhenFollowAmbig = false;
            }
        :
	    { expr = null; }
            lb:LBRACK
                (expr = expression)?
            rb:RBRACK
	    {
		brackets = new Brackets((JavaToken)lb, (JavaToken)rb);
		brackets.expr = expr;
		dims.add(brackets);
	    }
        )+
    ;

constant returns[Expr e = null]
    :   id1:INT_LITERAL    { e = new IntLiteral((JavaToken)id1); }
    |   id2:LONG_LITERAL   { e = new LongLiteral((JavaToken)id2); }
    |   id3:CHAR_LITERAL   { e = new CharLiteral((JavaToken)id3); }
    |   id4:STRING_LITERAL { e = new StringLiteral((JavaToken)id4); }
    |   id5:FLOAT_LITERAL  { e = new FloatLiteral((JavaToken)id5); }
    |   id6:DOUBLE_LITERAL { e = new DoubleLiteral((JavaToken)id6); }
    ;
