/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.urlgenerator;
import com.sap.tc.logging.Location;
import com.sapportals.wcm.util.logging.LoggingFormatter;

import com.sapportals.wcm.util.uri.*;

/**
 * Maps between RIDs and uri/uri references. <p>
 *
 * This interface is intended for managers which make resource from other
 * servers available as internal CM resource. External resources have to have a
 * hierarchical uri (e.g. http url of web pages) and will be given a CM resource
 * identifier (RID). <p>
 *
 * Every child of the root resource id is mapped to the corresponding child of
 * the root uri. Example:<br>
 * <pre>
 *             resource id           uri
 * root map    /mapped       <->     http://sapportals.com/test
 *             /mapped/name  <->     http://sapportals.com/test/name
 *             /mapped/a/b   <->     http://sapportals.com/test/a/b
 *             /nomap/x       ->     null
 *             null          <-      http://sapportals.com/xxx
 *             null          <-      http://greenbytes.de/test/name
 * </pre> <p>
 *
 * Copyright 2002 (c) SAP AG
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Revision: 1.1 $
 */
public interface IUriMapper {

  /**
   * Get the root uri of this mapping
   *
   * @return root uri of this mapping
   */
  public IHierarchicalUri getRootUri();

  /**
   * Get the root resource id of this mapping
   *
   * @return root resource id of this mapping
   */
  public RID getRootRID();

  /**
   * Map the given uri to its resource id.<br>
   * toAbsoluteUri("http://sapportals.com/test/a") -> /mapped/a
   *
   * @param uri to map to resource id
   * @return mapped resource id or null, if uri was not descendant of root uri
   */
  public RID toRID(IHierarchicalUri uri);

  /**
   * Map the given path to an (absolute) uri.<br>
   * toAbsoluteUri("/mapped/a") -> http://sapportals.com/test/a
   *
   * @param resourceID to map to uri
   * @return mapped uri or null, if resourceID was not descendant of root
   */
  public IHierarchicalUri toAbsoluteUri(RID resourceID);

  /**
   * Map the given resource id to an relative uri reference with absolute path
   * component.<br>
   * toAbsoluteUri("/mapped/a") -> /test/a
   *
   * @param resourceID to map to uri reference
   * @return mapped uri reference or null, if resourceID was not descendant of
   *      root
   */
  public IUriReference toAbsolutePath(RID resourceID);

  /**
   * Determine if this mapper is generic, e.g. handles conversions outside its
   * root.
   *
   * @return if mapper is generic
   */
  public boolean isGeneric();

}

