/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.urlgenerator;

import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.util.uri.IHierarchicalUri;
import com.sapportals.wcm.util.uri.IRidSet;
import com.sapportals.wcm.util.uri.IUri;
import com.sapportals.wcm.util.uri.IUriReference;
import com.sapportals.wcm.util.uri.RID;
import java.util.List;

/**
 * Offers methods to get relative or absolute URIs pointing to certain resources
 * of the CM system like iViews or repositories. <p>
 *
 * The URL generator service provides methods to get relative URIs (containing
 * absolute paths without protocol and server) to certain resources like the
 * explorer page or details page. It's recommended to use only these relative
 * URIs as long as possible. For example, it's sufficient to use a relative URI
 * for setting a link from one page to another page on the same server (see
 * example below) <p>
 *
 * The interface also offers the possibility to generate absolute URIs
 * (containing protocol and server). This feature is need, e.g., for integrating
 * URIs into emails to allow a direct link into the CM system (see example
 * below) <p>
 *
 * To obtain a valid instance of an URL generator, proceed as follows: <pre>
 *   IURLGeneratorService ug = (IURLGeneratorService)ResourceFactory
 *     .getInstance()
 *     .getServiceFactory()
 *     .getService(IServiceTypesConst.URLGENERATOR_SERVICE);
 * </pre> To set a link from one page to another on the same server, we only
 * need a relative URI; proceed as follows: <pre>
 *   RID rid = <rid of some resource>;
 *   IUriReference uriRef = ug.getResourcePageUri(PathKey.DETAILS_PAGE, rid,null);
 *
 *   com.sapportals.htmlb.Link  link = new Link("detail", "Details");
 *   link.setReference( uriRef.toString() );
 * </pre> To include a link to a page in an email, we need an absolute URI;
 * proceed as follows: <pre>
 *   RID rid = <rid of some resource>;
 *   IUriReference uriRef = ug.getResourcePageUri(PathKey.DETAILS_PAGE, rid,null);
 *   IHierarchicalUri uri = ug.createAbsoluteUri(uriRef);
 *   String  uriString = uri.toExternalForm();
 *
 *   // put <uriString> into an email
 * </pre> Copyright (c) SAP AG
 *
 * @author ralf.hammer@sap.com
 * @version $Id:$
 */
public interface IURLGeneratorService extends IService {
  //----------------------------------------------------------------------------
  // Constants
  //----------------------------------------------------------------------------

  // user subfolder
  public final static String FOLDER_USER_BRIEFCASE = "briefcase";
  public final static String FOLDER_USER_CLIPBOARD = "clipboard";
  public final static String FOLDER_USER_FAVORITES = "favorites";
  public final static String FOLDER_USER_INBOX = "inbox";
  public final static String FOLDER_USER_LINKS = "links";
  public final static String FOLDER_USER_QUERIES = "queries";
  public final static String FOLDER_USER_TEMP = "temp";

  // user resource formats
  public final static String MIME_TYPE_RDF = IUserManagementRidGenerator.MT_RDF;
  public final static String MIME_TYPE_VCARD = IUserManagementRidGenerator.MT_VCARD;
  public final static String MIME_TYPE_PHOTO = IUserManagementRidGenerator.MT_PHOTO;


  //----------------------------------------------------------------------------
  // Methods
  //----------------------------------------------------------------------------

  /**
   * Gets the relative URI of the path referred by <code>key</code> . The
   * returned <code>IUriReference</code> will not contain a protocol or
   * authority. <br>
   * For example, for getting the path of the explorer serlvet, just call <pre>
   *   getRelativeUri(PathKey.EXPLORER_SERVLET) -> /wcm/apps/explorer
   * </pre>
   *
   * @param key the key of the path in question
   * @return the relative URI of the path referred by <code>key</code>
   * @throws WcmException if the requested path is not configured
   */
  public IUriReference getRelativeUri(PathKey key)
    throws WcmException;

  /**
   * Gets the absolute URI of the path referred by <code>key</code> . The
   * absolute URI will consist of the configured protocol, server and port
   * concatenated with the path in question. <br>
   * For example, for getting the URI of the explorer servlet, just call <pre>
   *   getAbsoluteUri(PathKey.EXPLORER_SERVLET) -> http://localhost:8080/wcm/apps/explorer
   * </pre>
   *
   * @param key the key of the path in question
   * @return the absoulte URI of the path referred by <code>key</code>
   * @throws WcmException if the requested path is not configured
   */
  public IHierarchicalUri getAbsoluteUri(PathKey key)
    throws WcmException;

  /**
   * Gets the absolute URI of the path referred by <code>key</code> . The
   * absolute URI will consist of the configured (alternative) protocol, server and port
   * concatenated with the path in question. <br>
   * For example, for getting the alternative URI of the explorer servlet, just call <pre>
   *   getAlternativeAbsoluteUri(PathKey.EXPLORER_SERVLET) -> http://server2:8080/wcm/apps/explorer
   * </pre>
   *
   * @param key the key of the path in question
   * @return the absoulte URI of the path referred by <code>key</code>
   * @throws WcmException if the requested path is not configured
   */
  public IHierarchicalUri getAlternativeAbsoluteUri(PathKey key)
    throws WcmException;

  /**
   * Creates an absolute URI out of the specified relative URI (containing an
   * absolute path) by prepending the configured protocol, server and port. <br>
   * For example, for getting the absolute URI of the business card of the user
   * 'admin', just call <pre>
   *   IUriReference uriRef = urlGenerator.getBusinessCardUri("admin", resourceContext);
   *   IHierarchicalUri uri = urlGenerator.createAbsoluteUri(uriRef);
   * </pre>
   *
   * @param relativeUri the relative URI to be transformed into an absolute one
   *      (must contain an absolute path)
   * @return the absolute URI representation of the specified relative URI
   * @exception WcmException Exception raised in failure situation
   */
  public IHierarchicalUri createAbsoluteUri(IUriReference relativeUri)
    throws WcmException;

  /**
   * Creates an absolute URI out of the specified relative URI (containing an
   * absolute path) by prepending the configured (alternative) protocol, server and port. <br>
   * For example, for getting the alternative absolute URI of the business card of the user
   * 'admin', just call <pre>
   *   IUriReference uriRef = urlGenerator.getBusinessCardUri("admin", resourceContext);
   *   IHierarchicalUri uri = urlGenerator.createAlternativeAbsoluteUri(uriRef);
   * </pre>
   *
   * @param relativeUri the relative URI to be transformed into an absolute one
   *      (must contain an absolute path)
   * @return the absolute URI representation of the specified relative URI
   * @exception WcmException Exception raised in failure situation
   */
  public IHierarchicalUri createAlternativeAbsoluteUri(IUriReference relativeUri)
    throws WcmException;

  /**
   * Returns the name of the XML file which is set as default in the CM
   * configuration and used for the wdf-navigation.
   *
   * @return the name of the XML file in question
   * @throws WcmException if the default configuration file is not configured
   */
  public String getNameOfConfigfile()
    throws WcmException;

  /**
   * Gets the relative URI of the image <code>filename</code> .
   *
   * @param filename the name of the image file (without path)
   * @return the (relative) URI of the image file
   * @throws WcmException if the image path is not configured
   */
  public IUriReference getImageUri(String filename)
    throws WcmException;

  /**
   * Gets the relative URI for bringing up the business card of the user
   * specified by <code>userId</code> .
   *
   * @param userId the identifier of the user whose business card being
   *      questioned
   * @return the (relative) URI of the user's business card
   * @throws WcmException if the path of the businesscard is not configured
   * @deprecated As of Netweaver '04 SP6, replaced by {@link #getBusinessCardUri(String, IResourceContext)}
   */
  public IUriReference getBusinessCardUri(String userId)
    throws WcmException;

  /**
   * Gets the relative URI for bringing up the business card of the user
   * specified by <code>userId</code> .
   *
   * @param userId   the identifier of the user whose business card being
   *      questioned
   * @param context  a resource context containing an authenticated user
   * @return the (relative) URI of the user's business card
   * @throws WcmException if the path of the businesscard is not configured
   */
  public IUriReference getBusinessCardUri(String userId, IResourceContext context)
    throws WcmException;

  /**
   * Gets the relative URI for bringing up a page handling the resource
   * specified by <code>resourceRID</code> . The additional parameter <code>
   * startRID</code> has to be a parent RID of the given resource RID. It
   * forbids navigation to any element being a root for the given resource RID.
   *
   * @param resourceRID the RID of the resource in question
   * @param startRID
   * @param key TBD: Description of the incoming method parameter
   * @return the (relative) URI of the page handling the resource
   * @throws WcmException if the requested path is not configured
   */
  public IUriReference getResourcePageUri(PathKey key, RID resourceRID, RID startRID)
    throws WcmException;

  /**
   * Gets the relative URI for bringing up a service which may immediately carry
   * out some instructions (e.g. delete a resource or display a dialog). The
   * parameter <code>properties</code> must contain parameters identifying the
   * service to be started and the instructions to be performed. <p>
   *
   * The interface <code>IWcmConst</code> holds a set of parameters starting
   * with <code>SERVICE_SERVLET</code> which can be used as keys or default
   * values for the parameter <code>properties</code> . You can also use "free"
   * key-value pairs, but you have to make sure, that the receiving service
   * knows about these keys. <p>
   *
   * Example: <pre>
   *   Properties props = new Properties();
   *   props.setProperty(IWcmConst.SERVICE_SERVLET_PARAMETER_RESOURCE,  rid.toExternalForm());
   *   props.setProperty(IWcmConst.SERVICE_SERVLET_PARAMETER_STARTPAGE, "WcmExplorerPage");
   *   props.setProperty(IWcmConst.SERVICE_SERVLET_PARAMETER_XML_FILE,  "WcmController.xml");
   *
   *   IUriReference uriRef = urlGenerator.getServletCallUri(props);
   * </pre
   *
   * @param queryParameters TBD: Description of the incoming method parameter
   * @return the (relative) URI to call the service page
   * @throws WcmException if the path to the service page is not configured
   */
  public IUriReference getServicePageUri(List queryParameters)
    throws WcmException;

  /**
   * Gets the relative URI of the usermapping maintenance servlet.
   *
   * @param systemId the identifier of the system in the system landscape
   * @param successUrl
   * @param failureUrl
   * @param cancelUrl
   * @return the (relative) URI of the usermapping maintenance servlet
   * @throws WcmException if the path to the usermapping page is not configured
   */
  public IUriReference getUsermappingPageUri(String systemId, String successUrl,
    String failureUrl, String cancelUrl)
    throws WcmException;


  //----------------------------------------------------------------------------
  // Methods for getting RIDs of user folders
  //----------------------------------------------------------------------------

  /**
   * Gets the RID of the user's home folder.
   *
   * @param user the user whose home folder in question
   * @return the RID of the user's home folder
   * @exception WcmException Exception raised in failure situation
   */
  public RID getUserHomeRID(IUser user)
    throws WcmException;

  /**
   * Gets the RID of the specified subfolder of the user's home folder.
   *
   * @param user the user in question
   * @param folder the desired subfolder of the user's home folder
   * @return the RID of the user's subfolder (one of the constants <code>
   *      FOLDER_*</code> )
   * @exception WcmException Exception raised in failure situation
   */
  public RID getUserFolderRID(IUser user, String folder)
    throws WcmException;


  //----------------------------------------------------------------------------
  // Methods for getting RIDs of resources representing principals
  // (users, groups, roles) or information about principals
  //----------------------------------------------------------------------------

  /**
   * Sets <code>ridGenerator</code> as the new <code>IUserManagementRidGenerator
   * </code>. used by some of the methods in this interface to get RIDs of
   * resources representing principals (users. groups, roles) or information
   * about them. <p>
   *
   * @param ridGenerator the RID generator to be set
   * @see #getGroupResourceRID(String)
   * @see #getRoleResourceRID(String)
   * @see #getUserResourceRID(String)
   */
  public void setUMRidGenerator(IUserManagementRidGenerator ridGenerator);

  /**
   * Gets a set of RIDs of all collections representing principals (user,
   * groups, roles).
   *
   * @return set of all collections representing principals
   * @throws WcmException on error
   */
  public IRidSet getPrincipalCollectionRidSet()
    throws WcmException;

  /**
   * Gets the RID of a resource representing the specified user.
   *
   * @param user the user in question
   * @return the RID of the corresponding user resource
   * @throws WcmException on error
   */
  public RID getUserResourceRID(IUser user)
    throws WcmException;

  /**
   * Gets the RID of a resource in the specified format representing the
   * specified user.
   *
   * @param user the user in question
   * @param format the desired format (one of the constants <code>FORMAT_*
   *      </code>)
   * @return the RID of the corresponding user resource
   * @throws WcmException on error
   */
  public RID getUserResourceRID(IUser user, String format)
    throws WcmException;

  /**
   * Gets the RID of a resource representing the specified user.
   *
   * @param userId the identifier of the user in question
   * @return the RID of the corresponding user resource
   * @throws WcmException on error
   */
  public RID getUserResourceRID(String userId)
    throws WcmException;

  /**
   * Gets the RID of a resource in the specified format representing the
   * specified user.
   *
   * @param userId the identifier of the user in question
   * @param format the desired format (one of the constants <code>FORMAT_*
   *      </code>
   * @return the RID of the corresponding user resource
   * @throws WcmException on error
   */
  public RID getUserResourceRID(String userId, String format)
    throws WcmException;

  /**
   * Gets the RID of a collection representing the specified group.
   *
   * @param group the group in question
   * @return the RID of the corresponding group collection
   * @throws WcmException on error
   */
  public RID getGroupResourceRID(IGroup group)
    throws WcmException;

  /**
   * Gets the RID of a collection representing the specified group.
   *
   * @param groupId the identifier of the group in question
   * @return the RID of the corresponding group collection
   * @throws WcmException on error
   */
  public RID getGroupResourceRID(String groupId)
    throws WcmException;

  /**
   * Gets the RID of a collection representing the specified role.
   *
   * @param role the role in question
   * @return the RID of the corresponding role collection
   * @throws WcmException on error
   */
  public RID getRoleResourceRID(IRole role)
    throws WcmException;

  /**
   * Gets the RID of a collection representing the specified role.
   *
   * @param roleId the identifier of the role in question
   * @return the RID of the corresponding role collection
   * @throws WcmException on error
   */
  public RID getRoleResourceRID(String roleId)
    throws WcmException;


  //----------------------------------------------------------------------------
  // Methods for resolving placeholders
  //----------------------------------------------------------------------------

  /**
   * Registers <code>resolver</code> to this <code>IURLGeneratorService</code> .
   * <p>
   *
   * The resolver will be registered for handling the placeholder which is
   * returned by a call to {@link IPlaceholderResolver#getPlaceholder()}. <p>
   *
   * If another resolver has already been registered for handling this
   * placeholder, it depends on the classes of the two resolver wether an
   * exception will be thrown or not. <p>
   *
   * If both resolver are instances of the same class, the new resolver will
   * replace the old one. Otherwise, an <code>AlreadyRegisteredException</code>
   * will be thrown.
   *
   * @param resolver the <code>IPlaceholderResolver</code> to be registered
   * @throws AlreadyRegisteredException if a different class has already been
   *      registered for handling the corresponding placeholder
   */
  public void registerPlaceholderResolver(IPlaceholderResolver resolver)
    throws AlreadyRegisteredException;

  /**
   * Deregisters <code>resolver</code> from this <code>IURLGeneratorService
   * </code>. <p>
   *
   * If <code>resolver</code> is not registered for handling the placeholder
   * which is returned by a call to {@link IPlaceholderResolver#getPlaceholder()},
   * the call to this method will be ignored.
   *
   * @param resolver the <code>IPlaceholderResolver</code> to be deregistered
   */
  public void deregisterPlaceholderResolver(IPlaceholderResolver resolver);

  /**
   * Resolves all placeholders contained in <code>uri</code> by their
   * corresponding values.
   *
   * @param uri a <code>String</code> whose placeholder should be resolved
   * @param resource
   * @return TBD: Description of the outgoing return value
   * @throws ResourceException on error
   */
  public String resolvePlaceholders(String uri, IResource resource)
    throws ResourceException;

  /**
   * Resolves all placeholders contained in <code>uri</code> by their
   * corresponding values.
   *
   * @param uri a <code>String</code> whose placeholder should be resolved
   * @param context
   * @return TBD: Description of the outgoing return value
   * @throws ResourceException on error
   */
  public String resolvePlaceholders(String uri, IResourceContext context)
    throws ResourceException;


  //----------------------------------------------------------------------------
  // Methods for mapping web resources to CM resources
  //----------------------------------------------------------------------------

  /**
   * Registers <code>mapper</code> to this <code>IURLGeneratorService</code> .
   * <p>
   *
   * Whenever a web resource needs to be represented as a CM resource, the
   * method should be called which will use the registered mappers to accomplish
   * the task of mapping a given web URL to a RID of a CM resource.
   *
   * @param mapper the URI mapper to be registered
   */
  public void registerUriMapper(IUriMapper mapper);

  /**
   * Deregisters <code>mapper</code> from this <code>IURLGeneratorService</code>
   * .
   *
   * @param mapper the URI mapper to be deregistered
   */
  public void deregisterUriMapper(IUriMapper mapper);

  /**
   * Gets the RID of the CM resource which represents the given URI (e.g. http
   * resource).
   *
   * @param uri the URI to be mapped
   * @return the RID of a CM resource representing the given URI or <code>null
   *      </code> if mapping not possible
   * @exception WcmException Exception raised in failure situation
   */
  public RID mapUri(IUri uri)
    throws WcmException;

  /**
   * Gets the URI of the CM resource (e.g. http resource) represented by the 
   * given RID.
   *
   * @param rid the RID to be mapped
   * @return the URI of a CM resource represented by the given RID or <code>null
   *      </code> if mapping not possible
   * @exception WcmException Exception raised in failure situation
   */
  public IUri mapRID(RID rid)
    throws WcmException;

}// interface IURLGeneratorService
