/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.urlgenerator;

import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.ICollection;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.util.uri.*;
import java.util.Collection;
import java.util.Properties;

/**
 * Services, controls, repositories need to access resources, which are
 * accessible only through their respective URLs. This interface offers methods
 * to create URLs for certain resources, to build valid URLs from partial
 * information known to the calling object by adding information from the
 * configuration of the running system. This allows to create complete URLs,
 * pointing to a webpage that can e.g. be included in an email or as link on
 * another page. To obtain a valid instance of an URLGenerator, proceed as
 * follows: <pre>
 *   IURLGenerator ug = (IURLGenerator)ResourceFactory.getInstance()
 *                                    .getServiceFactory()
 *                                    .getService(IServiceTypesConst.URLGENERATOR_SERVICE);
 * </pre> A standard usecase would be to offer a link on a webpage that allows
 * the user to navigate to the details of a given resource: <pre>
 *   String linkreference = ug.createLinkToDetailsPageRID(resource.getRID()).toString();
 * </pre> Please note, that some of the methods will return a complete URL,
 * including protocol and server while others may only return a relative URL.
 * Also, some of the returned <code>RID</code> have a query-parameter attached,
 * so be careful when transforming into <code>String</code>. <p>
 *
 * Most field variables are explained in the
 *      package-html and in the configuration framework. They are needed for the
 *      internal mapping of configuration values to the internal variables. For
 *      the method <code>getValue</code> , use these field variables as
 *      parameter-keys. The method will then return the value configured for
 *      this key in the configuration framework. 
 * <p>
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 * 
 * @author andreas.heix@sapportals.com
 * @see RID
 * @version $Id: IURLGenerator.java,v 1.4 2004/05/14 14:01:49 mko Exp $
 * @deprecated as of EP 5.0 SP5, replaced by interface {@link IURLGeneratorService}
 */
public interface IURLGenerator {

  // user subfolder
  public final static String USER_INBOX = IURLGeneratorService.FOLDER_USER_INBOX;
  public final static String USER_FAVORITES = IURLGeneratorService.FOLDER_USER_FAVORITES;
  public final static String USER_LINKS = IURLGeneratorService.FOLDER_USER_LINKS;
  public final static String USER_BRIEFCASE = IURLGeneratorService.FOLDER_USER_BRIEFCASE;
  public final static String USER_QUERIES = IURLGeneratorService.FOLDER_USER_QUERIES;
  public final static String USER_CLIPBOARD = IURLGeneratorService.FOLDER_USER_CLIPBOARD;
  public final static String USER_TEMP = IURLGeneratorService.FOLDER_USER_TEMP;

  // user resource formats
  public final static String USER_RESOURCE_FORMAT_RDF = IURLGeneratorService.MIME_TYPE_RDF;
  public final static String USER_RESOURCE_FORMAT_VCARD = IURLGeneratorService.MIME_TYPE_VCARD;
  public final static String USER_RESOURCE_FORMAT_PHOTO = IURLGeneratorService.MIME_TYPE_PHOTO;

  /**
   * TAG-Variables are used as keys for the given properties
   */
  public final static String TAG_APPLOG_DOWNLOAD_SERVLET_PATH = PathKey.APPLOG_DOWNLOAD_PAGE.getKey();
  public final static String TAG_APPLOG_PATH = PathKey.APPLOG_PAGE.getKey();
  public final static String TAG_CONTENT_ACCESS_PATH = PathKey.CONTENT_ACCESS_PATH.getKey();
  public final static String TAG_DETAILSPAGE = PathKey.DETAILS_PAGE.getKey();
  public final static String TAG_EXPLORER_SERVLET_PATH = PathKey.EXPLORER_PAGE.getKey();
  public final static String TAG_FAVORITES = PathKey.FAVORITES_PAGE.getKey();
  public final static String TAG_ICE_PATH = PathKey.ICE_SERVER.getKey();
  public final static String TAG_IMAGE_PATH = PathKey.IMAGE_PATH.getKey();
  public final static String TAG_LINKLIST = PathKey.LINKLIST_PAGE.getKey();
  public final static String TAG_SERVICE_SERVLET_PATH = PathKey.SERVICE_PAGE.getKey();
  public final static String TAG_USERMAPPING_SERVLET_PATH = PathKey.USERMAPPING_PAGE.getKey();
  public final static String TAG_VIEWER_PATH = PathKey.VIEWER_PATH.getKey();
  public final static String TAG_XML_FORMS_CSS_URL = PathKey.XML_FORMS_PATH.getKey();
  public final static String TAG_XML_FORMS_EDITSERVLET_PATH = PathKey.XML_FORMS_EDIT_PAGE.getKey();
  public final static String TAG_XML_FORMS_LISTSERVLET_PATH = PathKey.XML_FORMS_LIST_PAGE.getKey();
  public final static String TAG_XML_FORMS_PREVIEWSERVLET_PATH = PathKey.XML_FORMS_PREVIEW_PAGE.getKey();
  public final static String TAG_XML_FORMS_XFBUILDERSERVLET_PATH = PathKey.XML_FORMS_XFBUILDER_PAGE.getKey();

  public final static String TAG_BUSINESS_CARD_PATH = "businesscardservlet";
  public final static String TAG_DEFAULT_CONFIG_FILE_NAME = "DefaultConfigFile";
  public final static String TAG_ENVIRONMENT = "environment";
  public final static String TAG_FOLDER_ACCESS_PATH = "folderaccesspath";
  public final static String TAG_HOST = "hostserver";
  public final static String TAG_GROUPRESOURCE = "groupresource";
  public final static String TAG_ROLERESOURCE = "roleresource";
  public final static String TAG_USERHOME = "userhome";
  public final static String TAG_USERRESOURCE = "userresource";
  public final static String TAG_WEBRM_ACCESS_PATH = "webrm";
  public final static String TAG_XML_FORMS_CSS = "XMLFormscss";

  public final static String TAG_USER_RESOURCE_FORMAT_RDF = TAG_USERRESOURCE + '-' + USER_RESOURCE_FORMAT_RDF;
  public final static String TAG_USER_RESOURCE_FORMAT_VCARD = TAG_USERRESOURCE + '-' + USER_RESOURCE_FORMAT_VCARD;
  public final static String TAG_USER_RESOURCE_FORMAT_PHOTO = TAG_USERRESOURCE + '-' + USER_RESOURCE_FORMAT_PHOTO;

  public final static String CONFIG_CHAROPEN = "<";
  public final static String CONFIG_CHARCLOSE = ">";

  /**
   * @deprecated as of NW04.
   */
  public final static String TAG_JBUILDER = "JBuilder";
  /**
   * @deprecated as of NW04.
   */
  public final static String TAG_TOMCAT = "Tomcat";


  /**
   * Returns the RID to the user's home folder.
   *
   * @param user the user we want to get the home folder for
   * @return the RID to the user's home folder
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserHomeRID(IUser)}
   */
  public RID getUserHomeRID(IUser user)
    throws WcmException;

  /**
   * Returns RID for a folder specified by user context and folder name.
   *
   * @param user the user in question
   * @param folder the folder in question
   * @return the RID to the user/folder combination
   * @exception WcmException must be thrown, when user is <code>null</code> ,
   *      implementations of this interface might decide to also throw the
   *      exception in additional situations
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserFolderRID(IUser, String)}
   */
  public RID getUserFolderRID(IUser user, String folder)
    throws WcmException;

  /**
   * Returns the ServerName of the current server the VM is running on.
   *
   * @return the ServerName of the current server the VM is running on. When the
   *      system is running in a cluster environment, the IP-Adress of the load
   *      balancer is returned.
   * @deprecated as of EP 5.0 SP5, this method has been removed
   */
  public String getHost();

  /**
   * Returns the ServiceServletPath set for the current server the VM is running
   * on.
   *
   * @return the ServiceServletPath set for the current server the VM is running
   *      on. it consists of the protocol, the host and the serviceservlet path
   *      specified in the configuration of the <code>IURLGenerator</code>
   *      instance, e.g. http://myserver:8080/serviceservlet
   * @deprecated as of EP 5.0 SP5, replaced by 
   * {@link IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#SERVICE_PAGE}.
   */
  public String getServiceServletPath();

  /**
   * Returns the ExplorerServletPath set for the current server the VM is
   * running on.
   *
   * @return the ExplorerServletPath set for the current server the VM is
   *      running on. It consists of the protocol, the host and the
   *      explorerservlet path specified in the configuration of the <code>
   *      IURLGenerator</code> instance, e.g. <code>
   *      http://myserver:8080/explorerservlet</code>
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#EXPLORER_PAGE}
   */
  public String getExplorerServletPath();

  /**
   * Returns the AppLogDownload set for the current server the VM is running on.
   *
   * @return the AppLogDownload set for the current server the VM is running on.
   *      It consists of the protocol, the host and the applogdownload path
   *      specified in the configuration of the <code>IURLGenerator</code>
   *      instance, e.g. <code>http://myserver:8080/applogdownload</code>
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#APPLOG_DOWNLOAD_PAGE}
   */
  public String getAppLogDownloadPath();

  /**
   * Returns the AppLogPath set for the current server the VM is running on.
   *
   * @return the AppLogPath set for the current server the VM is running on. It
   *      consists of the protocol, the host and the applogdownload path
   *      specified in the configuration of the <code>IURLGenerator</code>
   *      instance, e.g. <code>http://myserver:8080/applog</code>
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#APPLOG_PAGE}
   */
  public String getAppLogPath();

  /**
   * Returns the ImagePath set for the current server the VM is running on.
   *
   * @return the ImagePath set for the current server the VM is running on. It
   *      returns a URL that allows to download the wanted image, e.g. when
   *      displaying an image on a website. It consists of the <code>
   *      ContentAccessPath</code> and the repository and path information to
   *      the image folder.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#IMAGE_PATH}
   */
  public String getImagePath();

  /**
   * Returns the complete ImagePath set for the current server the VM is running
   * on and the filename given.
   *
   * @param filename the name of the file that represents the wanted Image
   * @return the complete ImagePath set for the current server the VM is running
   *      on and the filename given.
   * @see #getImagePath()
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getImageUri(String)}
   */
  public String getImage(String filename);

  /**
   * Returns the URL to communicate with the WebDAV server of the CM system
   * which allows to retrieve content to e.g. display it in a webbrowser window.
   *
   * @return the ContentAccessPath set for the current server the VM is running
   *      on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#CONTENT_ACCESS_PATH}
   */
  public String getContentAccessPath();

  /**
   * Returns an url-part to navigate to the servlet which is defined as default
   * to display the content (= children) of a folder
   *
   * @return the FolderAccessPath set for the current server the VM is running
   *      on.
   * @deprecated as of EP 5.0 SP5, this method has been removed
   */
  public String getFolderAccessPath();

  /**
   * Returns the ICE path set for the current server the VM is running on.
   *
   * @return the ICE path set for the current server the VM is running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#ICE_SERVER}
   */
  public String getIcePath();

  /**
   * Returns the FavoritesPath set for the current server the VM is running on.
   *
   * @return the FavoritesPath set for the current server the VM is running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#FAVORITES_PAGE}
   */
  public String getFavorites();

  /**
   * Returns the LinkListPath set for the current server the VM is running on.
   *
   * @return the LinkListPath set for the current server the VM is running on.
   *      It consists of the protocol, the host and the linklist servlet path
   *      specified in the configuration of the <code>IURLGenerator</code>
   *      instance, e.g. <code>http://myserver:8080/linklist</code>
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#LINKLIST_PAGE}
   */
  public String getLinkList();

  /**
   * Returns the XMLForms ListServletPath set for the current server the VM is
   * running on.
   *
   * @return the XMLForms ListServletPath set for the current server the VM is
   *      running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#XML_FORMS_LIST_PAGE}
   */
  public String getListServletPath();

  /**
   * Returns the XMLForms EditServletPath set for the current server the VM is
   * running on.
   *
   * @return the XMLForms EditServletPath set for the current server the VM is
   *      running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#XML_FORMS_EDIT_PAGE}
   */
  public String getEditServletPath();

  /**
   * Returns the XMLForms PreviewServletPath set for the current server the VM
   * is running on.
   *
   * @return the XMLForms PreviewServletPath set for the current server the VM
   *      is running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#XML_FORMS_PREVIEW_PAGE}
   */
  public String getPreviewServletPath();

  /**
   * Returns the XMLForms XFBuilderServletPath set for the current server the VM
   * is running on.
   *
   * @return the XMLForms XFBuilderServletPath set for the current server the VM
   *      is running on.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#XML_FORMS_XFBUILDER_PAGE}
   */
  public String getXFBuilderServletPath();

  /**
   * Returns a complete URL to put in a mail or on a html-button to open the
   * folder and its content
   *
   * @param folder the folder to show
   * @return a complete URL to put in a mail or on a html-button that will open
   *      a webpage, showing the folder and its content, returns <code>null
   *      </code>, if resource is not a folder Please use
   *      createLinkToExplorerPage instead
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #createLinkToExplorerPageRID(RID)}
   */
  public RID getCompleteRIDToShowFolder(ICollection folder)
    throws WcmException;

  /**
   * Returns the url to show the content of a resource
   *
   * @return the ViewerPath set for the current server the VM is running on. it
   *      consists of the protocol, the host and the viewer servlet path
   *      specified in the configuration of the <code>IURLGenerator</code>
   *      instance, e.g. <code>http://myserver:8080/myViewer</code>
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} with parameter constant {@link PathKey#VIEWER_PATH)}
   */
  public String getViewerPath();

  /**
   * Returns the value mapped to the given key.
   *
   * @param key the key of the property to return. Use <code>IURLGenerator.TAG_
   *      </code> constants as keys
   * @return the value mapped to the key given. returns <code>null</code> , if
   *      there is no value for the given key
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRelativeUri(PathKey)} or {@link
   *      IURLGeneratorService#getAbsoluteUri(PathKey)}
   * @deprecated as of NW04. Use methods in interface IURLGeneratorService instead
   */
  public String getValue(String key);

  /**
   * Returns the xmlfile which is set as default in the CM configuration and
   * used for the wdf-navigation.
   *
   * @return the xmlfile which is set as default in the CM configuration and
   *      used for the wdf-navigation
   * @deprecated as of NW04.
   */
  public String getDefaultConfigFile();

  /**
   * Returns the xmlfile which is set as default in the CM configuration and
   * used for the wdf-navigation
   *
   * @param key this is not used anymore
   * @return the xmlfile which is set as default in the CM configuration and
   *      used for the wdf-navigation
   * @deprecated as of NW04.
   */
  public String getDefaultConfigFile(String key);

  /**
   * Creates a complete URL to include in emails etc to start a service. The URL
   * will point to a certain control, which understands the given properties. It
   * might immediately carry out some instructions (e.g. delete a resource) or
   * display a dialog. The <code>IWcmConst</code> holds a whole set of
   * parameters starting with <code>SERVICE_SERVLET</code> which can be used as
   * keys or default values in the <code>Properties</code> parameter.
   *
   * @param properties this includes parameters as defined in the
   *      IWcmConst.ServiceServletParameters You can also use "free" key-value
   *      pairs, but have to make sure, that receiving service will know about
   *      these keys
   * @return the Uri to call the servlet
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getServicePageUri(java.util.List)}
   */
  public RID createServletCallRID(Properties properties)
    throws WcmException;


  /**
   * Returns the URL to a usermapping maintenance servlet
   *
   * @param systemId id of the system
   * @param successUrl URL in case of success
   * @param failureUrl URL in case of failure
   * @param cancelUrl URL in case user cancels
   * @return The UsermappingServlet value
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUsermappingPageUri(String, String, String,
   *      String)}
   */
  public String getUsermappingServlet(String systemId, String successUrl, String failureUrl, String cancelUrl)
    throws WcmException;

  /**
   * Returns a URL to navigate to the details-view of a resource. Objects of
   * type <code>IResource</code> can be viewed in a special dialog, showing
   * properties and offering services available for this resource. The returned
   * value consists of the protocol, the host and the details page specified in
   * the configuration of the <code>IURLGenerator</code> instance, e.g. <code>
   * http://myserver:8080/details</code>
   *
   * @param resourceRID the RID of the resource in question
   * @return the RID to navigate to this page
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getResourcePageUri(PathKey, RID, RID)} with parameter constant {@link PathKey#DETAILS_PAGE}
   */
  public RID createLinkToDetailsPageRID(RID resourceRID);

  /**
   * Returns a URL to navigate to the details-view of a resource.
   *
   * @param resourceRID The RID of the resource in question
   * @param startRID The RID which is returned as StartPath-parameter
   * @return the RID to navigate to this page
   * @see #createLinkToDetailsPageRID(RID resourceRID) The additional <code>
   *      StartUri</code> parameter must be a parent RID of the given resource
   *      RID. It forbids a navigation to any element being a root for the given
   *      resource RID.
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getResourcePageUri(PathKey, RID, RID)} with parameter constant {@link PathKey#DETAILS_PAGE}
   */
  public RID createLinkToDetailsPageRID(RID resourceRID, RID startRID);


  /**
   * Returns a URL to navigate to the explorer-view of a collection. Objects of
   * type <code>ICollection</code> can be viewed in a special dialog, showing
   * child resources, navigation information etc. The returned value consists of
   * the protocol, the host and the explorer page specified in the configuration
   * of the <code>IURLGenerator</code> instance, e.g. <code>
   * http://myserver:8080/explorer</code>
   *
   * @param resourceRID The Rid of the resource in question
   * @return the RID to navigate to this page
   * @exception WcmException Exception raised in failure situation. Is thrown
   *      when <code>resourceRID</code> is <code>null</code> .
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getResourcePageUri(PathKey, RID, RID)} with parameter constant {@link PathKey#EXPLORER_PAGE}
   */
  public RID createLinkToExplorerPageRID(RID resourceRID)
    throws WcmException;

  /**
   * Returns a URL to navigate to the explorer-view of a collection
   *
   * @param resourceRID The Rid of the resource in question
   * @param startRID The Rid which is returned as StartPath-parameter
   * @return the RID to navigate to this page
   * @exception WcmException Exception raised in failure situation. Is thrown
   *      when <code>resourceRID</code> is <code>null</code> .
   * @see #createLinkToExplorerPageRID(RID) The additional <code>StartUri</code>
   *      parameter must be a parent RID of the given resource RID. It forbids a
   *      navigation to any element being a root for given resource RID. gets
   *      RID to show explorer page of resource
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getResourcePageUri(PathKey, RID, RID)} with parameter constant {@link PathKey#EXPLORER_PAGE}
   */
  public RID createLinkToExplorerPageRID(RID resourceRID, RID startRID)
    throws WcmException;

  /**
   * Gets a resource RID representing a user
   *
   * @param user the user to get the RID for
   * @return The RID to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserResourceRID(IUser)}
   */
  public RID getUserResourceRID(IUser user)
    throws WcmException;

  /**
   * Gets a resource RID representing a user or user information in a specific
   * format
   *
   * @param user the user in question
   * @param format the wanted format
   * @return The RID to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserResourceRID(IUser, String)}
   */
  public RID getUserResourceRID(IUser user, String format)
    throws WcmException;

  /**
   * Gets a resource RID representing a user
   *
   * @param uid the id of the user
   * @return The RID to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserResourceRID(String)}
   */
  public RID getUserResourceRID(String uid)
    throws WcmException;

  /**
   * Gets a resource URI representing a user or user information in a specific
   * format
   *
   * @param uid the id of the user
   * @param format the format wanted
   * @return The URI to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getUserResourceRID(String, String)}
   */
  public RID getUserResourceRID(String uid, String format)
    throws WcmException;

  /**
   * Gets a resource RID representing a group
   *
   * @param group the group in question
   * @return The RID to a corresponding group collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getGroupResourceRID(IGroup)}
   */
  public RID getGroupResourceRID(IGroup group)
    throws WcmException;

  /**
   * Gets a resource RID representing a group
   *
   * @param gid the id of the group
   * @return The URI to a corresponding group collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getGroupResourceRID(String)}
   */
  public RID getGroupResourceRID(String gid)
    throws WcmException;

  /**
   * Gets a resource RID representing a role
   *
   * @param role the role in question
   * @return The URI to a corresponding role collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRoleResourceRID(IRole)}
   */
  public RID getRoleResourceRID(IRole role)
    throws WcmException;

  /**
   * Gets a resource RID representing a role
   *
   * @param rid The id of the role
   * @return The RID to a corresponding role collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getRoleResourceRID(String)}
   */
  public RID getRoleResourceRID(String rid)
    throws WcmException;

  /**
   * Gets a resource RID for bringing up a business card for a user
   *
   * @param user
   * @return The RID to a corresponding service
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getBusinessCardUri(String)}
   */
  public RID getBusinessCardAppRID(IUser user)
    throws WcmException;

  /**
   * Gets a resource RID for bringing up a business card for a user
   *
   * @param uid The id of the user
   * @return The RID to a corresponding service
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#getBusinessCardUri(String)}
   */
  public RID getBusinessCardAppRID(String uid)
    throws WcmException;

  /**
   * Replaces placeholders in URI
   *
   * @param uri the uri to replace the placeholders in
   * @param context the present context of the user who called this method
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   */
  public String replaceURIPlaceholders(String uri, IResourceContext context)
    throws ResourceException;


  /**
   * Replaces placeholders in URI
   *
   * @param uri
   * @param resource
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   */
  public String replaceURIPlaceholders(String uri, IResource resource)
    throws ResourceException;

  /**
   * methods with old URI-interface ***********************************************************************************
   *
   * @param user TBD: Description of the incoming method parameter
   * @param folder TBD: Description of the incoming method parameter
   * @return userFolder
   * @exception WcmException Exception raised in failure situation
   */

  /**
   * methods with old URI-interface ***********************************************************************************
   *
   * Gets URI for a folder specified by user context and folder name
   *
   * @param user TBD: Description of the incoming method parameter
   * @param folder TBD: Description of the incoming method parameter
   * @return the URI to the user/folder combination
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link #getUserFolderRID(IUser,
   *      String)}
   */
  public URI getUserFolder(IUser user, String folder)
    throws WcmException;

  /**
   * @param user
   * @return the URI to the user's home folder
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link #getUserHomeRID(IUser)}
   */
  public URI getUserHome(IUser user)
    throws WcmException;


  /**
   * @param folder
   * @return a complete URL to put in a mail or on a html-button that will open
   *      a webpage, showing the folder and its content, returns NULL, if
   *      resource is not a folder Please use createLinkToExplorerPage instead
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getCompleteRIDToShowFolder(ICollection)}
   */
  public URI getCompleteURIToShowFolder(ICollection folder)
    throws WcmException;

  /**
   * creates a complete URL to include in emails etc to start a service
   *
   * @param properties this includes parameters as defined in the
   *      IWcmConst.ServiceServletParameters You can also use "free" key-value
   *      pairs, but have to make sure, that receiving service will know about
   *      these keys creates a complete URL to include in emails etc to start a
   *      service
   * @return the Uri to call the servlet
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #createServletCallRID(Properties)}
   */
  public URI createServletCall(Properties properties)
    throws WcmException;

  /**
   * gets URI to show details page of resource
   *
   * @param resourceURI
   * @return the URI to navigate to this page
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #createLinkToDetailsPageRID(RID)}
   */
  public URI createLinkToDetailsPage(URI resourceURI);

  /**
   * gets URI to show explorer page of resource
   *
   * @param resourceURI
   * @return the URI to navigate to this page
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #createLinkToExplorerPageRID(RID)}
   */
  public URI createLinkToExplorerPage(URI resourceURI)
    throws WcmException;

  /**
   * Get a resource URI representing a user
   *
   * @param user
   * @return The URI to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getUserResourceRID(IUser)}
   */
  public URI getUserResource(IUser user)
    throws WcmException;

  /**
   * Get a resource RID representing a user or user information in a specific
   * format
   *
   * @param user the user in question
   * @param format the wanted format
   * @return The RID to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link #getUserResourceRID(IUser,
   *      String)}
   */
  public URI getUserResource(IUser user, String format)
    throws WcmException;

  /**
   * Get a resource URI representing a user
   *
   * @param uid
   * @return The URI to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getUserResourceRID(String)}
   */
  public URI getUserResource(String uid)
    throws WcmException;


  /**
   * Get a resource URI representing a user or user information in a specific
   * format
   *
   * @param uid the id of the user
   * @param format the format wanted
   * @return The URI to a corresponding user resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getUserResourceRID(String, String)}
   */
  public URI getUserResource(String uid, String format)
    throws WcmException;

  /**
   * Get a resource URI representing a group
   *
   * @param group
   * @return The URI to a corresponding group collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getGroupResourceRID(IGroup)}
   */
  public URI getGroupResource(IGroup group)
    throws WcmException;

  /**
   * Get a resource URI representing a group
   *
   * @param gid
   * @return The URI to a corresponding group collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getGroupResourceRID(String)}
   */
  public URI getGroupResource(String gid)
    throws WcmException;

  /**
   * Get a resource URI representing a role
   *
   * @param group
   * @return The URI to a corresponding role collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getRoleResourceRID(IRole)}
   */
  public URI getRoleResource(IRole group)
    throws WcmException;


  /**
   * Converts a given web URL in a RID of preset repository manager.
   *
   * @param webAddress a <code>URL</code> object to be converted
   * @return <code>RID</code> object with converted web url
   * @deprecated as of EP 5.0 SP5, replaced by {@link
   *      IURLGeneratorService#mapUri(IUri)}
   */
  public RID getRIDfromWebURL(URL webAddress);


  /**
   * Get a resource URI representing a role
   *
   * @param rid
   * @return The URI to a corresponding role collection resource
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getRoleResourceRID(String)}
   */
  public URI getRoleResource(String rid)
    throws WcmException;

  /**
   * Get a resource URI for bringing up a business card for a user
   *
   * @param user
   * @return The URI to a corresponding service
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getBusinessCardAppRID(IUser)}
   */
  public URI getBusinessCardApp(IUser user)
    throws WcmException;

  /**
   * Get a resource URI for bringing up a business card for a user
   *
   * @param uid
   * @return The URI to a corresponding service
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP2, replaced by {@link
   *      #getBusinessCardAppRID(String)}
   */
  public URI getBusinessCardApp(String uid)
    throws WcmException;

}// interface IURLGenerator
