/*
 * This class represents a geographical border in WGS84 format
 */
package com.sap.tc.webdynpro.clientserver.uielib.graphics.api;

/**
 * This class represents a geographical border 
 * i.e. a rectangle defined by its upper left and lower right corner in WGS84 format.
 * @author SAP
 */
public final class WDGeoBorder
{
	private double top;
	private double left;
	private double bottom;
	private double right;

	/**
	 * Creates a new border with the co-ordinates 0,0; 0,0
	 */
	public WDGeoBorder()
	{
		top = left = bottom = right = 0;
	}

	/**
	 * Creates a new border with the given co-ordinates
	 */
	public WDGeoBorder(double left, double top, double right, double bottom)
	{
		this.top = top;
		this.left = left;
		this.right = right;
		this.bottom = bottom;
	}

	/**
	 * Creates a new border with the given geo-positions
	 */
	public WDGeoBorder(WDGeoPosition topLeft, WDGeoPosition bottomRight)
	{
		top = topLeft.getLatitude();
		left = topLeft.getLongitude();
		bottom = bottomRight.getLatitude();
		right = bottomRight.getLongitude();
	}

	/**
	 * Creates a copy of a border
	 */
	public WDGeoBorder(WDGeoBorder srcBorder)
	{
		top = srcBorder.top;
		left = srcBorder.left;
		bottom = srcBorder.bottom;
		right = srcBorder.right;
	}

	/**
	 * Returns the top co-ordinate of the border
	 */
	public double getTop()
	{
		return top;
	}

	/**
	 * Returns the left co-ordinate of the border
	 */
	public double getLeft()
	{
		return left;
	}

	/**
	* Returns the bottom co-ordinate of the border
	*/
	public double getBottom()
	{
		return bottom;
	}

	/**
	* Returns the right co-ordinate of the border
	*/
	public double getRight()
	{
		return right;
	}

	/**
	 * Sets the co-ordinates of the border
	 */
	public void setBorder(double left, double top, double right, double bottom)
	{
		this.top = top;
		this.left = left;
		this.bottom = bottom;
		this.right = right;
	}

	/**
	 * Moves the border co-ordinates by x and y
	 */
	public void moveBorder(double x, double y)
	{
		this.left += x;
		this.right += x;
		this.top += y;
		this.bottom += y;
	}

	/**
	 * Sets the co-ordinates to new values
	 */
	public void setBorder(WDGeoPosition topLeft, WDGeoPosition bottomRight)
	{
		top = topLeft.getLatitude();
		left = topLeft.getLongitude();
		bottom = bottomRight.getLatitude();
		right = bottomRight.getLongitude();
	}

	/**
	 * Resets the border co-ordinates to zero
	 */
	public void setBorderEmpty()
	{
		top = left = bottom = right = 0;
	}

	/**
	 * Returns the difference between right and left co-ordinate.
	 */
	public double width()
	{
		return right - left;
	}

	/**
	 * Returns the difference between top and bottom co-ordinate
	 */
	public double height()
	{
		return top - bottom;
	}

	/**
	 * Returns the top-left position.
	 */
	public WDGeoPosition topLeft()
	{
		return new WDGeoPosition(left, top);
	}

	/**
	 * Returns the bottom-right position.
	 */
	public WDGeoPosition bottomRight()
	{
		return new WDGeoPosition(right, bottom);
	}

	/**
	 * Ensures that the left co-ordinate is smaller or equal to the top,
	 * and ateh bottom smaller or equal to the top
	 */
	public void normalizeBorder()
	{
		if (right < left)
		{
			double swap = right;
			right = left;
			left = swap;
		}
		if (bottom > top)
		{
			double swap = bottom;
			bottom = top;
			top = swap;
		}
	}

	/**
	 * Returns true if all co-ordinates are zero
	 */
	public boolean isBorderEmpty()
	{
		return (left == 0 && top == 0 && right == 0 && bottom == 0);
	}

	/**
	 * Returns true if the current border has the same co-ordinates as the second one.
	 */
	public boolean equals(WDGeoBorder border)
	{
		return (left == border.left && top == border.top && right == border.right && bottom == border.bottom);
	}

	/**
	 * Tests whether a position is inside the border.
	 */
	public boolean positionInBorder(WDGeoPosition pos)
	{
		return (
			(pos.getLatitude() <= top && pos.getLatitude() >= bottom)
				&& (pos.getLongitude() >= left && pos.getLongitude() <= right));
	}

	/**
	 * Merges the current border with a second one.
	 */
	public void unionBorder(WDGeoBorder border)
	{
		if (border.top > top)
			top = border.top;
		if (border.left < left)
			left = border.left;
		if (border.bottom < bottom)
			bottom = border.bottom;
		if (border.right > right)
			right = border.right;
	}

	/**
	 * Intersects the current border with a second one.
	 */
	public void intersectBorder(WDGeoBorder border)
	{
		if (border.top < top)
			top = border.top;
		if (border.left > left)
			left = border.left;
		if (border.bottom > bottom)
			bottom = border.bottom;
		if (border.right < right)
			right = border.right;
	}

	/**
	 * Merges the first border with the second one and saves the result in the current border.
	 * @return false if the resulting border is empty
	 */
	public boolean intersectBorder(WDGeoBorder border1, WDGeoBorder border2)
	{
		if (((border1.left <= border2.right && border1.left >= border2.left)
			|| (border1.right <= border2.right && border1.right >= border2.left))
			|| ((border1.left >= border2.left && border1.right <= border2.right)
				|| (border1.left <= border2.left && border1.right >= border2.right)))
		{
			if (((border1.top >= border2.bottom && border1.top <= border2.top)
				|| (border1.bottom >= border2.bottom && border1.bottom <= border2.top))
				|| ((border1.top <= border2.top && border1.bottom >= border2.bottom)
					|| (border1.top >= border2.top && border1.bottom <= border2.bottom)))
			{
				left = border1.left >= border2.left ? border1.left : border2.left;
				top = border1.top <= border2.top ? border1.top : border2.top;
				right = border1.right <= border2.right ? border1.right : border2.right;
				bottom = border1.bottom >= border2.bottom ? border1.bottom : border2.bottom;

				return true;
			}
		}
		left = top = right = bottom = 0;

		return false;
	}

	/**
	 * Returns the center position of the border.
	 */
	public WDGeoPosition centerPosition()
	{
		return new WDGeoPosition((left + right) / 2, (bottom + top) / 2);
	}

	public String toString()
	{
		return "top:" + top + " left:" + left + " bottom:" + bottom + " right:" + right;
	}
}
